///
//  Little cms - profiler construction set
//  Copyright (C) 1998-2001 Marti Maria
//
// THIS SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
// EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
// WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
// IN NO EVENT SHALL MARTI MARIA BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
// INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
// OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
// WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF
// LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
// OF THIS SOFTWARE.
//
// This file is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, if you
// distribute this file as part of a program that contains a
// configuration script generated by Autoconf, you may include it under
// the same distribution terms that you use for the rest of that program.
//
// Version 1.08a



#include "qtlcmswidgets.h"
#include "qtmeasurementtooldialog.h"


#ifdef USE_KDE
#include <kfiledialog.h>
#endif

// GridLayout, represents one patch

int GridLayout::Scale(int n, int amax)   
{ 
    return (int) (((double) (n*amax)/LAYOUT_MAX)+.5); 
}

void GridLayout::ScaleTo(int w, int h, QRect& Scaled)
{
        Scaled.setRect(Scale(rect.x(), w),
                       Scale(rect.y(), h),
                       Scale(rect.width(), w),
                       Scale(rect.height(), h));
}


// A ICC profile transform on whole image


void ImageItem::TransformImage(const QString OutputProfile, 
                               const QString InputProfile, 
                               QImage& p)
{

    cmsHTRANSFORM xform;
    cmsHPROFILE hIn, hOut;

    hIn  = cmsOpenProfileFromFile(InputProfile, "r");
    hOut = cmsOpenProfileFromFile(OutputProfile, "r");

    xform = cmsCreateTransform(hIn, TYPE_RGBA_8, hOut, TYPE_RGBA_8, 
                        INTENT_PERCEPTUAL, cmsFLAGS_WHITEBLACKCOMPENSATION);

    for (int i=0; i < p.height(); i++) {

            LPBYTE ptr = p.scanLine(i);
            cmsDoTransform(xform, ptr, ptr, p.width());
    }

    cmsDeleteTransform(xform);
    cmsCloseProfile(hIn);
    cmsCloseProfile(hOut);
}





ImageItem::ImageItem(QString ImageFile, const QString Monitor, const QString Scanner, QCanvas *canvas )
    : QCanvasRectangle( canvas ), image()
{


    lProof = FALSE;
	Reload(ImageFile, Monitor, Scanner, canvas);
}



void ImageItem::Reload(QString ImageFile, const QString Monitor, const QString Scanner, QCanvas *canvas )
{
	if (ImageFile == QString::null) return;

	if (!image.load(ImageFile)) {
    
    QMessageBox mb( "Load image failure",
                    "This image format is unsupported by Qt\n"
                    "in your system. Sorry. Supported ones \n"
                    "are TIFF, PNG, BMP, XBM, XPM and PNM.",
                    QMessageBox::Critical,
                    QMessageBox::Ok | QMessageBox::Default,
                    QMessageBox::NoButton,  QMessageBox::NoButton);

    mb.exec();  
    return;
    }

    // try again


    QImage scaled = image.smoothScale(canvas->width(), canvas->height());

    setSize( scaled.width(), scaled.height() );
    
    sx = (double) image.width() / scaled.width();
    sy = (double) image.height() / scaled.height();

    pixmap.convertFromImage(scaled); 

    if (lProof) {

    if (Monitor != QString::null &&
        Scanner != QString::null)  
            TransformImage(Monitor, Scanner, scaled);
    }
	
    proof.convertFromImage(scaled);

    move(0, 0);
    setZ(5);
    show();
}


void ImageItem::drawShape( QPainter &p )
{
    
    p.drawPixmap( int(x()), int(y()), lProof? proof : pixmap );
}



// ---------------------------------------------------------------- The grid item

GridItem::GridItem( QCanvas *canvas, QString Template, int SafeFrame )
    : QCanvasRectangle( 10, 10, canvas->width() - 10, canvas->height() - 10, canvas)
{

    UnShearFactor = 0.0;
    SafeFrameFactor = SafeFrame / 100.;
    setBrush( NoBrush) ; 

    Layout.setAutoDelete(TRUE);
    LCMSHANDLE hIT8 = cmsxIT8LoadFromFile((const char *) Template);

    if (hIT8) {

        int nRects = (int) cmsxIT8GetPropertyDbl(hIT8, "NUMBER_OF_SETS");
        char PatchName[256];

        for (int i=0; i < nRects; i++) {

            if (cmsxIT8GetDataSetByPos(hIT8, 0, i, PatchName, 256)) {

                GridLayout* Patch = new GridLayout;

                double x, y, xSize, ySize;
                
                cmsxIT8GetDataSetDbl(hIT8, PatchName, "POS_X", &x);
                cmsxIT8GetDataSetDbl(hIT8, PatchName, "POS_Y", &y);
                cmsxIT8GetDataSetDbl(hIT8, PatchName, "SIZE_X", &xSize);
                cmsxIT8GetDataSetDbl(hIT8, PatchName, "SIZE_Y", &ySize);

                strcpy(Patch -> p.Name, (const char *) PatchName);
                Patch -> rect.setRect(int(x), int(y), int(xSize), int (ySize));

                Layout.append(Patch);

            }
        }

        cmsxIT8Free(hIT8);
    }

    setZ(10);
    show();
}



static
void DoDeflateRect(QRect& rect, int sx, int sy)
{

    rect.setRect(rect.x() + sx, rect.y() + sy, 
                  rect.width()-sx*2, rect.height()-sx*2);
}


void GridItem::ComputePatchRects(QRect& OurViewport, GridLayout* Patch, 
                                 QRect& Frame, QRect& HotZone)
{

        QRect Scaled;

        Patch ->ScaleTo(OurViewport.width(), OurViewport.height(), Scaled);

        int translateX = OurViewport.x();
        int translateY = OurViewport.y();

        int UnShearX = (int) ((double) Scaled.x() * UnShearFactor / 16.);
        int UnShearY = (int) ((double) Scaled.y() * UnShearFactor / 16.);

        translateX += UnShearY;
        translateY += UnShearX;
        
        Scaled.moveBy(translateX, translateY);

        Frame = Scaled;
        
        int SafeFrame = (int) ((double) Scaled.width() * SafeFrameFactor / 2.);
                
        DoDeflateRect(Scaled, SafeFrame, SafeFrame);

        HotZone = Scaled;
        
}


// Compute our viewport extension

void GridItem::ComputeViewport(QRect& OurViewport, double sx, double sy)
{

    double UnShearMax = UnShearFactor * height() * sy / 16.;
    
    OurViewport.setRect(rect().x() * sx, rect().y()*sy,
                      rect().width() * sx, rect().height() * sy);

    OurViewport.setRight(OurViewport.right() + UnShearMax);
    OurViewport.setLeft(OurViewport.left() - UnShearMax);

}


// Uh-oh.... viewPort() was buggy and slow when I wrote this,
// so, I do by hand... 
void GridItem::drawShape( QPainter &p )
{   
    GridLayout* Patch;
    QRect OurViewport;

    
    ComputeViewport(OurViewport, 1.0, 1.0);
    
    for (Patch = Layout.first(); 
            Patch;
            Patch = Layout.next()) {
                
                QRect Frame, HotZone;

                // Scale to our viewport
                ComputePatchRects(OurViewport, Patch, Frame, HotZone);
            
                // Draw
                p.setPen( Qt::white );    
                p.drawRect(Frame);

                p.setPen( Qt::green );    
                p.drawRect(HotZone);
                
    }
    
    
    if (UnShearFactor == 0.0) {

        
    p.setPen( Qt::yellow );    
    p.drawRect(rect()); 
    }

}


void GridItem::Pick(QString InputProfileName, QImage& Image, double sx, double sy)
{   
    GridLayout* Patch;
    QRect OurViewport;
    cmsHTRANSFORM hInputXFORM;
    cmsHPROFILE hLab = NULL, hInputProfile = NULL;
    const char* cInputProfile = (const char*) InputProfileName;
    
    if (cInputProfile) {

        hLab          = cmsCreateLabProfile(NULL);
        hInputProfile = cmsOpenProfileFromFile(cInputProfile, "r");
        hInputXFORM   = cmsCreateTransform(hInputProfile, TYPE_RGB_16, 
                                        hLab, TYPE_Lab_16, INTENT_ABSOLUTE_COLORIMETRIC, 
                                        cmsFLAGS_NOTPRECALC);
    }
    else hInputXFORM = NULL;


    ComputeViewport(OurViewport, sx, sy);
    
    for (Patch = Layout.first(); 
            Patch;
            Patch = Layout.next()) {
                
                // Scale to our viewport

                QRect Frame, HotZone;
                
                ComputePatchRects(OurViewport, Patch, Frame, HotZone);

                // Ok, take

                Statistic r, g, b;
                
                int xstart = HotZone.x();
                int ystart = HotZone.y();
                int xsize  = HotZone.width();
                int ysize  = HotZone.height();

                Patch ->p.Colorant.RGB[0] = 0;
                Patch ->p.Colorant.RGB[1] = 0;
                Patch ->p.Colorant.RGB[2] = 0;
                Patch ->p.dwFlags         = 0;
                

                if ((xstart < 0) || (ystart < 0) || 
                    ((xstart + xsize) > Image.width()) ||
                    ((ystart + ysize > Image.height()))) 
                                continue;

                for (int i=0; i < ysize; i++)
                    for (int j=0; j < xsize; j++) {

                        QRgb rgb = Image.pixel(xstart+j, ystart+i);

                        int rv =  qRed(rgb);
                        int gv =  qGreen(rgb);
                        int bv =  qBlue(rgb);

                        r.Add(rv); g.Add(gv); b.Add(bv);
                        
                    }

                Patch ->p.dwFlags = PATCH_HAS_RGB|PATCH_HAS_CHISQ;

                Patch ->p.Colorant.RGB[0] = r.Mean();
                Patch ->p.Colorant.RGB[1] = g.Mean();
                Patch ->p.Colorant.RGB[2] = b.Mean();

                
                Patch->p.ChiSq = (r.Std() + g.Std() + b.Std()) / 3.0;


                if (hInputXFORM) {

                    WORD RGB[3];
                    WORD LabEncoded[3];
                
                    RGB[0] =(WORD) floor(Patch -> p.Colorant.RGB[0] * 257. + .5);
                    RGB[1] =(WORD) floor(Patch -> p.Colorant.RGB[1] * 257. + .5);
                    RGB[2] =(WORD) floor(Patch -> p.Colorant.RGB[2] * 257. + .5);

                    cmsDoTransform(hInputXFORM, RGB, LabEncoded, 1);
                    cmsLabEncoded2Float(&Patch ->p.Lab, LabEncoded);
                    cmsLab2XYZ(NULL, &Patch ->p.XYZ, &Patch ->p.Lab);
                }
    }
        
    if (hInputXFORM) {

        cmsDeleteTransform(hInputXFORM);
        cmsCloseProfile(hInputProfile);
        cmsCloseProfile(hLab);
    }

}


// ------------------------------------------------------------ The figure editor

void FigureEditor::resizeEvent( QResizeEvent* e )
{
  QCanvasView::resizeEvent( e );
  canvas()->resize(width()-4,height()-4);
  MeasurementDialog.Reload();
}

void FigureEditor::contentsMousePressEvent(QMouseEvent* e)
{
   
    moving   = NULL;
    sizing   = NULL;
    unshearing = NULL;

    QCanvasItemList l=canvas()->collisions(e->pos());

    for (QCanvasItemList::Iterator it=l.begin(); it!=l.end(); ++it) {

    if ( (*it)->rtti() == imageRTTI )   
         continue;
    
    
    if (e -> state() & Qt::ControlButton) {

            unshearing = (GridItem*) *it;
            moving_start = e->pos();
            int Start = (int) ((double)  moving_start.x() - unshearing -> UnShearFactor * 128.);
            moving_start.setX(Start);
            contentsMouseMoveEvent(e);

    }
    else
#if 0
    if (e -> button() & Qt::RightButton) {
#else
    if (e -> state() & Qt::ShiftButton) {
#endif
            sizing = (GridItem*) *it;                   
            contentsMouseMoveEvent(e);
    }
    else 
    if (e -> button() & Qt::LeftButton) {
            moving = (GridItem*) *it;
            moving_start = e->pos();
    }

        return;
    }

    
}


void FigureEditor::contentsMouseMoveEvent(QMouseEvent* e)
{

    if ( moving ) {

        moving->moveBy(e->pos().x() - moving_start.x(),
                       e->pos().y() - moving_start.y());

        moving_start = e->pos();
        canvas()->update();
    }
    else
    if ( sizing) {
            
            int p1x = (int) sizing -> x();
            int p1y = (int) sizing -> y();
            int p2x = e -> pos().x();
            int p2y = e -> pos().y();
            int sx = QABS(p2x - p1x);
            int sy = QABS(p2y - p1y);
            
            sizing -> setSize(sx, sy);
            canvas()->update();
        }
    else {

        if (unshearing) {

            double  shear = e -> pos().x() - moving_start.x();
            // if (shear > 128.) shear = 128;
            // if (shear < 0.) shear = 0.;          

            unshearing -> UnShearFactor = shear / 128.;         
            canvas()-> setAllChanged();
            canvas()->update();         
        }
    }
}


void FigureEditor::clear()
{
    QCanvasItemList list = canvas()->allItems();
    QCanvasItemList::Iterator it = list.begin();
    for (; it != list.end(); ++it) {
    if ( *it )
        delete *it;
    }

    moving = NULL;
    sizing = NULL;
    unshearing = NULL;
}


// ------------------------------------------------------------ The main dialog

qtMeasurementToolDialog::qtMeasurementToolDialog( QWidget* parent, const char* name, bool modal, WFlags f )
  : qtMeasurementToolDialogBase( parent, name, modal, f )  
{
    
	fn = QString::null;
    ConfigFile = "lcmsmt.cfg";
    WorkTempFileName = "$$q60$$.$$$";
    
    TheImage        = NULL;
    TheGrid         = NULL;
    CurrentTemplate = "*";
    CurrentSafeFrame = -1;
    

    Status->setText("Hello!");

    int w = CanvasFrame -> width();
    int h = CanvasFrame -> height();

    CanvasFrameLayout =new QVBoxLayout( CanvasFrame ); 
    CanvasFrameLayout->setSpacing( 6 );
    CanvasFrameLayout->setMargin( 11 );

    Canvas = new QCanvas(w-4, h-4); 
    Canvas -> setAdvancePeriod(30);   
    FigEditor = new FigureEditor(*this, *Canvas, CanvasFrame);
    FigEditor -> clear();
    //  FigEditor-> resize(w, h);
    CanvasFrameLayout->addWidget(FigEditor);

    LoadConfig();

    QDate Today = QDate::currentDate();
    EditCreated->setText(Today.toString());

    
}


qtMeasurementToolDialog::~qtMeasurementToolDialog()
{
    SaveConfig();
}

void qtMeasurementToolDialog::AddTemplate()
{
    
    QString Template     = SelectTemplate -> getFilename();
    double  SafeFrameVal = SafeFrame->value();

    // Same values?
    
    if ((CurrentTemplate == Template) && 
        (CurrentSafeFrame == SafeFrameVal)) return;
    

    // Values has changed
    if ((TheImage != NULL) && (Template != QString::null)) {
        
        if (TheGrid != NULL)
            delete TheGrid;

        TheGrid = new GridItem(Canvas, Template, SafeFrameVal);
        CurrentTemplate  = Template;
        CurrentSafeFrame = SafeFrameVal;
    }
}                   



void qtMeasurementToolDialog::LoadImage()
{
#ifdef USE_KDE
  QString const caption("Select scanout image");
  QString const filter("*.gif *.png *.jpg *.bmp *.pnm|Image files\n*.*|All files");
  QString fn=KFileDialog::getOpenFileName( QString::null, 
                       filter,
                       this,
                       caption);
  
#else
    fn = QFileDialog::getOpenFileName( "", 
                                                "Image files (*.*)\nAll files (*.*)",
                                                this,
                                                QString::null,
                                                "Select scanout image");
#endif
    if (fn == QString::null) return;

    FigEditor->clear();
    TheGrid = NULL;
    CurrentTemplate = "";

	if (ProofToggle -> isOn())
		ProofToggle -> toggle();

    TheImage = new ImageItem(fn, MonitorProfileSelector ->getFilename(),
                                 InputProfileSelector->getFilename(),  
                                 Canvas);

    if (EditMaterial->text() == "")
            EditMaterial->setText("Image '"+fn+"'");

    AddTemplate();
    slotTabChanged();

}


void qtMeasurementToolDialog::LoadIT8( const char *fileName )
{
    QFile f( fileName );

    if ( !f.open( IO_ReadOnly ) )
        return;

    IT8Edit->setAutoUpdate( FALSE );
    IT8Edit->clear();

    QTextStream t(&f);
    while ( !t.eof() ) {
    QString s = t.readLine();
    IT8Edit->append( s );
    }
    f.close();

    IT8Edit->setAutoUpdate( TRUE );
    IT8Edit->repaint();
    IT8Edit->setEdited( FALSE );

    // Try to parse

    LCMSHANDLE hIT8 = cmsxIT8LoadFromFile(fileName);

    EditOriginator   -> setText(cmsxIT8GetProperty(hIT8, "ORIGINATOR"));
    EditDescriptor   -> setText(cmsxIT8GetProperty(hIT8, "DESCRIPTOR"));
    EditCreated      -> setText(cmsxIT8GetProperty(hIT8, "CREATED"));
    EditSerial       -> setText(cmsxIT8GetProperty(hIT8, "SERIAL"));
    EditMaterial     -> setText(cmsxIT8GetProperty(hIT8, "MATERIAL"));
    EditManufacturer -> setText(cmsxIT8GetProperty(hIT8, "MANUFACTURER"));

    cmsxIT8Free(hIT8);
}


void qtMeasurementToolDialog::SaveIT8(const char *filename)
{
    
    QString text = IT8Edit->text();
    QFile f( filename );
    if ( !f.open( IO_WriteOnly ) ) {
    return;
    }

    QTextStream t( &f );
    t << text;
    f.close();

    IT8Edit->setEdited( FALSE );    
}




void qtMeasurementToolDialog::DumpResultsToIT8(QList<GridLayout>& Layout)
{
        
    LCMSHANDLE it8 = cmsxIT8Alloc();
    
    cmsxIT8SetSheetType(it8, "LCMS/MEASUREMENT");
    cmsxIT8SetProperty(it8, "ORIGINATOR",   (const char *) EditOriginator -> text());
    cmsxIT8SetProperty(it8, "DESCRIPTOR",   (const char *) EditDescriptor -> text());
    cmsxIT8SetProperty(it8, "MANUFACTURER", (const char *) EditManufacturer -> text());
    cmsxIT8SetProperty(it8, "CREATED",      (const char *) EditCreated -> text());
    cmsxIT8SetProperty(it8, "SERIAL",       (const char *) EditSerial -> text());
    cmsxIT8SetProperty(it8, "MATERIAL",     (const char *) EditMaterial -> text());

    cmsxIT8SetProperty(it8, "CHISQ_DOF",    "3");

    GridLayout* Patch;
    int nSets = 0;
    int nFields = 5;

    for (Patch = Layout.first(); 
            Patch;
            Patch = Layout.next()) {
                    if (Patch -> p.dwFlags != 0) 
                        nSets++;
            }



    cmsxIT8SetPropertyDbl(it8, "NUMBER_OF_SETS", nSets);

    if (RadioButtonLabXYZ -> isChecked()) nFields += 3;

    cmsxIT8SetPropertyDbl(it8, "NUMBER_OF_FIELDS", nFields);

    nFields = 0;
    cmsxIT8SetDataFormat(it8, nFields++, "SAMPLE_ID");
    
    if (RadioButtonRGB -> isChecked()) {

    cmsxIT8SetDataFormat(it8, nFields++, "RGB_R");
    cmsxIT8SetDataFormat(it8, nFields++, "RGB_G");
    cmsxIT8SetDataFormat(it8, nFields++, "RGB_B");
    }

    if (RadioButtonLab -> isChecked() ||
        RadioButtonLabXYZ -> isChecked()) {

    cmsxIT8SetDataFormat(it8, nFields++, "LAB_L");
    cmsxIT8SetDataFormat(it8, nFields++, "LAB_A");
    cmsxIT8SetDataFormat(it8, nFields++, "LAB_B");
    }

    if (RadioButtonXYZ -> isChecked() ||
        RadioButtonLabXYZ -> isChecked()) {

    cmsxIT8SetDataFormat(it8, nFields++, "XYZ_X");
    cmsxIT8SetDataFormat(it8, nFields++, "XYZ_Y");
    cmsxIT8SetDataFormat(it8, nFields++, "XYZ_Z");
    }


    cmsxIT8SetDataFormat(it8, nFields++, "CHI_SQD_PAR");
    
    

    for (Patch = Layout.first(); 
            Patch;
            Patch = Layout.next()) {

                if (Patch -> p.dwFlags != 0) {

                cmsxIT8SetDataSet(it8, Patch->p.Name, "SAMPLE_ID", Patch->p.Name);

                if (RadioButtonRGB -> isChecked()) {

                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "RGB_R", Patch ->p.Colorant.RGB[0]);
                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "RGB_G", Patch ->p.Colorant.RGB[1]);
                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "RGB_B", Patch ->p.Colorant.RGB[2]);
                }

                if (RadioButtonLab -> isChecked() || RadioButtonLabXYZ -> isChecked()) {

                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "LAB_L", Patch ->p.Lab.L);
                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "LAB_A", Patch ->p.Lab.a);
                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "LAB_B", Patch ->p.Lab.b);
                }

                if (RadioButtonXYZ -> isChecked() || RadioButtonLabXYZ -> isChecked()) {

                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "XYZ_X", Patch ->p.XYZ.X * 100.);
                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "XYZ_Y", Patch ->p.XYZ.Y * 100.);
                    cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "XYZ_Z", Patch ->p.XYZ.Z * 100.);
                }


                cmsxIT8SetDataSetDbl(it8, Patch->p.Name, "CHI_SQD_PAR", Patch ->p.ChiSq);

                }
            }

    cmsxIT8SaveToFile(it8, WorkTempFileName);
    cmsxIT8Free(it8);
    LoadIT8(WorkTempFileName);
        
}


BOOL qtMeasurementToolDialog::LoadConfig()
{
    
    QFile Check(ConfigFile);

    if (Check.exists()) {

        LCMSHANDLE hCfg     = cmsxIT8LoadFromFile(ConfigFile);
        if (hCfg == (LCMSHANDLE) NULL) return FALSE;

        SelectTemplate->Dir = (cmsxIT8GetProperty(hCfg, "TEMPLATE_DIR"));   
        SelectTemplate->FillValues();
        SelectTemplate->setCurrItem(cmsxIT8GetPropertyDbl(hCfg, "TEMPLATE_NUM"));

        MonitorProfileSelector ->Dir = (cmsxIT8GetProperty(hCfg,"MONITOR_PROFILES_DIR"));
        MonitorProfileSelector ->RestrictColorspace = icSigRgbData;
        MonitorProfileSelector ->RestrictClass = icSigDisplayClass;
        MonitorProfileSelector -> FillValues();
        MonitorProfileSelector ->setFilename(cmsxIT8GetProperty(hCfg,   "MONITOR_PROFILE"));

        InputProfileSelector->Dir = (cmsxIT8GetProperty(hCfg,   "INPUT_PROFILES_DIR"));
        InputProfileSelector-> RestrictColorspace = icSigRgbData;
        InputProfileSelector->FillValues();
        InputProfileSelector->setFilename(cmsxIT8GetProperty(hCfg,  "INPUT_PROFILE"));

        SafeFrame -> setValue(cmsxIT8GetPropertyDbl(hCfg, "SAFE_FRAME"));
        
        cmsxIT8Free(hCfg);
        return TRUE;

    }


    SafeFrame -> setValue(60);
    SelectTemplate -> Dir = "template";
	SelectTemplate->FillValues();
	SelectTemplate->setCurrItem(0);
    MonitorProfileSelector ->Dir = "profiles";
	MonitorProfileSelector ->FillValues();
    InputProfileSelector ->Dir = "profiles";
	InputProfileSelector->FillValues();

    return FALSE;
    
}
    
void qtMeasurementToolDialog::SaveConfig()
{

    LCMSHANDLE hCfg = cmsxIT8Alloc();

    cmsxIT8SetProperty(hCfg,    "TEMPLATE_DIR", SelectTemplate->Dir);
    cmsxIT8SetPropertyDbl(hCfg, "TEMPLATE_NUM", SelectTemplate->getCurrItem());
    cmsxIT8SetProperty(hCfg,    "MONITOR_PROFILES_DIR", MonitorProfileSelector->Dir);
    cmsxIT8SetProperty(hCfg,    "MONITOR_PROFILE", MonitorProfileSelector->getFilename());

    cmsxIT8SetProperty(hCfg,    "INPUT_PROFILES_DIR", InputProfileSelector->Dir);
    cmsxIT8SetProperty(hCfg,    "INPUT_PROFILE", InputProfileSelector->getFilename());
    cmsxIT8SetPropertyDbl(hCfg, "SAFE_FRAME", SafeFrame->value());
    cmsxIT8SaveToFile(hCfg,     ConfigFile);
    cmsxIT8Free(hCfg);

}

void qtMeasurementToolDialog::Reload()
{
    TheImage -> Reload(fn, MonitorProfileSelector ->getFilename(),
                                 InputProfileSelector->getFilename(),  
                                 Canvas);
}



// ------------------------------------------------------------------------------ Slots

void qtMeasurementToolDialog::slotSelSheet()
{
#ifdef USE_KDE
  QString const caption("Select destination sheet filename open");
  QString const filter("*.it *.cgt|IT8-CGATS.5 Files \n*.*|All files");
  QString fn=KFileDialog::getOpenFileName( QString::null, 
                       filter,
                       this,
                       caption);
#else

    QString fn = QFileDialog::getOpenFileName( "", 
                                                "IT8/CGATS.5 Files (*.it? *.cgt)\nAll files (*.*)",
                                                this,
                                                QString::null,
                                                "Select destination sheet filename");
#endif
    if (fn != QString::null) {
            
            QFile f(fn);
            if (f.exists()) {

                LoadIT8(fn);
            }
    }

}

void qtMeasurementToolDialog::slotPick()
{
    if (TheImage && TheGrid) {

                TheGrid -> Pick(InputProfileSelector->getFilename(), TheImage -> image, TheImage -> sx, TheImage -> sy);
                DumpResultsToIT8(TheGrid -> Layout);
                Tabs -> setCurrentPage(1);
    }
}

void qtMeasurementToolDialog::slotLoad()
{
    switch (Tabs -> currentPageIndex()) {

    case 0: LoadImage(); break;
    case 1: slotSelSheet(); break;

    default:;
    }

}

void qtMeasurementToolDialog::slotTabChanged()
{
    

    switch (Tabs -> currentPageIndex()) {

    case 0: 
            LoadBtn->show();
            LoadBtn->setTextLabel("Load image");

            if (TheImage == NULL) {

                Status->setText("Press LOAD IMAGE to start");
                PickBtn->hide();                
                ProofToggle->hide();

            } else
            {
                Status->setText("Click & drag mouse to place the grid: Left - Move, Shift - Size, Ctrl - UnShear. PICK when finished.");
                PickBtn->show();                                        
                ProofToggle -> show();
            }

            // Check whatsever template has changed
            AddTemplate();
            break;
            
    case 1: Status->setText("This is the raw ASCII IT8 sheet. SAVE to store on disk.");
            PickBtn->hide();
            LoadBtn->show();
            LoadBtn->setTextLabel("Load IT8 sheet");            
            ProofToggle -> hide();
            break;

    case 2: Status->setText("Current sheet properties.");
            PickBtn->hide();
            LoadBtn->hide();
            SaveBtn->hide();
            ProofToggle -> hide();
            return;
    
    default:
            Status->setText("Settings to apply on loading new images.");
            PickBtn->hide();
            LoadBtn->hide();
            SaveBtn->hide();
            ProofToggle -> hide();
            return;

    }

    if (IT8Edit->text().length() <= 0)
        SaveBtn->hide();    
    else
        SaveBtn->show();


}



void qtMeasurementToolDialog::slotSave()
{
#ifdef USE_KDE
  QString const caption("Select destination sheet filename");
  QString const filter("*.it? *.cgt|IT8-CGATS.5 Files\n*.*|All files");
  QString fn=KFileDialog::getSaveFileName( QString::null, 
                       filter,
                       this,
                       caption);
#else

        QString fn = QFileDialog::getSaveFileName( "", 
                                                "IT8/CGATS.5 Files (*.it? *.cgt)\nAll files (*.*)",
                                                this,
                                                "",
                                                "Select destination sheet filename");
#endif
    if (fn != QString::null) {
            
            SaveIT8(fn);
    }
}
    
    
void qtMeasurementToolDialog::slotProof()
{
    if (TheImage) {
            TheImage -> lProof = ProofToggle -> isOn();

			TheImage -> Reload(fn, MonitorProfileSelector ->getFilename(),
                                 InputProfileSelector->getFilename(),  
                                 Canvas);

            Canvas->setChanged(rect());
            Canvas->update();
    }

}


