'''
Defines default L{AccAdapt.Adapter}s for the L{LSRInterfaces.IEventHandler} 
interface on L{pyLinAcc.Accessible} objects.

@var event_to_method_map: Maps event names to method names to be called to 
  handle them
@type event_to_method_map: dictionary

@author: Pete Brunet
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
from POR import POR
from AEEvent import *
from AccAdapt import Adapter
from LSRInterfaces import *
from pyLinAcc import Interfaces, Constants
import pyLinAcc

event_to_method_map = {
  'window:activate': '_handleViewChange',
  'focus' : '_handleFocusEvent',
  'object:text-changed:insert' : '_handleTextEvent',
  'object:text-changed:delete' : '_handleTextEvent',
  'object:text-caret-moved' : '_handleCaretEvent',
  'object:text-selection-changed' : '_handleTextSelectionEvent',
  'object:selection-changed' : '_handleSelectionChangedEvent',
  'object:active-descendant-changed': '_handleDescendantEvent',
  'object:property-change:accessible-name': '_handlePropertyEvent',
  'object:property-change:accessible-role': '_handlePropertyEvent',
  'object:property-change:accessible-description': '_handlePropertyEvent',
  'object:property-change:accessible-value': '_handlePropertyEvent',
  'object:property-change:accessible-table-caption': '_handlePropertyEvent',
  'object:property-change:accessible-table-summary' : '_handlePropertyEvent',
  'object:property-change:accessible-table-column-description' : 
  '_handlePropertyEvent',
  'object:property-change:accessible-table-row-description' : 
  '_handlePropertyEvent',
  'object:children-changed:add' : '_handleChildrenEvent',
  'object:children-changed:remove' : '_handleChildrenEvent',
  'object:row-inserted' : '_handleTableEvent',
  'object:row-deleted' : '_handleTableEvent',
  'object:row-reordered' : '_handleTableEvent',
  'object:column-inserted' : '_handleTableEvent',
  'object:column-deleted' : '_handleTableEvent',
  'object:column-reordered' : '_handleTableEvent',
  'object:visible-data-changed' : '_handleScreenEvent',
  'object:text-bounds-changed' : '_handleScreenEvent',
  'object:object-bounds-changed' : '_handleScreenEvent'
  # PP: not handling accessible parent as it seems to cause problems
  #'object:property-change:accessible-parent' : '_handleHierarchyEvent',
}

class DefaultEventHandlerAdapter(Adapter):
  '''
  Adapts all events from AT-SPI accessibles to the interfaces defined in
  L{provides}. No condition for adaption is given implying that this adapter is
  used as a default by L{AccAdapt} when no better adapter is available.
  
  This class is meant to be subclassed by more specific event handlers. Only the
  protected handler methods (those starting with _handle) need to be overridden 
  as the public method L{getAEEvents} will call the most child implementation of 
  the appropriate event handling method.
  '''
  provides = [IEventHandler]
  # event are handled serially, no need to make lots of adapters
  #singleton = True
  
  @pyLinAcc.errorToLookupError
  def getAEEvents(self, event):
    '''
    Determines what L{AEEvent}s should be posted to L{EventManager} for later
    execution by a L{Tier} based on the provided raw L{pyLinAcc.Event.Event}.
    Makes an initial decision about whether the event occurred in a focused
    control or an unfocused control.
    
    @param event: Raw event
    @type event: L{pyLinAcc.Event.Event}
    @return: Events to be posted to the active L{Tier} or None if there is no
      handler
    @rtype: tuple of L{AEEvent}
    '''
    # abort immediately if the accessible has state defunct or just went defunct
    # and is now a dummy CORBA object
    try:
      ss = self.subject.getState()
      if ss.contains(Constants.STATE_DEFUNCT):
        return None
    except AttributeError:
      return None
    
    try:
      # try to hash against an exact method
      name = event_to_method_map[event.type.name]
      method = getattr(self, name)
    except (KeyError, AttributeError):
      # if that fails, try to check the minor event type
      if event.type.major == 'state-changed':
        method = self._handleStateEvent
      else:
        return None
      
    # call the method that handles this event
    # keyword arguments will be passed as-is to AEEvent constructors
    return method(event, focused=ss.contains(Constants.STATE_FOCUSED))

  def _handleFocusEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.FocusChange} indicating that the accessible being
    adapted has gained the focus. Also creates a L{AEEvent.SelectorChange}.
    These two L{AEEvent}s will be posted by the caller.
    
    @param event: Raw focus change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.FocusChange} and L{AEEvent.SelectorChange}
    @rtype: tuple of L{AEEvent}
    '''
    por = POR(self.subject, None, 0)
    # adapt the accessible to an adapter which provides an IAccesssibleInfo
    # interface and get the accessible's item text
    item = IAccessibleInfo(por).getAccItemText()
    # focus events are always in the focus layer
    kwargs['focused'] = True
    return (FocusChange(por, True, **kwargs),  
            SelectorChange(por, item, **kwargs))
  
  def _handlePropertyEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.PropertyChange} indicating that some simple property of
    an accessible changed. These two L{AEEvent}s will be posted by the caller.
    
    The L{POR} for the event soruce returned by this method will be marked as 
    incomplete as the accessible may actually be an item. It will be resolved
    at a later time by the L{AEEvent} if the event will actually be processed.
    
    @param event: Raw property change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: Property change event
    @rtype: tuple of L{AEEvent}
    '''
    por = POR(self.subject, None, 0, incomplete=True)
    # strip off the initial word
    name = event.type.minor[len('accessible-'):]
    if name == 'value':
      # try to get the property value from the accessible
      iv = Interfaces.IValue(self.subject)
      value = iv.currentValue
    elif name == 'role':
      value = unicode(self.subject.getLocalizedRoleName(), 'utf-8')
    else:
      # otherwise use the string that ships with the event
      value = unicode(event.any_data, 'utf-8')
    return (PropertyChange(por, name, value, **kwargs),)
  
  def _handleStateEvent(self, event, focused, **kwargs):
    '''
    Creates an L{AEEvent.StateChange} indicating that some simple property of
    an accessible changed. These two L{AEEvent}s will be posted by the caller.
    
    The L{POR} for the event soruce returned by this method will be marked as 
    incomplete as the accessible may actually be an item. It will be resolved
    at a later time by the L{AEEvent} if the event will actually be processed.
    
    @param event: Raw state change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.StateChange} event
    @rtype: tuple of L{AEEvent}
    '''
    por = POR(self.subject, None, 0, incomplete=True)
    return (StateChange(por, event.type.minor, event.detail1, **kwargs),)
  
  def _handleChildrenEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.ChildrenChange} indicating that a child has been added/
    removed to an accessible. This L{AEEvent} will be posted by the caller.
    
    The L{POR} for the event soruce returned by this method will be marked as 
    incomplete as the accessible may actually be an item. It will be resolved
    at a later time by the L{AEEvent} if the event will actually be processed.
    
    @param event: Raw children change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.ChildrenChange} event
    @rtype: tuple of L{AEEvent}
    '''    
    por = POR(self.subject, None, 0)
    child_por = POR(event.any_data, None, 0, incomplete=True)
    return (ChildrenChange(por, event.type.minor == 'add', child_por, 
                           **kwargs),)
    
  def _handleTableEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.TableChange} indicating a insert/delete/reorder of a 
    row/column in a table-based accessible. This L{AEEvent} will be posted by 
    the caller.
    
    @param event: Raw table change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.TableChange} event
    @rtype: tuple of L{AEEvent}
    '''
    por = POR(self.subject, None, 0)
    # per AT-SPI 1.7 specification
    it = Interfaces.ITable(por.accessible) 
    # index of first child of ins/del/reorder
    index1 = it.getIndexAt(event.detail1, 0)
    # index of last child of ins/del/reorder
    index2 = it.getIndexAt(event.detail1 + event.detail2 - 1, 0)   
    first_child_por = POR(self.subject, index1, 0)
    last_child_por = POR(self.subject, index2, 0)
    name = event.type.major
    # determine whether a row or column event?
    is_row = name.startswith('row')
    # event: inserted, deleted or reordered?
    if name.endswith('inserted'):
      added = True
    elif name.endswith('deleted'):
      added = False
    else: # re-ordered
      added = None
    return (TableChange(por, is_row, added, first_child_por,
                        last_child_por, **kwargs),)

  def _handleScreenEvent(self, event, **kwargs):
    '''
    Creates an L{AEEvent.ScreenChange} event in response to a visible data or
    bounds change on an object or text.
    
    @param event: Raw visibility or bounds change event
    @type event: L{pyLinAcc.Event.Event}
    @param kwargs: Parameters to be passed to any created L{AEEvent}
    @type kwargs: dictionary
    @return: L{AEEvent.ScreenChange} event
    @rtype: tuple of L{AEEvent}
    '''
    major = event.type.major
    if major.startswith('text'):
      kind = ScreenChange.TEXT_BOUNDS
    elif major.startswith('bounds'):
      kind = ScreenChange.OBJECT_BOUNDS
    else:
      kind = ScreenChange.VISIBLE_DATA
    por = POR(self.subject, None, 0, incomplete=True)
    return (ScreenChange(por, kind, **kwargs),)
