'''
Defines a class that subclasses L{GSpeech} to support the Festival speech engine
via gnome-speech.

@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import AEOutput
import GSpeech

class FestivalStyleDefault(AEOutput.AudioStyleDefault):
  '''
  Empty shell for a default style class that will have its attributes 
  programmatically determined.
  '''
  pass
FestivalStyleDefault = FestivalStyleDefault() # singleton

class FestivalSpeech(GSpeech.GSpeech):
  '''
  Defines a class to send output from LSR to Festival using the gnome-speech
  API. Stores the Festival speech server OAFIID in L{DEVICE_IID} so that the
  base class can initialize the engine.
  '''
  DEVICE_IID = 'OAFIID:GNOME_Speech_SynthesisDriver_Festival:proto0.3'
  USE_THREAD = False
  
  def createDistinctStyles(self, num_styles):
    '''
    Creates num_styles new style objects and returns them to the caller. The 
    style objects are maximally distinct in terms of their voice modulo the
    number of voices.
    
    @param num_styles: Number of styles to create
    @type num_styles: integer
    '''
    styles = []
    # bounds on the number of voices
    FestivalStyleDefault.MaxVoice = len(self.voices)-1
    FestivalStyleDefault.MinVoice = 0
    for i in range(0, num_styles):
      # compute a valid voice number
      v_num = i % len(self.voices)
      # get that voice from the list of all voices
      v = self.voices[v_num]
      # create a speaker with that voice
      sp = self.driver.createSpeaker(v)
      # create a style object
      s = AEOutput.Style(FestivalStyleDefault)
      # store the current voice number
      s.Voice = v_num
      # iterate over the supported parameters and stick them on the style object
      # as style attributes; this may give us more than our known default set
      # of styles, but it's OK because they'll just be ignored by LSR
      for param in sp.getSupportedParameters():
        cap_name = param.name.title()
        val = sp.getParameterValue(param.name)
        try:
          def_val = getattr(FestivalStyleDefault, cap_name)
        except AttributeError:
          # put this value on the default if it does not already exist
          target = FestivalStyleDefault
        else:
          # put value on the current style if it differs from the default
          if def_val is None:
            target = FestivalStyleDefault
          elif val != def_val:
            target = s
        setattr(target, cap_name, val)
        # max and min should probably go in the default style; I guess they
        # could change per speaker, but that would be really weird
        setattr(FestivalStyleDefault, 'Max%s' % cap_name, param.max) 
        setattr(FestivalStyleDefault, 'Min%s' % cap_name, param.min) 
      styles.append(s)
    return styles

  def _applyStyle(self, style):
    '''
    Applies a given style to this output device. All output following this 
    call will be presented in the given style until this method is called again
    with a different style.

    @param style: Style to apply to future output
    @type style: L{AEOutput.Style}
    '''  
    sp = self.driver.createSpeaker(self.voices[style.Voice])
    if self.speaker is not None:
      self.speaker.unref()
    
    for param in sp.getSupportedParameters():
      cap_name = param.name.title()
      sp.setParameterValue(param.name, getattr(style, cap_name))
    self.speaker = sp

  def getDefaultStyle(self):
    '''
    Gets the default style for Festival.
    
    @return: default style of the device
    @rtype: L{AEOutput.AudioStyleDefault}
    '''
    return FestivalStyleDefault
  