'''
Contains the L{GSpeech} abstract base class which provides support for all
gnome-speech devices. Device definitions using gnome-speech should derive from
this class and override the L{GSpeech.USE_THREAD} and L{GSpeech.COMMAND_CHARS}
properties if desired. The subclass should also provide createDistinctStyles
and _applyStyle methods. This module should never be directly registered as
its own speech device with L{UIRegistrar} as the GSpeech class does not fully
implement the L{AEOutput.Base.AEOutput} interface.

@var MAX_BUFFER: Maximum number of characters to buffer before forcing a 
  L{GSpeech.sendTalk} call.
@type MAX_BUFFER: integer
@var SAY_WAIT_RETRY: Time to wait between calls to say if say fails
@type SAY_WAIT_RETRY: float

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import time
import AEOutput
from i18n import _

import ORBit, bonobo
# try to get the typelib
ORBit.load_typelib('GNOME_Speech')
import GNOME.Speech, GNOME__POA.Speech

# constants specific to GSpeech devices
SAY_WAIT_RETRY = 0.01
MAX_BUFFER = 80

class GSpeech(AEOutput.Audio):
  '''
  Defines an abstract base class to send output from LSR to a speech device via
  gnome-speech U{http://cvs.gnome.org/viewcvs/gnome-speech/}.

  To be compliant with LSR requirements, this implements the interface defined 
  in the L{AEOutput.Audio} class.
  
  @ivar driver: Reference to a speech engine server
  @type driver: GNOME.Speech.SynthesisDriver
  @ivar speaker: Speaker that will synthesize speech from buffered text
  @type speaker: GNOME.Speech.Speaker
  @ivar buffer: Buffer of text and styles to be sent to the device
  @type buffer: list
  @cvar DEVICE_IID: Interface identifier for the gnome-speech device. Defaults 
    to None and should be overridden in a subclass.
  @type DEVICE_IID: string
  ''' 
  DEVICE_IID = None
  
  def _applyStyle(self, style):
    '''
    Applies a new style immediately to the synthesis engine. Must be overridden
    in a subclass.
    
    @raise NotImplementedError: Always
    '''
    raise NotImplementedError
    
  def init(self):
    '''
    Initializes the speech driver through gnome-speech.

    @raise AEOutput.InitError: When the device can not be initialized
    '''
    self.driver = None
    self.speaker = None
    self.voices = []
    self.buffer = []
    
    # initialize CORBA
    ORBit.CORBA.ORB_init()
    # try to activate Festival
    self.driver = bonobo.activation.activate_from_id(self.DEVICE_IID, 0, False)
    try:
      # check if already initialized
      bInit = self.driver.isInitialized()
    except AttributeError:
      # driver is None when the engine is not available
      raise AEOutput.InitError
    # could have been initialized by someone else
    if not bInit:
      if not self.driver.driverInit():
        # driverInit fails when engine is not in a "good state"
        raise AEOutput.InitError
    # store all voices for later reference
    self.voices = self.driver.getAllVoices()

  def _say(self, text):
    '''
    Loops until L{speaker}.say succeeds by returning a non-negative value. 

    @note: gnome-speech returns a negative value when there is a backlog of 
        events that must be processed by a GNOME.Speech.SpeechCallback before
        say will succeed
    @todo: PP: store the return value of say to correlate callback markers
    @param text: Text to say
    @type text: string
    '''
    while 1:
      try:
        text = text.encode('utf-8', 'replace')
        sid = self.speaker.say(text)
      except (TypeError, UnicodeDecodeError), e:
        break
      if sid >= 0: break
      time.sleep(SAY_WAIT_RETRY)

  def close(self):
    '''
    Stops and closes the speech device.
    '''
    if self.speaker is not None:
      self.speaker.stop()
    del self.speaker
    del self.driver

  def sendStop(self, style=None):
    '''
    Stops speech immediately.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    if self.speaker is not None:
      self.speaker.stop()
    self.buffer = []

  def sendTalk(self, style=None):
    '''
    Begins synthesis of the buffered text.
  
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    stream = []
    for text, style in self.buffer:
      if style is not None:
        self._applyStyle(style)
        self._say(' '.join(stream))
        stream = []
      if text is not None:
        stream.append(text)
    if stream:
      self._say(' '.join(stream))
    self.buffer = []

  def sendString(self, text, style):
    '''
    Adds the given string to the text to be synthesized. The string is buffered
    locally in this object since gnome-speech does not support buffering in
    the speech engine driver, even if the engine does support it.

    @param text: String to buffer on the device
    @type text: string
    @param style: Style with which this string should be output; None means no
      style change should be applied
    @type style: integer
    '''
    # just buffer the text since gnome-speech will try speaking it right away
    # actually send the text in sendTalk
    if style is not None:
      style = style.copy()
    self.buffer.append((text, style))

  def isActive(self):
    '''
    Indicates whether the device is active meaning it is busy doing output or
    has text waiting to be output.

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    return self.speaker.isSpeaking()

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    @return: The localized name for the device
    @rtype: string
    '''
    return '%s %s' % (self.driver.synthesizerName,
                      self.driver.synthesizerVersion)
   
  def sendIndex(self, marker, style):
    '''
    Sends an index marker to the device driver. The driver should notify the
    device when the marker has been reached during output.
    
    @todo: PP: implement when index working

    @param style: Style indicating channel in which the marker will be appended
    @type style: L{AEOutput.Style}
    @param marker: The value that should be returned when output has been 
      processed up to this point
    @type marker: integer
    '''
    pass
  
