/** vim:sw=4:sts=4
 * Hash lookup routines for minimal perfect hashes as generated by the
 * cmph utility.
 */

#include "hash-cmph.h"
#include "config.h"		    /* CMPH_ALGORITHM */
#include <stdio.h>		    /* NULL */
#include <endian.h>

/**
 * Hash table lookup.  The key is used to calculate a bucket number.  The
 * hash value stored there is verified; if it matches, then an entry has
 * been found.  The offset stored in the bucket is used to find the data.
 * The length of the entry is determined by looking at the following bucket;
 * the data of all buckets is stored in order.
 *
 * Returns: NULL on failure, otherwise the pointer to the data; *datalen is
 * filled with the length in bytes of the data.
 */
const unsigned char *hash_search(const struct hash_info *hi, const char *key,
    int keylen, int *datalen)
{
    int bucket_nr, bucket_size = hi->offset_size + hi->hash_size;
    const unsigned char *bucket;
    unsigned int hash_value, data_offset, data_offset2, mask;

    /* determine the bucket number, and the bucket address */
    bucket_nr = CMPH_ALGORITHM(hi, (const unsigned char*) key, keylen,
	&hash_value);
    bucket = hi->index + bucket_nr * bucket_size;

    /* check hash value - but just the first "hash_size" bytes. */
    mask = (1 << (hi->hash_size << 3)) - 1;
    unsigned int val = * (unsigned int*) bucket;

#if (__BYTE_ORDER == __BIG_ENDIAN)
    /* Correct for Big Endianness */
    unsigned int shift = (sizeof(unsigned int) - hi->hash_size) << 3;
    val >>= shift;
#endif
    if ((hash_value ^ val) & mask)
	return NULL;

    /* found! get the data address */
    bucket += hi->hash_size;
    data_offset = * (unsigned int*) bucket;

    /* get next bucket to determine the length of the data entry */
    bucket += bucket_size;
    data_offset2 = * (unsigned int*)bucket;
    mask = (1 << (hi->offset_size << 3)) - 1;

#if (__BYTE_ORDER == __BIG_ENDIAN)
    data_offset >>= shift;
    data_offset2 >>= shift;
#endif

    *datalen = (data_offset2 & mask) - (data_offset & mask);
    return hi->data + (data_offset & mask);
}

