/* -*- mode: c; c-basic-offset: 8; indent-tabs-mode: nil; -*-
 * vim:expandtab:shiftwidth=8:tabstop=8:
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <liblustre.h>
#include <lustre_lib.h>
#include <lustre/lustre_idl.h>
#include <lustre_disk.h>

#define BLANK_LINE()                                            \
do {                                                            \
        printf("\n");                                           \
} while(0)

#define COMMENT(c)                                              \
do {                                                            \
        printf("        /* "c" */\n");                          \
} while(0)

#define STRINGIFY(a) #a


#define CHECK_CDEFINE(a)                                        \
        printf("        CLASSERT("#a" == "STRINGIFY(a) ");\n")

#define CHECK_CVALUE(a)                                         \
        printf("        CLASSERT("#a" == %lld);\n", (long long)a)

#define CHECK_DEFINE(a)                                         \
do {                                                            \
        printf("        LASSERTF("#a" == "STRINGIFY(a)          \
               ",\" found %%lld\\n\",\n                 "       \
               "(long long)"#a");\n");   \
} while(0)

#define CHECK_VALUE(a)                                          \
do {                                                            \
        printf("        LASSERTF("#a                            \
               " == %lld, \" found %%lld\\n\",\n                 "\
               "(long long)"#a");\n", (long long)a);            \
} while(0)

#define CHECK_VALUE_64(a)                                       \
do {                                                            \
        printf("        LASSERTF("#a                            \
               " == %lldULL, \" found %%lld\\n\",\n                 "\
               "(long long)"#a");\n", (long long)a);            \
} while(0)

#define CHECK_MEMBER_OFFSET(s,m)                                \
do {                                                            \
        CHECK_VALUE((int)offsetof(struct s, m));                \
} while(0)

#define CHECK_MEMBER_SIZEOF(s,m)                                \
do {                                                            \
        CHECK_VALUE((int)sizeof(((struct s *)0)->m));           \
} while(0)

#define CHECK_MEMBER(s,m)                                       \
do {                                                            \
        CHECK_MEMBER_OFFSET(s, m);                              \
        CHECK_MEMBER_SIZEOF(s, m);                              \
} while(0)

#define CHECK_STRUCT(s)                                         \
do {                                                            \
        COMMENT("Checks for struct "#s);                        \
                CHECK_VALUE((int)sizeof(struct s));             \
} while(0)


static void
check_lustre_handle(void)
{
        BLANK_LINE();
        CHECK_STRUCT(lustre_handle);
        CHECK_MEMBER(lustre_handle, cookie);
}

void
check_lustre_msg_v1(void)
{
        BLANK_LINE();
        CHECK_STRUCT(lustre_msg_v1);
        CHECK_MEMBER(lustre_msg_v1, lm_handle);
        CHECK_MEMBER(lustre_msg_v1, lm_magic);
        CHECK_MEMBER(lustre_msg_v1, lm_type);
        CHECK_MEMBER(lustre_msg_v1, lm_version);
        CHECK_MEMBER(lustre_msg_v1, lm_opc);
        CHECK_MEMBER(lustre_msg_v1, lm_last_xid);
        CHECK_MEMBER(lustre_msg_v1, lm_last_committed);
        CHECK_MEMBER(lustre_msg_v1, lm_transno);
        CHECK_MEMBER(lustre_msg_v1, lm_status);
        CHECK_MEMBER(lustre_msg_v1, lm_flags);
        CHECK_MEMBER(lustre_msg_v1, lm_conn_cnt);
        CHECK_MEMBER(lustre_msg_v1, lm_bufcount);
        CHECK_MEMBER(lustre_msg_v1, lm_buflens[0]);
}

void
check_lustre_msg_v2(void)
{
        BLANK_LINE();
        CHECK_STRUCT(lustre_msg_v2);
        CHECK_MEMBER(lustre_msg_v2, lm_bufcount);
        CHECK_MEMBER(lustre_msg_v2, lm_secflvr);
        CHECK_MEMBER(lustre_msg_v2, lm_magic);
        CHECK_MEMBER(lustre_msg_v2, lm_repsize);
        CHECK_MEMBER(lustre_msg_v2, lm_cksum);
        CHECK_MEMBER(lustre_msg_v2, lm_padding_1);
        CHECK_MEMBER(lustre_msg_v2, lm_padding_2);
        CHECK_MEMBER(lustre_msg_v2, lm_padding_3);
        CHECK_MEMBER(lustre_msg_v2, lm_buflens[0]);
}

void
check_ptlrpc_body(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ptlrpc_body);
        CHECK_MEMBER(ptlrpc_body, pb_handle);
        CHECK_MEMBER(ptlrpc_body, pb_type);
        CHECK_MEMBER(ptlrpc_body, pb_version);
        CHECK_MEMBER(ptlrpc_body, pb_opc);
        CHECK_MEMBER(ptlrpc_body, pb_status);
        CHECK_MEMBER(ptlrpc_body, pb_last_xid);
        CHECK_MEMBER(ptlrpc_body, pb_last_seen);
        CHECK_MEMBER(ptlrpc_body, pb_last_committed);
        CHECK_MEMBER(ptlrpc_body, pb_transno);
        CHECK_MEMBER(ptlrpc_body, pb_flags);
        CHECK_MEMBER(ptlrpc_body, pb_op_flags);
        CHECK_MEMBER(ptlrpc_body, pb_conn_cnt);
        CHECK_MEMBER(ptlrpc_body, pb_timeout);
        CHECK_MEMBER(ptlrpc_body, pb_service_time);
        CHECK_MEMBER(ptlrpc_body, pb_slv);
        CHECK_MEMBER(ptlrpc_body, pb_limit);
}

static void check_obd_connect_data(void)
{
        BLANK_LINE();
        CHECK_STRUCT(obd_connect_data);
        CHECK_MEMBER(obd_connect_data, ocd_connect_flags);
        CHECK_MEMBER(obd_connect_data, ocd_version);
        CHECK_MEMBER(obd_connect_data, ocd_grant);
        CHECK_MEMBER(obd_connect_data, ocd_index);
        CHECK_MEMBER(obd_connect_data, ocd_brw_size);
        CHECK_MEMBER(obd_connect_data, ocd_ibits_known);
        CHECK_MEMBER(obd_connect_data, ocd_nllu);
        CHECK_MEMBER(obd_connect_data, ocd_nllg);
        CHECK_MEMBER(obd_connect_data, padding1);
        CHECK_MEMBER(obd_connect_data, padding2);
        CHECK_MEMBER(obd_connect_data, padding3);
        CHECK_MEMBER(obd_connect_data, padding4);

        CHECK_CDEFINE(OBD_CONNECT_RDONLY);
        CHECK_CDEFINE(OBD_CONNECT_INDEX);
        CHECK_CDEFINE(OBD_CONNECT_GRANT);
        CHECK_CDEFINE(OBD_CONNECT_SRVLOCK);
        CHECK_CDEFINE(OBD_CONNECT_VERSION);
        CHECK_CDEFINE(OBD_CONNECT_REQPORTAL);
        CHECK_CDEFINE(OBD_CONNECT_ACL);
        CHECK_CDEFINE(OBD_CONNECT_XATTR);
        CHECK_CDEFINE(OBD_CONNECT_CROW);
        CHECK_CDEFINE(OBD_CONNECT_TRUNCLOCK);
        CHECK_CDEFINE(OBD_CONNECT_TRANSNO);
        CHECK_CDEFINE(OBD_CONNECT_IBITS);
        CHECK_CDEFINE(OBD_CONNECT_JOIN);
        CHECK_CDEFINE(OBD_CONNECT_ATTRFID);
        CHECK_CDEFINE(OBD_CONNECT_NODEVOH);
        CHECK_CDEFINE(OBD_CONNECT_LCL_CLIENT);
        CHECK_CDEFINE(OBD_CONNECT_RMT_CLIENT);
        CHECK_CDEFINE(OBD_CONNECT_BRW_SIZE);
        CHECK_CDEFINE(OBD_CONNECT_QUOTA64);
        CHECK_CDEFINE(OBD_CONNECT_MDS_CAPA);
        CHECK_CDEFINE(OBD_CONNECT_OSS_CAPA);
        CHECK_CDEFINE(OBD_CONNECT_CANCELSET);
        CHECK_CDEFINE(OBD_CONNECT_SOM);
        CHECK_CDEFINE(OBD_CONNECT_AT);
        CHECK_CDEFINE(OBD_CONNECT_LRU_RESIZE);
        CHECK_CDEFINE(OBD_CONNECT_MDS_MDS);
        CHECK_CDEFINE(OBD_CONNECT_REAL);
}

static void
check_obdo(void)
{
        BLANK_LINE();
        CHECK_STRUCT(obdo);
        CHECK_MEMBER(obdo, o_valid);
        CHECK_MEMBER(obdo, o_id);
        CHECK_MEMBER(obdo, o_gr);
        CHECK_MEMBER(obdo, o_fid);
        CHECK_MEMBER(obdo, o_size);
        CHECK_MEMBER(obdo, o_mtime);
        CHECK_MEMBER(obdo, o_atime);
        CHECK_MEMBER(obdo, o_ctime);
        CHECK_MEMBER(obdo, o_blocks);
        CHECK_MEMBER(obdo, o_grant);
        CHECK_MEMBER(obdo, o_blksize);
        CHECK_MEMBER(obdo, o_mode);
        CHECK_MEMBER(obdo, o_uid);
        CHECK_MEMBER(obdo, o_gid);
        CHECK_MEMBER(obdo, o_flags);
        CHECK_MEMBER(obdo, o_nlink);
        CHECK_MEMBER(obdo, o_generation);
        CHECK_MEMBER(obdo, o_misc);
        CHECK_MEMBER(obdo, o_easize);
        CHECK_MEMBER(obdo, o_mds);
        CHECK_MEMBER(obdo, o_stripe_idx);
        CHECK_MEMBER(obdo, o_padding_1);
        CHECK_MEMBER(obdo, o_inline);

        CHECK_VALUE(OBD_INLINESZ);

        CHECK_CDEFINE(OBD_MD_FLID);
        CHECK_CDEFINE(OBD_MD_FLATIME);
        CHECK_CDEFINE(OBD_MD_FLMTIME);
        CHECK_CDEFINE(OBD_MD_FLCTIME);
        CHECK_CDEFINE(OBD_MD_FLSIZE);
        CHECK_CDEFINE(OBD_MD_FLBLOCKS);
        CHECK_CDEFINE(OBD_MD_FLBLKSZ);
        CHECK_CDEFINE(OBD_MD_FLMODE);
        CHECK_CDEFINE(OBD_MD_FLTYPE);
        CHECK_CDEFINE(OBD_MD_FLUID);
        CHECK_CDEFINE(OBD_MD_FLGID);
        CHECK_CDEFINE(OBD_MD_FLFLAGS);
        CHECK_CDEFINE(OBD_MD_FLNLINK);
        CHECK_CDEFINE(OBD_MD_FLGENER);
        CHECK_CDEFINE(OBD_MD_FLINLINE);
        CHECK_CDEFINE(OBD_MD_FLRDEV);
        CHECK_CDEFINE(OBD_MD_FLEASIZE);
        CHECK_CDEFINE(OBD_MD_LINKNAME);
        CHECK_CDEFINE(OBD_MD_FLHANDLE);
        CHECK_CDEFINE(OBD_MD_FLCKSUM);
        CHECK_CDEFINE(OBD_MD_FLQOS);
        CHECK_CDEFINE(OBD_MD_FLCOOKIE);
        CHECK_CDEFINE(OBD_MD_FLGROUP);
        CHECK_CDEFINE(OBD_MD_FLFID);
        CHECK_CDEFINE(OBD_MD_FLEPOCH);
        CHECK_CDEFINE(OBD_MD_FLGRANT);
        CHECK_CDEFINE(OBD_MD_FLDIREA);
        CHECK_CDEFINE(OBD_MD_FLUSRQUOTA);
        CHECK_CDEFINE(OBD_MD_FLGRPQUOTA);
        CHECK_CDEFINE(OBD_MD_FLMODEASIZE);
        CHECK_CDEFINE(OBD_MD_MDS);
        CHECK_CDEFINE(OBD_MD_REINT);
        CHECK_CDEFINE(OBD_MD_FLXATTR);
        CHECK_CDEFINE(OBD_MD_FLXATTRLS);
        CHECK_CDEFINE(OBD_MD_FLXATTRRM);
        CHECK_CDEFINE(OBD_MD_FLACL);

        CHECK_CDEFINE(OBD_FL_INLINEDATA);
        CHECK_CDEFINE(OBD_FL_OBDMDEXISTS);
        CHECK_CDEFINE(OBD_FL_DELORPHAN);
        CHECK_CDEFINE(OBD_FL_NORPC);
        CHECK_CDEFINE(OBD_FL_IDONLY);
        CHECK_CDEFINE(OBD_FL_RECREATE_OBJS);
        CHECK_CDEFINE(OBD_FL_DEBUG_CHECK);
        CHECK_CDEFINE(OBD_FL_NO_USRQUOTA);
        CHECK_CDEFINE(OBD_FL_NO_GRPQUOTA);
        CHECK_CDEFINE(OBD_FL_CREATE_CROW);
}

static void
check_lov_mds_md_v1(void)
{
        BLANK_LINE();
        CHECK_STRUCT(lov_mds_md_v1);
        CHECK_MEMBER(lov_mds_md_v1, lmm_magic);
        CHECK_MEMBER(lov_mds_md_v1, lmm_pattern);
        CHECK_MEMBER(lov_mds_md_v1, lmm_object_id);
        CHECK_MEMBER(lov_mds_md_v1, lmm_object_gr);
        CHECK_MEMBER(lov_mds_md_v1, lmm_stripe_size);
        CHECK_MEMBER(lov_mds_md_v1, lmm_stripe_count);
        CHECK_MEMBER(lov_mds_md_v1, lmm_objects);

        BLANK_LINE();
        CHECK_STRUCT(lov_ost_data_v1);
        CHECK_MEMBER(lov_ost_data_v1, l_object_id);
        CHECK_MEMBER(lov_ost_data_v1, l_object_gr);
        CHECK_MEMBER(lov_ost_data_v1, l_ost_gen);
        CHECK_MEMBER(lov_ost_data_v1, l_ost_idx);

        CHECK_CDEFINE(LOV_MAGIC_V1);
        CHECK_CDEFINE(LOV_MAGIC_JOIN);

        CHECK_VALUE(LOV_PATTERN_RAID0);
        CHECK_VALUE(LOV_PATTERN_RAID1);
}

static void
check_lov_mds_md_join(void)
{
        BLANK_LINE();
        CHECK_STRUCT(lov_mds_md_join);
        CHECK_MEMBER(lov_mds_md_join, lmmj_md);
        CHECK_MEMBER(lov_mds_md_join, lmmj_array_id);
        CHECK_MEMBER(lov_mds_md_join, lmmj_extent_count);
}

static void
check_obd_statfs(void)
{
        BLANK_LINE();
        CHECK_STRUCT(obd_statfs);
        CHECK_MEMBER(obd_statfs, os_type);
        CHECK_MEMBER(obd_statfs, os_blocks);
        CHECK_MEMBER(obd_statfs, os_bfree);
        CHECK_MEMBER(obd_statfs, os_bavail);
        CHECK_MEMBER(obd_statfs, os_ffree);
        CHECK_MEMBER(obd_statfs, os_fsid);
        CHECK_MEMBER(obd_statfs, os_bsize);
        CHECK_MEMBER(obd_statfs, os_namelen);
        CHECK_MEMBER(obd_statfs, os_state);
        CHECK_MEMBER(obd_statfs, os_spare1);
        CHECK_MEMBER(obd_statfs, os_spare2);
        CHECK_MEMBER(obd_statfs, os_spare3);
        CHECK_MEMBER(obd_statfs, os_spare4);
        CHECK_MEMBER(obd_statfs, os_spare5);
        CHECK_MEMBER(obd_statfs, os_spare6);
        CHECK_MEMBER(obd_statfs, os_spare7);
        CHECK_MEMBER(obd_statfs, os_spare8);
        CHECK_MEMBER(obd_statfs, os_spare9);
}

static void
check_obd_ioobj(void)
{
        BLANK_LINE();
        CHECK_STRUCT(obd_ioobj);
        CHECK_MEMBER(obd_ioobj, ioo_id);
        CHECK_MEMBER(obd_ioobj, ioo_gr);
        CHECK_MEMBER(obd_ioobj, ioo_type);
        CHECK_MEMBER(obd_ioobj, ioo_bufcnt);
}

static void
check_obd_quotactl(void)
{
        BLANK_LINE();
        CHECK_STRUCT(obd_quotactl);
        CHECK_MEMBER(obd_quotactl, qc_cmd);
        CHECK_MEMBER(obd_quotactl, qc_type);
        CHECK_MEMBER(obd_quotactl, qc_id);
        CHECK_MEMBER(obd_quotactl, qc_stat);
        CHECK_MEMBER(obd_quotactl, qc_dqinfo);
        CHECK_MEMBER(obd_quotactl, qc_dqblk);

        BLANK_LINE();
        CHECK_STRUCT(obd_dqinfo);
        CHECK_MEMBER(obd_dqinfo, dqi_bgrace);
        CHECK_MEMBER(obd_dqinfo, dqi_igrace);
        CHECK_MEMBER(obd_dqinfo, dqi_flags);
        CHECK_MEMBER(obd_dqinfo, dqi_valid);

        BLANK_LINE();
        CHECK_STRUCT(obd_dqblk);
        CHECK_MEMBER(obd_dqblk, dqb_bhardlimit);
        CHECK_MEMBER(obd_dqblk, dqb_bsoftlimit);
        CHECK_MEMBER(obd_dqblk, dqb_curspace);
        CHECK_MEMBER(obd_dqblk, dqb_ihardlimit);
        CHECK_MEMBER(obd_dqblk, dqb_isoftlimit);
        CHECK_MEMBER(obd_dqblk, dqb_curinodes);
        CHECK_MEMBER(obd_dqblk, dqb_btime);
        CHECK_MEMBER(obd_dqblk, dqb_itime);
        CHECK_MEMBER(obd_dqblk, dqb_valid);
        CHECK_MEMBER(obd_dqblk, padding);

        CHECK_DEFINE(Q_QUOTACHECK);
        CHECK_DEFINE(Q_INITQUOTA);
        CHECK_DEFINE(Q_GETOINFO);
        CHECK_DEFINE(Q_GETOQUOTA);
}

static void
check_niobuf_remote(void)
{
        BLANK_LINE();
        CHECK_STRUCT(niobuf_remote);
        CHECK_MEMBER(niobuf_remote, offset);
        CHECK_MEMBER(niobuf_remote, len);
        CHECK_MEMBER(niobuf_remote, flags);

        CHECK_VALUE(OBD_BRW_READ);
        CHECK_VALUE(OBD_BRW_WRITE);
        CHECK_VALUE(OBD_BRW_SYNC);
        CHECK_VALUE(OBD_BRW_FROM_GRANT);
        CHECK_VALUE(OBD_BRW_NOQUOTA);
}

static void
check_ost_body(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ost_body);
        CHECK_MEMBER(ost_body, oa);
}

static void
check_ll_fid(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ll_fid);
        CHECK_MEMBER(ll_fid, id);
        CHECK_MEMBER(ll_fid, generation);
        CHECK_MEMBER(ll_fid, f_type);
}

static void
check_mds_status_req(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_status_req);
        CHECK_MEMBER(mds_status_req, flags);
        CHECK_MEMBER(mds_status_req, repbuf);
}

static void
check_mds_body(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_body);
        CHECK_MEMBER(mds_body, fid1);
        CHECK_MEMBER(mds_body, fid2);
        CHECK_MEMBER(mds_body, handle);
        CHECK_MEMBER(mds_body, size);
        CHECK_MEMBER(mds_body, blocks);
        CHECK_MEMBER(mds_body, io_epoch);
        CHECK_MEMBER(mds_body, ino);
        CHECK_MEMBER(mds_body, valid);
        CHECK_MEMBER(mds_body, fsuid);
        CHECK_MEMBER(mds_body, fsgid);
        CHECK_MEMBER(mds_body, capability);
        CHECK_MEMBER(mds_body, mode);
        CHECK_MEMBER(mds_body, uid);
        CHECK_MEMBER(mds_body, gid);
        CHECK_MEMBER(mds_body, mtime);
        CHECK_MEMBER(mds_body, ctime);
        CHECK_MEMBER(mds_body, atime);
        CHECK_MEMBER(mds_body, flags);
        CHECK_MEMBER(mds_body, rdev);
        CHECK_MEMBER(mds_body, nlink);
        CHECK_MEMBER(mds_body, generation);
        CHECK_MEMBER(mds_body, suppgid);
        CHECK_MEMBER(mds_body, eadatasize);
        CHECK_MEMBER(mds_body, aclsize);
        CHECK_MEMBER(mds_body, max_mdsize);
        CHECK_MEMBER(mds_body, max_cookiesize);
        CHECK_MEMBER(mds_body, padding_4);

        CHECK_VALUE(FMODE_READ);
        CHECK_VALUE(FMODE_WRITE);
        CHECK_VALUE(MDS_FMODE_EXEC);

        CHECK_CDEFINE(MDS_OPEN_CREAT);
        CHECK_CDEFINE(MDS_OPEN_EXCL);
        CHECK_CDEFINE(MDS_OPEN_TRUNC);
        CHECK_CDEFINE(MDS_OPEN_APPEND);
        CHECK_CDEFINE(MDS_OPEN_SYNC);
        CHECK_CDEFINE(MDS_OPEN_DIRECTORY);
        CHECK_CDEFINE(MDS_OPEN_DELAY_CREATE);
        CHECK_CDEFINE(MDS_OPEN_OWNEROVERRIDE);
        CHECK_CDEFINE(MDS_OPEN_JOIN_FILE);
        CHECK_CDEFINE(MDS_OPEN_HAS_EA);
        CHECK_CDEFINE(MDS_OPEN_HAS_OBJS);

        /* these should be identical to their EXT3_*_FL counterparts, and
         * are redefined only to avoid dragging in ext3_fs.h */
        CHECK_CDEFINE(MDS_SYNC_FL);
        CHECK_CDEFINE(MDS_IMMUTABLE_FL);
        CHECK_CDEFINE(MDS_APPEND_FL);
        CHECK_CDEFINE(MDS_NOATIME_FL);
        CHECK_CDEFINE(MDS_DIRSYNC_FL);
        CHECK_CDEFINE(MDS_BFLAG_EXT_FLAGS);

        CHECK_CDEFINE(MDS_INODELOCK_LOOKUP);
        CHECK_CDEFINE(MDS_INODELOCK_UPDATE);
        CHECK_CDEFINE(MDS_INODELOCK_OPEN);
}

static void
check_mds_rec_setattr(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_rec_setattr);
        CHECK_MEMBER(mds_rec_setattr, sa_opcode);
        CHECK_MEMBER(mds_rec_setattr, sa_fsuid);
        CHECK_MEMBER(mds_rec_setattr, sa_fsgid);
        CHECK_MEMBER(mds_rec_setattr, sa_cap);
        CHECK_MEMBER(mds_rec_setattr, sa_suppgid);
        CHECK_MEMBER(mds_rec_setattr, sa_mode);
        CHECK_MEMBER(mds_rec_setattr, sa_fid);
        CHECK_MEMBER(mds_rec_setattr, sa_valid);
        CHECK_MEMBER(mds_rec_setattr, sa_size);
        CHECK_MEMBER(mds_rec_setattr, sa_mtime);
        CHECK_MEMBER(mds_rec_setattr, sa_atime);
        CHECK_MEMBER(mds_rec_setattr, sa_ctime);
        CHECK_MEMBER(mds_rec_setattr, sa_uid);
        CHECK_MEMBER(mds_rec_setattr, sa_gid);
        CHECK_MEMBER(mds_rec_setattr, sa_attr_flags);
        CHECK_CDEFINE(MDS_ATTR_MODE);
        CHECK_CDEFINE(MDS_ATTR_UID);
        CHECK_CDEFINE(MDS_ATTR_GID);
        CHECK_CDEFINE(MDS_ATTR_SIZE);
        CHECK_CDEFINE(MDS_ATTR_ATIME);
        CHECK_CDEFINE(MDS_ATTR_MTIME);
        CHECK_CDEFINE(MDS_ATTR_CTIME);
        CHECK_CDEFINE(MDS_ATTR_ATIME_SET);
        CHECK_CDEFINE(MDS_ATTR_MTIME_SET);
        CHECK_CDEFINE(MDS_ATTR_FORCE);
        CHECK_CDEFINE(MDS_ATTR_ATTR_FLAG);
        CHECK_CDEFINE(MDS_ATTR_KILL_SUID);
        CHECK_CDEFINE(MDS_ATTR_KILL_SGID);
        CHECK_CDEFINE(MDS_ATTR_CTIME_SET);
        CHECK_CDEFINE(MDS_ATTR_FROM_OPEN);
}

static void
check_mds_rec_create(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_rec_create);
        CHECK_MEMBER(mds_rec_create, cr_opcode);
        CHECK_MEMBER(mds_rec_create, cr_fsuid);
        CHECK_MEMBER(mds_rec_create, cr_fsgid);
        CHECK_MEMBER(mds_rec_create, cr_cap);
        CHECK_MEMBER(mds_rec_create, cr_flags);
        CHECK_MEMBER(mds_rec_create, cr_mode);
        CHECK_MEMBER(mds_rec_create, cr_fid);
        CHECK_MEMBER(mds_rec_create, cr_replayfid);
        CHECK_MEMBER(mds_rec_create, cr_time);
        CHECK_MEMBER(mds_rec_create, cr_rdev);
        CHECK_MEMBER(mds_rec_create, cr_suppgid);
}

static void
check_mds_rec_link(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_rec_link);
        CHECK_MEMBER(mds_rec_link, lk_opcode);
        CHECK_MEMBER(mds_rec_link, lk_fsuid);
        CHECK_MEMBER(mds_rec_link, lk_fsgid);
        CHECK_MEMBER(mds_rec_link, lk_cap);
        CHECK_MEMBER(mds_rec_link, lk_suppgid1);
        CHECK_MEMBER(mds_rec_link, lk_suppgid2);
        CHECK_MEMBER(mds_rec_link, lk_fid1);
        CHECK_MEMBER(mds_rec_link, lk_fid2);
        CHECK_MEMBER(mds_rec_link, lk_time);
}

static void
check_mds_rec_unlink(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_rec_unlink);
        CHECK_MEMBER(mds_rec_unlink, ul_opcode);
        CHECK_MEMBER(mds_rec_unlink, ul_fsuid);
        CHECK_MEMBER(mds_rec_unlink, ul_fsgid);
        CHECK_MEMBER(mds_rec_unlink, ul_cap);
        CHECK_MEMBER(mds_rec_unlink, ul_suppgid);
        CHECK_MEMBER(mds_rec_unlink, ul_mode);
        CHECK_MEMBER(mds_rec_unlink, ul_fid1);
        CHECK_MEMBER(mds_rec_unlink, ul_fid2);
        CHECK_MEMBER(mds_rec_unlink, ul_time);
}

static void
check_mds_rec_rename(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_rec_rename);
        CHECK_MEMBER(mds_rec_rename, rn_opcode);
        CHECK_MEMBER(mds_rec_rename, rn_fsuid);
        CHECK_MEMBER(mds_rec_rename, rn_fsgid);
        CHECK_MEMBER(mds_rec_rename, rn_cap);
        CHECK_MEMBER(mds_rec_rename, rn_suppgid1);
        CHECK_MEMBER(mds_rec_rename, rn_suppgid2);
        CHECK_MEMBER(mds_rec_rename, rn_fid1);
        CHECK_MEMBER(mds_rec_rename, rn_fid2);
        CHECK_MEMBER(mds_rec_rename, rn_time);
}

static void
check_mds_rec_join(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_rec_join);
        CHECK_MEMBER(mds_rec_join, jr_fid);
        CHECK_MEMBER(mds_rec_join, jr_headsize);
}

static void
check_lov_desc(void)
{
        BLANK_LINE();
        CHECK_STRUCT(lov_desc);
        CHECK_MEMBER(lov_desc, ld_tgt_count);
        CHECK_MEMBER(lov_desc, ld_active_tgt_count);
        CHECK_MEMBER(lov_desc, ld_default_stripe_count);
        CHECK_MEMBER(lov_desc, ld_pattern);
        CHECK_MEMBER(lov_desc, ld_default_stripe_size);
        CHECK_MEMBER(lov_desc, ld_default_stripe_offset);
        CHECK_MEMBER(lov_desc, ld_qos_maxage);
        CHECK_MEMBER(lov_desc, ld_padding_1);
        CHECK_MEMBER(lov_desc, ld_padding_2);
        CHECK_MEMBER(lov_desc, ld_uuid);
}

static void
check_ldlm_res_id(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_res_id);
        CHECK_MEMBER(ldlm_res_id, name[RES_NAME_SIZE]);
}

static void
check_ldlm_extent(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_extent);
        CHECK_MEMBER(ldlm_extent, start);
        CHECK_MEMBER(ldlm_extent, end);
        CHECK_MEMBER(ldlm_extent, gid);
}

static void
check_ldlm_inodebits(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_inodebits);
        CHECK_MEMBER(ldlm_inodebits, bits);
}

static void
check_ldlm_flock(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_flock);
        CHECK_MEMBER(ldlm_flock, start);
        CHECK_MEMBER(ldlm_flock, end);
        CHECK_MEMBER(ldlm_flock, blocking_pid);
        CHECK_MEMBER(ldlm_flock, pid);
}

static void
check_ldlm_intent(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_intent);
        CHECK_MEMBER(ldlm_intent, opc);
}

static void
check_ldlm_resource_desc(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_resource_desc);
        CHECK_MEMBER(ldlm_resource_desc, lr_type);
        CHECK_MEMBER(ldlm_resource_desc, lr_padding);
        CHECK_MEMBER(ldlm_resource_desc, lr_name);
}

static void
check_ldlm_lock_desc(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_lock_desc);
        CHECK_MEMBER(ldlm_lock_desc, l_resource);
        CHECK_MEMBER(ldlm_lock_desc, l_req_mode);
        CHECK_MEMBER(ldlm_lock_desc, l_granted_mode);
        CHECK_MEMBER(ldlm_lock_desc, l_policy_data);
}

static void
check_ldlm_request(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_request);
        CHECK_MEMBER(ldlm_request, lock_flags);
        CHECK_MEMBER(ldlm_request, lock_count);
        CHECK_MEMBER(ldlm_request, lock_desc);
        CHECK_MEMBER(ldlm_request, lock_handle);
}

static void
check_ldlm_reply(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ldlm_reply);
        CHECK_MEMBER(ldlm_reply, lock_flags);
        CHECK_MEMBER(ldlm_reply, lock_padding);
        CHECK_MEMBER(ldlm_reply, lock_desc);
        CHECK_MEMBER(ldlm_reply, lock_handle);
        CHECK_MEMBER(ldlm_reply, lock_policy_res1);
        CHECK_MEMBER(ldlm_reply, lock_policy_res2);
}

static void
check_ldlm_lvb(void)
{
        BLANK_LINE();
        CHECK_STRUCT(ost_lvb);
        CHECK_MEMBER(ost_lvb, lvb_size);
        CHECK_MEMBER(ost_lvb, lvb_mtime);
        CHECK_MEMBER(ost_lvb, lvb_atime);
        CHECK_MEMBER(ost_lvb, lvb_ctime);
        CHECK_MEMBER(ost_lvb, lvb_blocks);
}


static void
check_llog_logid(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_logid);
        CHECK_MEMBER(llog_logid, lgl_oid);
        CHECK_MEMBER(llog_logid, lgl_ogr);
        CHECK_MEMBER(llog_logid, lgl_ogen);

        CHECK_CVALUE(OST_SZ_REC);
        CHECK_CVALUE(OST_RAID1_REC);
        CHECK_CVALUE(MDS_UNLINK_REC);
        CHECK_CVALUE(MDS_SETATTR_REC);
        CHECK_CVALUE(OBD_CFG_REC);
        CHECK_CVALUE(PTL_CFG_REC);
        CHECK_CVALUE(LLOG_GEN_REC);
        CHECK_CVALUE(LLOG_JOIN_REC);
        CHECK_CVALUE(LLOG_HDR_MAGIC);
        CHECK_CVALUE(LLOG_LOGID_MAGIC);
}

static void
check_llog_catid(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_catid);
        CHECK_MEMBER(llog_catid, lci_logid);
        CHECK_MEMBER(llog_catid, lci_padding1);
        CHECK_MEMBER(llog_catid, lci_padding2);
        CHECK_MEMBER(llog_catid, lci_padding3);
}

static void
check_llog_rec_hdr(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_rec_hdr);
        CHECK_MEMBER(llog_rec_hdr, lrh_len);
        CHECK_MEMBER(llog_rec_hdr, lrh_index);
        CHECK_MEMBER(llog_rec_hdr, lrh_type);
        CHECK_MEMBER(llog_rec_hdr, padding);
}

static void
check_llog_rec_tail(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_rec_tail);
        CHECK_MEMBER(llog_rec_tail, lrt_len);
        CHECK_MEMBER(llog_rec_tail, lrt_index);
}

static void
check_llog_logid_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_logid_rec);
        CHECK_MEMBER(llog_logid_rec, lid_hdr);
        CHECK_MEMBER(llog_logid_rec, lid_id);
        CHECK_MEMBER(llog_logid_rec, padding1);
        CHECK_MEMBER(llog_logid_rec, padding2);
        CHECK_MEMBER(llog_logid_rec, padding3);
        CHECK_MEMBER(llog_logid_rec, padding4);
        CHECK_MEMBER(llog_logid_rec, padding5);
        CHECK_MEMBER(llog_logid_rec, lid_tail);
}

static void
check_llog_create_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_create_rec);
        CHECK_MEMBER(llog_create_rec, lcr_hdr);
        CHECK_MEMBER(llog_create_rec, lcr_fid);
        CHECK_MEMBER(llog_create_rec, lcr_oid);
        CHECK_MEMBER(llog_create_rec, lcr_ogen);
        CHECK_MEMBER(llog_create_rec, padding);
}

static void
check_llog_orphan_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_orphan_rec);
        CHECK_MEMBER(llog_orphan_rec, lor_hdr);
        CHECK_MEMBER(llog_orphan_rec, lor_oid);
        CHECK_MEMBER(llog_orphan_rec, lor_ogen);
        CHECK_MEMBER(llog_orphan_rec, padding);
        CHECK_MEMBER(llog_orphan_rec, lor_tail);
}

static void
check_llog_unlink_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_unlink_rec);
        CHECK_MEMBER(llog_unlink_rec, lur_hdr);
        CHECK_MEMBER(llog_unlink_rec, lur_oid);
        CHECK_MEMBER(llog_unlink_rec, lur_ogen);
        CHECK_MEMBER(llog_unlink_rec, padding);
        CHECK_MEMBER(llog_unlink_rec, lur_tail);
}

static void
check_llog_setattr_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_setattr_rec);
        CHECK_MEMBER(llog_setattr_rec, lsr_hdr);
        CHECK_MEMBER(llog_setattr_rec, lsr_oid);
        CHECK_MEMBER(llog_setattr_rec, lsr_ogen);
        CHECK_MEMBER(llog_setattr_rec, lsr_uid);
        CHECK_MEMBER(llog_setattr_rec, lsr_gid);
        CHECK_MEMBER(llog_setattr_rec, padding);
        CHECK_MEMBER(llog_setattr_rec, lsr_tail);
}

static void
check_llog_size_change_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_size_change_rec);
        CHECK_MEMBER(llog_size_change_rec, lsc_hdr);
        CHECK_MEMBER(llog_size_change_rec, lsc_fid);
        CHECK_MEMBER(llog_size_change_rec, lsc_io_epoch);
        CHECK_MEMBER(llog_size_change_rec, padding);
        CHECK_MEMBER(llog_size_change_rec, lsc_tail);
}

static void
check_llog_gen(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_gen);
        CHECK_MEMBER(llog_gen, mnt_cnt);
        CHECK_MEMBER(llog_gen, conn_cnt);
}

static void
check_llog_gen_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_gen_rec);
        CHECK_MEMBER(llog_gen_rec, lgr_hdr);
        CHECK_MEMBER(llog_gen_rec, lgr_gen);
        CHECK_MEMBER(llog_gen_rec, lgr_tail);
}

static void
check_llog_log_hdr(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_log_hdr);
        CHECK_MEMBER(llog_log_hdr, llh_hdr);
        CHECK_MEMBER(llog_log_hdr, llh_timestamp);
        CHECK_MEMBER(llog_log_hdr, llh_count);
        CHECK_MEMBER(llog_log_hdr, llh_bitmap_offset);
        CHECK_MEMBER(llog_log_hdr, llh_size);
        CHECK_MEMBER(llog_log_hdr, llh_flags);
        CHECK_MEMBER(llog_log_hdr, llh_cat_idx);
        CHECK_MEMBER(llog_log_hdr, llh_tgtuuid);
        CHECK_MEMBER(llog_log_hdr, llh_reserved);
        CHECK_MEMBER(llog_log_hdr, llh_bitmap);
        CHECK_MEMBER(llog_log_hdr, llh_tail);
}

static void
check_llog_cookie(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_cookie);
        CHECK_MEMBER(llog_cookie, lgc_lgl);
        CHECK_MEMBER(llog_cookie, lgc_subsys);
        CHECK_MEMBER(llog_cookie, lgc_index);
        CHECK_MEMBER(llog_cookie, lgc_padding);
}

static void
check_llogd_body(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llogd_body);
        CHECK_MEMBER(llogd_body, lgd_logid);
        CHECK_MEMBER(llogd_body, lgd_ctxt_idx);
        CHECK_MEMBER(llogd_body, lgd_llh_flags);
        CHECK_MEMBER(llogd_body, lgd_index);
        CHECK_MEMBER(llogd_body, lgd_saved_index);
        CHECK_MEMBER(llogd_body, lgd_len);
        CHECK_MEMBER(llogd_body, lgd_cur_offset);

        CHECK_CVALUE(LLOG_ORIGIN_HANDLE_CREATE);
        CHECK_CVALUE(LLOG_ORIGIN_HANDLE_NEXT_BLOCK);
        CHECK_CVALUE(LLOG_ORIGIN_HANDLE_READ_HEADER);
        CHECK_CVALUE(LLOG_ORIGIN_HANDLE_WRITE_REC);
        CHECK_CVALUE(LLOG_ORIGIN_HANDLE_CLOSE);
        CHECK_CVALUE(LLOG_ORIGIN_CONNECT);
        CHECK_CVALUE(LLOG_CATINFO);
        CHECK_CVALUE(LLOG_ORIGIN_HANDLE_PREV_BLOCK);
        CHECK_CVALUE(LLOG_ORIGIN_HANDLE_DESTROY);
}

static void
check_llogd_conn_body(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llogd_conn_body);
        CHECK_MEMBER(llogd_conn_body, lgdc_gen);
        CHECK_MEMBER(llogd_conn_body, lgdc_logid);
        CHECK_MEMBER(llogd_conn_body, lgdc_ctxt_idx);
}

static void
check_mds_extent_desc(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mds_extent_desc);
        CHECK_MEMBER(mds_extent_desc, med_start);
        CHECK_MEMBER(mds_extent_desc, med_len);
        CHECK_MEMBER(mds_extent_desc, med_lmm);
}

static void
check_llog_array_rec(void)
{
        BLANK_LINE();
        CHECK_STRUCT(llog_array_rec);
        CHECK_MEMBER(llog_array_rec, lmr_hdr);
        CHECK_MEMBER(llog_array_rec, lmr_med);
        CHECK_MEMBER(llog_array_rec, lmr_tail);
}

static void
check_qunit_data(void)
{
        BLANK_LINE();
        CHECK_STRUCT(qunit_data);
        CHECK_MEMBER(qunit_data, qd_id);
        CHECK_MEMBER(qunit_data, qd_flags);
        CHECK_MEMBER(qunit_data, qd_count);
}

static void
check_qunit_data_old(void)
{
        BLANK_LINE();
        CHECK_STRUCT(qunit_data_old);
        CHECK_MEMBER(qunit_data_old, qd_id);
        CHECK_MEMBER(qunit_data_old, qd_type);
        CHECK_MEMBER(qunit_data_old, qd_count);
        CHECK_MEMBER(qunit_data_old, qd_isblk);
}

static void
check_mgs_target_info(void)
{
        BLANK_LINE();
        CHECK_STRUCT(mgs_target_info);
        CHECK_MEMBER(mgs_target_info, mti_lustre_ver);
        CHECK_MEMBER(mgs_target_info, mti_stripe_index);
        CHECK_MEMBER(mgs_target_info, mti_config_ver);
        CHECK_MEMBER(mgs_target_info, mti_flags);
        CHECK_MEMBER(mgs_target_info, mti_nid_count);
        CHECK_MEMBER(mgs_target_info, mti_fsname);
        CHECK_MEMBER(mgs_target_info, mti_svname);
        CHECK_MEMBER(mgs_target_info, mti_uuid);
        CHECK_MEMBER(mgs_target_info, mti_nids);
        CHECK_MEMBER(mgs_target_info, mti_params);
}

static void
check_lustre_disk_data(void)
{
        BLANK_LINE();
        CHECK_STRUCT(lustre_disk_data);
        CHECK_MEMBER(lustre_disk_data, ldd_magic);
        CHECK_MEMBER(lustre_disk_data, ldd_feature_compat);
        CHECK_MEMBER(lustre_disk_data, ldd_feature_rocompat);
        CHECK_MEMBER(lustre_disk_data, ldd_feature_incompat);
        CHECK_MEMBER(lustre_disk_data, ldd_config_ver);
        CHECK_MEMBER(lustre_disk_data, ldd_flags);
        CHECK_MEMBER(lustre_disk_data, ldd_svindex);
        CHECK_MEMBER(lustre_disk_data, ldd_mount_type);
        CHECK_MEMBER(lustre_disk_data, ldd_fsname);
        CHECK_MEMBER(lustre_disk_data, ldd_svname);
        CHECK_MEMBER(lustre_disk_data, ldd_uuid);
        CHECK_MEMBER(lustre_disk_data, ldd_userdata);
        CHECK_MEMBER(lustre_disk_data, ldd_mount_opts);
        CHECK_MEMBER(lustre_disk_data, ldd_params);
}

static void
system_string (char *cmdline, char *str, int len)
{
        int   fds[2];
        int   rc;
        pid_t pid;

        rc = pipe(fds);
        if (rc != 0)
                abort();

        pid = fork();
        if (pid == 0) {
                /* child */
                int   fd = fileno(stdout);

                rc = dup2(fds[1], fd);
                if (rc != fd)
                        abort();

                exit(system(cmdline));
                /* notreached */
        } else if ((int)pid < 0) {
                abort();
        } else {
                FILE *f = fdopen(fds[0], "r");

                if (f == NULL)
                        abort();

                close(fds[1]);

                if (fgets(str, len, f) == NULL)
                        abort();

                if (waitpid(pid, &rc, 0) != pid)
                        abort();

                if (!WIFEXITED(rc) || WEXITSTATUS(rc) != 0)
                        abort();

                if (strnlen(str, len) == len)
                        str[len - 1] = 0;

                if (str[strlen(str) - 1] == '\n')
                        str[strlen(str) - 1] = 0;

                fclose(f);
        }
}

int
main(int argc, char **argv)
{
        char unameinfo[80];
        char gccinfo[80];

        system_string("uname -a", unameinfo, sizeof(unameinfo));
        system_string(CC " -v 2>&1 | tail -1", gccinfo, sizeof(gccinfo));

        printf ("void lustre_assert_wire_constants(void)\n"
                "{\n"
                "        /* Wire protocol assertions generated by 'wirecheck'\n"
                "         * (make -C lustre/utils newwirecheck)\n"
                "         * running on %s\n"
                "         * with %s */\n"
                "\n", unameinfo, gccinfo);

        BLANK_LINE ();

        COMMENT("Constants...");
        CHECK_DEFINE(LUSTRE_MSG_MAGIC_V1);
        CHECK_DEFINE(LUSTRE_MSG_MAGIC_V2);
        CHECK_DEFINE(PTLRPC_MSG_VERSION);

        CHECK_VALUE(PTL_RPC_MSG_REQUEST);
        CHECK_VALUE(PTL_RPC_MSG_ERR);
        CHECK_VALUE(PTL_RPC_MSG_REPLY);

        CHECK_VALUE(MSG_LAST_REPLAY);
        CHECK_VALUE(MSG_RESENT);
        CHECK_VALUE(MSG_REPLAY);
        CHECK_VALUE(MSG_AT_SUPPORT);

        CHECK_VALUE(MSG_CONNECT_RECOVERING);
        CHECK_VALUE(MSG_CONNECT_RECONNECT);
        CHECK_VALUE(MSG_CONNECT_REPLAYABLE);

        CHECK_VALUE(OST_REPLY);
        CHECK_VALUE(OST_GETATTR);
        CHECK_VALUE(OST_SETATTR);
        CHECK_VALUE(OST_READ);
        CHECK_VALUE(OST_WRITE);
        CHECK_VALUE(OST_CREATE);
        CHECK_VALUE(OST_DESTROY);
        CHECK_VALUE(OST_GET_INFO);
        CHECK_VALUE(OST_CONNECT);
        CHECK_VALUE(OST_DISCONNECT);
        CHECK_VALUE(OST_PUNCH);
        CHECK_VALUE(OST_OPEN);
        CHECK_VALUE(OST_CLOSE);
        CHECK_VALUE(OST_STATFS);
        CHECK_VALUE(OST_SYNC);
        CHECK_VALUE(OST_QUOTACHECK);
        CHECK_VALUE(OST_QUOTACTL);
        CHECK_VALUE(OST_LAST_OPC);

        CHECK_DEFINE(OBD_OBJECT_EOF);

        CHECK_VALUE(MDS_GETATTR);
        CHECK_VALUE(MDS_GETATTR_NAME);
        CHECK_VALUE(MDS_CLOSE);
        CHECK_VALUE(MDS_REINT);
        CHECK_VALUE(MDS_READPAGE);
        CHECK_VALUE(MDS_CONNECT);
        CHECK_VALUE(MDS_DISCONNECT);
        CHECK_VALUE(MDS_GETSTATUS);
        CHECK_VALUE(MDS_STATFS);
        CHECK_VALUE(MDS_PIN);
        CHECK_VALUE(MDS_UNPIN);
        CHECK_VALUE(MDS_SYNC);
        CHECK_VALUE(MDS_DONE_WRITING);
        CHECK_VALUE(MDS_SET_INFO);
        CHECK_VALUE(MDS_QUOTACHECK);
        CHECK_VALUE(MDS_QUOTACTL);
        CHECK_VALUE(MDS_LAST_OPC);

        CHECK_VALUE(REINT_SETATTR);
        CHECK_VALUE(REINT_CREATE);
        CHECK_VALUE(REINT_LINK);
        CHECK_VALUE(REINT_UNLINK);
        CHECK_VALUE(REINT_RENAME);
        CHECK_VALUE(REINT_OPEN);
        CHECK_VALUE(REINT_MAX);

        CHECK_VALUE(MGS_CONNECT);
        CHECK_VALUE(MGS_DISCONNECT);
        CHECK_VALUE(MGS_EXCEPTION);
        CHECK_VALUE(MGS_TARGET_REG);
        CHECK_VALUE(MGS_TARGET_DEL);
        CHECK_VALUE(MGS_SET_INFO);

        CHECK_VALUE(DISP_IT_EXECD);
        CHECK_VALUE(DISP_LOOKUP_EXECD);
        CHECK_VALUE(DISP_LOOKUP_NEG);
        CHECK_VALUE(DISP_LOOKUP_POS);
        CHECK_VALUE(DISP_OPEN_CREATE);
        CHECK_VALUE(DISP_OPEN_OPEN);

        CHECK_VALUE(MDS_STATUS_CONN);
        CHECK_VALUE(MDS_STATUS_LOV);

        CHECK_VALUE(LDLM_ENQUEUE);
        CHECK_VALUE(LDLM_CONVERT);
        CHECK_VALUE(LDLM_CANCEL);
        CHECK_VALUE(LDLM_BL_CALLBACK);
        CHECK_VALUE(LDLM_CP_CALLBACK);
        CHECK_VALUE(LDLM_GL_CALLBACK);
        CHECK_VALUE(LDLM_LAST_OPC);

        CHECK_VALUE(LCK_EX);
        CHECK_VALUE(LCK_PW);
        CHECK_VALUE(LCK_PR);
        CHECK_VALUE(LCK_CW);
        CHECK_VALUE(LCK_CR);
        CHECK_VALUE(LCK_NL);
        CHECK_VALUE(LCK_GROUP);
        CHECK_VALUE(LCK_MAXMODE);

        CHECK_CVALUE(LDLM_PLAIN);
        CHECK_CVALUE(LDLM_EXTENT);
        CHECK_CVALUE(LDLM_FLOCK);
        CHECK_CVALUE(LDLM_IBITS);

        CHECK_VALUE(OBD_PING);
        CHECK_VALUE(OBD_LOG_CANCEL);
        CHECK_VALUE(OBD_QC_CALLBACK);
        CHECK_VALUE(OBD_LAST_OPC);

        CHECK_VALUE(QUOTA_DQACQ);
        CHECK_VALUE(QUOTA_DQREL);

        CHECK_VALUE(MGS_CONNECT);
        CHECK_VALUE(MGS_DISCONNECT);
        CHECK_VALUE(MGS_EXCEPTION);   
        CHECK_VALUE(MGS_TARGET_REG);
        CHECK_VALUE(MGS_TARGET_DEL);
        CHECK_VALUE(MGS_SET_INFO);

        COMMENT("Sizes and Offsets");
        BLANK_LINE();
        CHECK_STRUCT(obd_uuid);
        check_lustre_handle();
        check_lustre_msg_v1();
        check_lustre_msg_v2();
        printf("        LASSERT(offsetof(struct lustre_msg_v1, lm_magic) == "
               "offsetof(struct lustre_msg_v2, lm_magic));\n");
        check_ptlrpc_body();
        check_obd_connect_data();
        check_obdo();
        check_lov_mds_md_v1();
        check_lov_mds_md_join();
        check_obd_statfs();
        check_obd_ioobj();
        check_obd_quotactl();
        check_niobuf_remote();
        check_ost_body();
        check_ll_fid();
        check_mds_status_req();
        check_mds_body();
        check_mds_rec_setattr();
        check_mds_rec_create();
        check_mds_rec_link();
        check_mds_rec_unlink();
        check_mds_rec_rename();
        check_mds_rec_join();
        check_lov_desc();
        check_ldlm_res_id();
        check_ldlm_extent();
        check_ldlm_flock();
        check_ldlm_inodebits();
        check_ldlm_intent();
        check_ldlm_resource_desc();
        check_ldlm_lock_desc();
        check_ldlm_request();
        check_ldlm_reply();
        check_ldlm_lvb();
        check_llog_logid();
        check_llog_catid();
        check_llog_rec_hdr();
        check_llog_rec_tail();
        check_llog_logid_rec();
        check_llog_create_rec();
        check_llog_orphan_rec();
        check_llog_unlink_rec();
        check_llog_setattr_rec();
        check_llog_size_change_rec();
        check_llog_gen();
        check_llog_gen_rec();
        check_llog_log_hdr();
        check_llog_cookie();
        check_llogd_body();
        check_llogd_conn_body();
        check_llog_array_rec();
        check_mds_extent_desc();
        check_qunit_data();
        check_qunit_data_old();
        check_mgs_target_info();
        check_lustre_disk_data();


        printf("}\n\n");

        return(0);
}
