/* -*- mode: c; c-basic-offset: 8; indent-tabs-mode: nil; -*-
 * vim:expandtab:shiftwidth=8:tabstop=8:
 *
 *  lustre/quota/quota_ctl.c
 *
 *  Copyright (c) 2005 Cluster File Systems, Inc.
 *
 *   This file is part of Lustre, http://www.lustre.org.
 *
 *   No redistribution or use is permitted outside of Cluster File Systems, Inc.
 *
 */
#ifndef EXPORT_SYMTAB
# define EXPORT_SYMTAB
#endif
#define DEBUG_SUBSYSTEM S_MDS

#ifdef __KERNEL__
# include <linux/version.h>
# include <linux/module.h>
# include <linux/init.h>
# include <linux/fs.h>
# include <linux/jbd.h>
# include <linux/ext3_fs.h>
# include <linux/quota.h>
# if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
#  include <linux/smp_lock.h>
#  include <linux/buffer_head.h>
#  include <linux/workqueue.h>
#  include <linux/mount.h>
# else
#  include <linux/locks.h>
# endif
#else /* __KERNEL__ */
# include <liblustre.h>
#endif

#include <obd_class.h>
#include <lustre_mds.h>
#include <lustre_dlm.h>
#include <lustre_cfg.h>
#include <obd_ost.h>
#include <lustre_fsfilt.h>
#include <lustre_quota.h>
#include "quota_internal.h"

#ifdef __KERNEL__
int mds_quota_ctl(struct obd_export *exp, struct obd_quotactl *oqctl)
{
        struct obd_device *obd = exp->exp_obd;
        int rc = 0;
        ENTRY;

        switch (oqctl->qc_cmd) {
        case Q_QUOTAON:
                rc = mds_quota_on(obd, oqctl);
                break;
        case Q_QUOTAOFF:
                mds_quota_off(obd, oqctl);
                break;
        case Q_SETINFO:
                rc = mds_set_dqinfo(obd, oqctl);
                break;
        case Q_GETINFO:
                rc = mds_get_dqinfo(obd, oqctl);
                break;
        case Q_SETQUOTA:
                rc = mds_set_dqblk(obd, oqctl);
                break;
        case Q_GETQUOTA:
                rc = mds_get_dqblk(obd, oqctl);
                break;
        case Q_GETOINFO:
        case Q_GETOQUOTA:
                rc = mds_get_obd_quota(obd, oqctl);
                break;
        case LUSTRE_Q_INVALIDATE:
                rc = mds_quota_invalidate(obd, oqctl);
                break;
        default:
                CERROR("%s: unsupported mds_quotactl command: %d\n",
                       obd->obd_name, oqctl->qc_cmd);
                RETURN(-EFAULT);
        }

        if (rc)
                CDEBUG(D_INFO, "mds_quotactl admin quota command %d, id %u, "
                               "type %d, failed: rc = %d\n",
                       oqctl->qc_cmd, oqctl->qc_id, oqctl->qc_type, rc);

        RETURN(rc);
}

int filter_quota_ctl(struct obd_export *exp, struct obd_quotactl *oqctl)
{
        struct obd_device *obd = exp->exp_obd;
        struct obd_device_target *obt = &obd->u.obt;
        struct lvfs_run_ctxt saved;
        int rc = 0;
        ENTRY;

        switch (oqctl->qc_cmd) {
        case Q_QUOTAON:
        case Q_QUOTAOFF:
                if (!atomic_dec_and_test(&obt->obt_quotachecking)) {
                        CDEBUG(D_INFO, "other people are doing quotacheck\n");
                        atomic_inc(&obt->obt_quotachecking);
                        rc = -EBUSY;
                        break;
                }
        case Q_GETOINFO:
        case Q_GETOQUOTA:
        case Q_GETQUOTA:
                /* In recovery scenario, this pending dqacq/dqrel might have
                 * been processed by master successfully before it's dquot
                 * on master enter recovery mode. We must wait for this 
                 * dqacq/dqrel done then return the correct limits to master */
                if (oqctl->qc_stat == QUOTA_RECOVERING)
                        qctxt_wait_pending_dqacq(&obd->u.obt.obt_qctxt,
                                                 oqctl->qc_id, oqctl->qc_type, 
                                                 1);

                push_ctxt(&saved, &obd->obd_lvfs_ctxt, NULL);
                rc = fsfilt_quotactl(obd, obd->u.obt.obt_sb, oqctl);
                pop_ctxt(&saved, &obd->obd_lvfs_ctxt, NULL);

                if (oqctl->qc_cmd == Q_QUOTAON || oqctl->qc_cmd == Q_QUOTAOFF) {
                        if (!rc)
                                obt->obt_qctxt.lqc_status = 
                                        (oqctl->qc_cmd == Q_QUOTAON) ? 1 : 0;
                        atomic_inc(&obt->obt_quotachecking);
                }
                break;
        case Q_SETQUOTA:
                /* currently, it is only used for nullifying the quota */
                qctxt_wait_pending_dqacq(&obd->u.obt.obt_qctxt,
                                         oqctl->qc_id, oqctl->qc_type, 1);

                push_ctxt(&saved, &obd->obd_lvfs_ctxt, NULL);
                rc = fsfilt_quotactl(obd, obd->u.obt.obt_sb, oqctl);

                if (!rc) {
                        oqctl->qc_cmd = Q_SYNC;
                        fsfilt_quotactl(obd, obd->u.obt.obt_sb, oqctl);
                        oqctl->qc_cmd = Q_SETQUOTA;
                }
                pop_ctxt(&saved, &obd->obd_lvfs_ctxt, NULL);
                break;
        case Q_INITQUOTA:
                {
                unsigned int uid = 0, gid = 0;

                /* Initialize quota limit to MIN_QLIMIT */
                LASSERT(oqctl->qc_dqblk.dqb_valid == QIF_BLIMITS);
                LASSERT(oqctl->qc_dqblk.dqb_bsoftlimit == 0);

                /* There might be a pending dqacq/dqrel (which is going to
                 * clear stale limits on slave). we should wait for it's
                 * completion then initialize limits */
                qctxt_wait_pending_dqacq(&obd->u.obt.obt_qctxt,
                                         oqctl->qc_id, oqctl->qc_type, 1);

                if (!oqctl->qc_dqblk.dqb_bhardlimit)
                        goto adjust;

                LASSERT(oqctl->qc_dqblk.dqb_bhardlimit == MIN_QLIMIT);
                push_ctxt(&saved, &obd->obd_lvfs_ctxt, NULL);
                rc = fsfilt_quotactl(obd, obd->u.obt.obt_sb, oqctl);

                /* Update on-disk quota, in case of lose the changed limits
                 * (MIN_QLIMIT) on crash, which cannot be recovered.*/
                if (!rc) {
                        oqctl->qc_cmd = Q_SYNC;
                        fsfilt_quotactl(obd, obd->u.obt.obt_sb, oqctl);
                        oqctl->qc_cmd = Q_INITQUOTA;
                }
                pop_ctxt(&saved, &obd->obd_lvfs_ctxt, NULL);

                if (rc)
                        RETURN(rc);
adjust:
                /* Trigger qunit pre-acquire */
                if (oqctl->qc_type == USRQUOTA)
                        uid = oqctl->qc_id;
                else
                        gid = oqctl->qc_id;

                rc = qctxt_adjust_qunit(obd, &obd->u.obt.obt_qctxt, 
                                        uid, gid, 1, 0);
                if (rc == -EDQUOT || rc == -EBUSY) {
                        CDEBUG(D_QUOTA, "rc: %d.\n", rc);
                        rc = 0;
                }

                break;
                }
        default:
                CERROR("%s: unsupported filter_quotactl command: %d\n",
                       obd->obd_name, oqctl->qc_cmd);
                RETURN(-EFAULT);
        }

        RETURN(rc);
}
#endif /* __KERNEL__ */

int client_quota_ctl(struct obd_export *exp, struct obd_quotactl *oqctl)
{
        struct ptlrpc_request *req;
        struct obd_quotactl *oqc;
        int size[2] = { sizeof(struct ptlrpc_body), sizeof(*oqctl) };
        int ver, opc, rc;
        ENTRY;

        if (!strcmp(exp->exp_obd->obd_type->typ_name, LUSTRE_MDC_NAME)) {
                ver = LUSTRE_MDS_VERSION,
                opc = MDS_QUOTACTL;
        } else if (!strcmp(exp->exp_obd->obd_type->typ_name, LUSTRE_OSC_NAME)) {
                ver = LUSTRE_OST_VERSION,
                opc = OST_QUOTACTL;
        } else {
                RETURN(-EINVAL);
        }

        req = ptlrpc_prep_req(class_exp2cliimp(exp), ver, opc, 2, size, NULL);
        if (!req)
                GOTO(out, rc = -ENOMEM);

        oqc = lustre_msg_buf(req->rq_reqmsg, REQ_REC_OFF, sizeof(*oqctl));
        *oqc = *oqctl;

        ptlrpc_req_set_repsize(req, 2, size);

        rc = ptlrpc_queue_wait(req);
        oqc = lustre_swab_repbuf(req, REPLY_REC_OFF, sizeof(*oqc),
                                 lustre_swab_obd_quotactl);
        if (oqc == NULL) {
                CERROR ("Can't unpack obd_quotactl\n");
                GOTO(out, rc = -EPROTO);
        }

        *oqctl = *oqc;
        EXIT;
out:
        ptlrpc_req_finished(req);
        return rc;
}

int lov_quota_ctl(struct obd_export *exp, struct obd_quotactl *oqctl)
{
        struct obd_device *obd = class_exp2obd(exp);
        struct lov_obd *lov = &obd->u.lov;
        __u64 curspace = 0;
        __u64 bhardlimit = 0;
        int i, rc = 0;
        ENTRY;

        if (oqctl->qc_cmd != Q_QUOTAON && oqctl->qc_cmd != Q_QUOTAOFF &&
            oqctl->qc_cmd != Q_GETOQUOTA && oqctl->qc_cmd != Q_INITQUOTA &&
            oqctl->qc_cmd != Q_SETQUOTA) {
                CERROR("bad quota opc %x for lov obd", oqctl->qc_cmd);
                RETURN(-EFAULT);
        }

        for (i = 0; i < lov->desc.ld_tgt_count; i++) {
                int err;

                if (!lov->lov_tgts[i] || !lov->lov_tgts[i]->ltd_active) {
                        if (oqctl->qc_cmd == Q_GETOQUOTA) {
                                CERROR("ost %d is inactive\n", i);
                                rc = -EIO;
                        } else {
                                CDEBUG(D_HA, "ost %d is inactive\n", i);
                        }
                        continue;
                }

                err = obd_quotactl(lov->lov_tgts[i]->ltd_exp, oqctl);
                if (err) {
                        if (lov->lov_tgts[i]->ltd_active && !rc)
                                rc = err;
                        continue;
                }

                if (oqctl->qc_cmd == Q_GETOQUOTA) {
                        curspace += oqctl->qc_dqblk.dqb_curspace;
                        bhardlimit += oqctl->qc_dqblk.dqb_bhardlimit;
                }
        }

        if (oqctl->qc_cmd == Q_GETOQUOTA) {
                oqctl->qc_dqblk.dqb_curspace = curspace;
                oqctl->qc_dqblk.dqb_bhardlimit = bhardlimit;
        }
        RETURN(rc);
}

