/*
 *  client/edit.c  --  Routines for editing texts
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */

#include <config.h>
#if HAVE_STDLIB_H
#include <stdlib.h>
#ifdef __vax__
#define __SIZE_T is included from stdlib.h
#define __WCHAR_T is included from stdlib.h
#endif
#endif
#include <stddef.h>
#include <assert.h>

#include <kom-types.h>
#include <zmalloc.h>
#include <s-string.h>

#include "read-line.h"
#include "error.h"

#include "edit.h"

#include "defaults.h"

#define Export


/*  List of the texts you are currently editing  */
Export  Edited_text	* edited_texts	= NULL;

/*  Size of the list of edited texts (number of allocated cells)  */
Export  int	  size_of_text_list	= 0;

/*  Current active text.  The one editing commands affect.  */
Export  int	  active_text		= -1;

/*  Flag. Set by the parsing of command line args in this simple */
/*  version. */
Export	int	  tomrad_avslutar	= 0;



#define	newline()	putchar('\n')



/****************************************************************
*	Functions to manipulate the list of edited texts	*
****************************************************************/


/*
 *  Shorthand.  Meny functions modify just this element in the
 *  array (really not an array for C) 'edited_texts', and it gets
 *  rather tedious typing this all the time, and since there is no
 *  WITH statement, as in Modula-[23], this is the solution.
 */
#define	EDN	(edited_texts[edit_no])


Export  int
edit_create_new_text (void)
{
    static const int		  text_list_chunk	= 16;
    int				  index;
    Edited_text			* temp_list;
    static const Edited_text	  NEW_EDITED_TEXT = EMPTY_EDITED_TEXT_i;

    /* Find an unused slot in list */
    index = 0;
    while (index < size_of_text_list && edited_texts[index].active )
    {
	index++;
    }

    /* If none found, increase size of list */
    if (index >= size_of_text_list)
    {
	temp_list = zrealloc (edited_texts,
			      sizeof(Edited_text) * (size_of_text_list
						      + text_list_chunk));
	/* If we can't increase size, signal an error */
	if (temp_list == NULL)
	{
	    return -1;
	}
	else
	{
	    edited_texts = temp_list;
	    /* Mark all new texts as inactive */
	    for ( index = 0 ;  index < text_list_chunk ;  index++ )
		edited_texts[size_of_text_list + index].active = FALSE;
	    size_of_text_list += text_list_chunk;
	    index -= text_list_chunk;
	}
    }

    /* Initialize slot */
    edited_texts[index] = NEW_EDITED_TEXT;
    return  index;
}



/*
 *  Deallocate a text and mark it's slot in the list of edited
 *  texts as free.  Returns the text number to be the new "active
 *  text".
 */
Export  int
edit_dealloc_text (int	edit_no)

{
    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    s_clear (&EDN.text);
    if (EDN.miscs != NULL)
	zfree (EDN.miscs);
    EDN.active = FALSE;
    return  -1;
}


/********************************************************
*	Functions to manipulate separate texts		*
********************************************************/


/*
 *  Check if the recipient is already a recipient of that text.
 */
static Bool
edit_already_recipient(int 	edit_no,
		       Conf_no	recipient)
{
    int r;

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    for (r = 0; r < EDN.no_of_miscs; r++)
    {
	if (EDN.miscs [r].type == recpt
	    && EDN.miscs [r].datum.recipient == recipient)
	{
	    return TRUE;
	}

	if (EDN.miscs [r].type == cc_recpt
	    && EDN.miscs [r].datum.recipient == recipient)
	{
	    return TRUE;
	}
    }

    return FALSE;
}


/*
 *  Add the conference RECIPIENT as a recipient of the unfinished
 *  text with the working number EDIT_NO.
 *
 *  Returns FAILURE if and only if the recipient is already a recipient.
 */
Export  Success
edit_add_recipient (int		edit_no,
		    Conf_no	recipient)

{
    Misc_info	* temp_list;		/* Temporary result from realloc() */

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    /* Check that it is not already a recipient. */
    if (edit_already_recipient(edit_no, recipient))
	return FAILURE;

    /* Increase the size of the 'misc' list */
    temp_list = zrealloc (EDN.miscs,
			  (EDN.no_of_miscs+1) * sizeof (Misc_info));
    if (temp_list == NULL)	/* Oops, we couldn't */
	return  FAILURE;	/* This should be fatal, shouldn't it. */
    EDN.miscs = temp_list;
    EDN.no_of_miscs++;

    EDN.miscs [EDN.no_of_miscs-1].type = recpt;
    EDN.miscs [EDN.no_of_miscs-1].datum.recipient = recipient;

    return  OK;
}



/*
 *  Add the conference RECIPIENT as a carbon copy recipient of the
 *  unfinished text with the working number EDIT_NO.
 *
 *  Returns FAILURE if and only if cc_recipient is already a recipient.
 */
Export  Success
edit_add_cc_recipient (int	edit_no,
		       Conf_no	cc_recipient)

{
    Misc_info	* temp_list;		/* Temporary result from realloc() */

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    /* Check that it is not already a recipient. */
    if (edit_already_recipient(edit_no, cc_recipient))
	return FAILURE;

    /* Increase the size of the 'misc' list */
    temp_list = zrealloc (EDN.miscs,
			  (EDN.no_of_miscs+1) * sizeof (Misc_info));
    if (temp_list == NULL)	/* Oops, we couldn't */
	return  FAILURE;
    EDN.miscs = temp_list;
    EDN.no_of_miscs++;

    EDN.miscs [EDN.no_of_miscs-1].type = cc_recpt;
    EDN.miscs [EDN.no_of_miscs-1].datum.recipient = cc_recipient;

    return  OK;
}



/*
 *  Add the text TEXT-NO as being commented by the text with the 
 *  working number EDIT_NO.
 */
Export  Success
edit_add_commenting (int	edit_no,
		     Text_no	text_no)
{
    Misc_info	* temp_list;		/* Temporary result from realloc() */

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    /* Increase the size of the 'misc' list */
    temp_list = zrealloc (EDN.miscs,
			  (EDN.no_of_miscs+1) * sizeof (Misc_info));
    if (temp_list == NULL)	/* Oops, we couldn't */
	return  FAILURE;
    EDN.miscs = temp_list;
    EDN.no_of_miscs++;

    EDN.miscs [EDN.no_of_miscs-1].type = comm_to;
    EDN.miscs [EDN.no_of_miscs-1].datum.comment_to = text_no;

    return  OK;
}



/*
 *  Add the text TEXT-NO as being footnoted by the text with the 
 *  working number EDIT_NO.
 */
Export  Success
edit_add_footnote (int	edit_no,
	           Text_no text_no)
{
    Misc_info	* temp_list;		/* Temporary result from realloc() */

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    /* Increase the size of the 'misc' list */
    temp_list = zrealloc (EDN.miscs,
			  (EDN.no_of_miscs+1) * sizeof (Misc_info));
    if (temp_list == NULL)	/* Oops, we couldn't */
	return  FAILURE;
    EDN.miscs = temp_list;
    EDN.no_of_miscs++;

    EDN.miscs [EDN.no_of_miscs-1].type = footn_to;
    EDN.miscs [EDN.no_of_miscs-1].datum.footnote_to = text_no;

    return  OK;
}



/*
 *  Remove conference RECIPIENT from the list of recipients to
 *  the unfinished text EDIT_NO.
 */
extern  Success
edit_sub_recipient (int		edit_no,
		    Conf_no	recipient);



/*
 *  Let the user enter text
 */
Export  Edit_result
edit_enter_text (int	edit_no)
{
#define FUNCTION "edit_enter_text()"

    String		  line		= EMPTY_STRING;
    int			  retval;	/* Status from input_line() */

#define CLEAN_UP() 	do { s_clear(&line); } while (0)

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    while (TRUE)	/* Until user is tired of typing */
    {
	retval = input_line (stdin, stdout, &line,
			     -64, "\033\n", NULL, EOF); /* More magic */
	if (retval == 0)
	    fatal1 (CLIENT_OUT_OF_MEMORY, "input_line() failed");

	if (retval == '\033')		/* Remove trailing escape */
	    line.len--;

	if (edit_add_string (edit_no, line) == FAILURE)
	    fatal1 (CLIENT_OUT_OF_MEMORY, "edit_add_string() failed");

	switch (retval)
	{
	case  EOF:
	    /* User wants to enter this text as is */
	    CLEAN_UP();
	    return  enter_this_text;
	    /*NOTREACHED*/
	    break;

	case  '\033':
	    /* User wants to do some commands */
	    CLEAN_UP();
	    return  to_be_continued;
	    /*NOTREACHED*/
	    break;

	case  '\n':
	    /* Om tomrad_avslutar {r satt s} skall vi kolla om i {r i */
	    /* slutet av inl{gget och det {r en tomrad f|re och i s} */
	    /* fall skall vi l{gga in texten. */
	    if (tomrad_avslutar
		&& EDN.cursor_pos.pos == s_strlen (EDN.text) /* i slutet */
		&& s_strcmp(s_fsubstr(EDN.text, 
				      s_strlen(EDN.text) - 2,
				      END_OF_STRING),
			    s_fcrea_str("\n\n")) == 0) {
		return enter_this_text;
	    }
	    newline();
	    break;

	default:
	    /* input_line() shouldn't return anything else */
	    fatal2 (CLIENT_SHOULDNT_HAPPEN, 
		    "input_line() returned strange: %d", retval);
	    break;
	}
    }   /* While TRUE (until user is tired of typing) */
#undef CLEAN_UP
#undef FUNCTION
}



/*
 *  Add a String to a text.  This does not free the string added to the text.
 */
Export  Success
edit_add_string (int	  edit_no,
		 String	  str	  )

{
    String		  temp		= EMPTY_STRING_i;
#define FUNCTION "edit_add_string()"

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    if (s_strcpy (&temp, s_fsubstr (EDN.text, 0,
				    EDN.cursor_pos.pos - 1)) != OK)
	fatal1 (CLIENT_OUT_OF_MEMORY, "s_strcat() failed");
    if (s_strcat (&temp, str) != OK)
	fatal1 (CLIENT_OUT_OF_MEMORY, "s_strcat() failed");
    if (s_strcat (&temp, s_fsubstr (EDN.text, EDN.cursor_pos.pos,
				    s_strlen (EDN.text) - 1)) != OK)
	fatal1 (CLIENT_OUT_OF_MEMORY, "s_strcat() failed");
    s_clear (&EDN.text);
    EDN.text = temp;
    EDN.cursor_pos.pos += s_strlen (str);

    return  OK;
#undef FUNCTION
}



/*
 *  Delete a number of characters from a text.  Deletes forward if
 *  positive number of characters, backward if negative.
 */
Export  Success
edit_del_string (int	  edit_no,
		 int	  no_of_chars)

{
    Success		retval;

    assert(edit_no >= 0);
    assert(edit_no < size_of_text_list);
    assert(EDN.active);

    if (no_of_chars >= 0)
    {
	retval = s_strdel (&EDN.text,
			   EDN.cursor_pos.pos,
			   EDN.cursor_pos.pos + no_of_chars - 1);
    }
    else
    {
	retval = s_strdel (&EDN.text,
			   EDN.cursor_pos.pos + no_of_chars,
			   EDN.cursor_pos.pos - 1);
	EDN.cursor_pos.pos -= abs (no_of_chars);
    }

    return  retval;
}
