/*  Lziprecover - Data recovery tool for lzip files
    Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _FILE_OFFSET_BITS 64

#include <cerrno>
#include <climits>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <stdint.h>
#include <unistd.h>
#include <sys/stat.h>

#include "lzip.h"
#include "file_index.h"
#include "mtester.h"


int seek_write( const int fd, const uint8_t * const buf, const int size,
                const long long pos )
  {
  if( lseek( fd, pos, SEEK_SET ) == pos )
    return writeblock( fd, buf, size );
  return 0;
  }


int repair_file( const std::string & input_filename,
                 const std::string & output_filename, const int verbosity,
                 const bool force )
  {
  struct stat in_stats;
  const int infd = open_instream( input_filename.c_str(), &in_stats, true, true );
  if( infd < 0 ) return 1;

  Pretty_print pp( input_filename, verbosity );
  const File_index file_index( infd );
  if( file_index.retval() != 0 )
    { pp( file_index.error().c_str() ); return file_index.retval(); }

  int outfd = -1;
  for( long i = 0; i < file_index.members(); ++i )
    {
    const long long mpos = file_index.mblock( i ).pos();
    const long long msize = file_index.mblock( i ).size();
    if( !safe_seek( infd, mpos ) )
      cleanup_and_fail( output_filename, outfd, 1 );
    long long failure_pos = 0;
    if( try_decompress_member( infd, msize, &failure_pos ) ) continue;
    if( failure_pos >= msize - 8 ) failure_pos = msize - 8 - 1;
    if( failure_pos < File_header::size )
      { show_error( "Can't repair error in input file." );
        cleanup_and_fail( output_filename, outfd, 2 ); }

    if( verbosity >= 1 )		// damaged member found
      {
      std::printf( "Repairing member %ld  (failure pos = %llu)\n",
                   i + 1, mpos + failure_pos );
      std::fflush( stdout );
      }
    uint8_t * const mbuffer = read_member( infd, mpos, msize );
    if( !mbuffer )
      cleanup_and_fail( output_filename, outfd, 1 );
    long pos = failure_pos;
    bool done = false;
    while( pos >= File_header::size && pos > failure_pos - 20000 && !done )
      {
      const long min_pos = std::max( (long)File_header::size, pos - 1000 );
      const LZ_mtester * master = prepare_master( mbuffer, msize, min_pos - 16 );
      if( !master )
        cleanup_and_fail( output_filename, outfd, 1 );
      for( ; pos >= min_pos && !done ; --pos )
        {
        if( verbosity >= 1 )
          {
          std::printf( "Trying position %llu \r", mpos + pos );
          std::fflush( stdout );
          }
        for( int j = 0; j < 256; ++j )
          {
          ++mbuffer[pos];
          if( j == 255 ) break;
          if( test_member_rest( *master ) )
            {
            done = true;
            if( outfd < 0 )		// first damaged member repaired
              {
              if( !safe_seek( infd, 0 ) ) return 1;
              outfd = open_outstream_rw( output_filename, force );
              if( outfd < 0 ) { close( infd ); return 1; }
              if( !copy_file( infd, outfd ) )		// copy whole file
                cleanup_and_fail( output_filename, outfd, 1 );
              }
            if( seek_write( outfd, mbuffer + pos, 1, mpos + pos ) != 1 )
              { show_error( "Error writing output file", errno );
                cleanup_and_fail( output_filename, outfd, 1 ); }
            break;
            }
          }
        }
      delete master;
      }
    delete[] mbuffer;
    if( verbosity >= 1 ) std::printf( "\n" );
    if( !done )
      {
      show_error( "Can't repair input file. Error is probably larger than 1 byte." );
      cleanup_and_fail( output_filename, outfd, 2 );
      }
    }

  if( outfd < 0 )
    {
    if( verbosity >= 1 )
      std::printf( "Input file has no errors. Recovery is not needed.\n" );
    return 0;
    }
  if( close( outfd ) != 0 )
    {
    show_error( "Error closing output file", errno );
    cleanup_and_fail( output_filename, -1, 1 );
    }
  if( verbosity >= 1 )
    std::printf( "Copy of input file repaired successfully.\n" );
  return 0;
  }
