/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Ant" and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.taskdefs;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.Commandline;
import org.apache.tools.ant.util.JavaEnvUtils;

import java.io.File;

import junit.framework.TestCase;

/**
 * Simple testcase for the ExecuteJava class - mostly stolen from
 * ExecuteWatchdogTest.
 *
 * @author <a href="mailto:sbailliez@imediation.com">Stephane Bailliez</a>
 * @author <a href="mailto:stefan.bodewig@epost.de">Stefan Bodewig</a>
 */
public class ExecuteJavaTest extends TestCase {

    private final static int TIME_OUT = 5000;

    private final static int CLOCK_ERROR=200;
    private final static int TIME_OUT_TEST=TIME_OUT-CLOCK_ERROR;
    
    private ExecuteJava ej;
    private Project project;

    public ExecuteJavaTest(String name) {
        super(name);
    }

    protected void setUp(){
        ej = new ExecuteJava();
        ej.setTimeout(new Long(TIME_OUT));
        project = new Project();
        project.setBasedir(".");
        ej.setClasspath(new Path(project, getTestClassPath()));
    }

    private Commandline getCommandline(int timetorun) throws Exception {
        Commandline cmd = new Commandline();
        cmd.setExecutable(TimeProcess.class.getName());
        cmd.createArgument().setValue(String.valueOf(timetorun));
        return cmd;
    }

    public void testNoTimeOut() throws Exception {
        Commandline cmd = getCommandline(TIME_OUT/2);
        ej.setJavaCommand(cmd);
        ej.execute(project);
        assertTrue("process should not have been killed", !ej.killedProcess());
    }

    // test that the watchdog ends the process
    public void testTimeOut() throws Exception {
        Commandline cmd = getCommandline(TIME_OUT*2);
        ej.setJavaCommand(cmd);
        long now = System.currentTimeMillis();
        ej.execute(project);
        long elapsed = System.currentTimeMillis() - now;
        assertTrue("process should have been killed", ej.killedProcess());

        assertTrue("elapse time of "+elapsed
                   +" ms is less than timeout value of "+TIME_OUT_TEST+" ms", 
                   elapsed >= TIME_OUT_TEST);
        assertTrue("elapse time of "+elapsed
                   +" ms is greater than run value of "+(TIME_OUT*2)+" ms", 
                   elapsed < TIME_OUT*2);
    }


    /**
     * Dangerous method to obtain the classpath for the test. This is
     * severely tighted to the build.xml properties.
     */
    private static String getTestClassPath(){
        String classpath = System.getProperty("build.tests");
        if (classpath == null) {
            System.err.println("WARNING: 'build.tests' property is not available !");
            classpath = System.getProperty("java.class.path");
        }

        // JDK 1.1 needs classes.zip in -classpath argument
        if (JavaEnvUtils.isJavaVersion(JavaEnvUtils.JAVA_1_1)) {
            classpath +=   File.pathSeparator
                + System.getProperty("java.home")
                + File.separator + "lib"
                + File.separator + "classes.zip";
        }

        return classpath;
    }

}
