/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon public domain software
 * originally written at the National Center for Supercomputing Applications,
 * University of Illinois, Urbana-Champaign.
 */

/*
 * mod_iptos: set IP type-of-service bits according to directory
 *
 * v1.0
 * 
 * author: dean gaudet <dean@arctic.org>
 *
 * 0.1: initial implementation
 * 0.2: added keyword "none"
 * 1.0: what the heck, it's official
 */

/*
	documentation:

	type-of-service bits in IP packets can be used for such things
	as quality-of-service guarantees.  this module allows you to
	set IP TOS bits on a per-directory basis (it'll also work for
	location and files containers of course).

	the four TOS bits are:

		lowdelay
		throughput
		reliability
		lowcost

	note that in ancient IP history you were permitted only to
	set none or one of those bits.  at some point someone gave
	meaning to the bitwise-OR of the bits... and i'm too lazy
	to go look up those meanings right now.  but since there are
	meanings to the bitwise-OR, this module accepts the '|'
	character as an OR operator to combine these bits.

	the magic keyword "none" will result in no IPTOS bits being set.
	(useful for a nested directory container to disable a TOS
	you've set on a parent directory.)

	there's only one command:  IPTOS bits

	as in:

	# default to lowdelay TOS
	IPTOS lowdelay

	# choose throughput TOS for the video directory
	<Directory /my/big/video/directory>
		IPTOS throughput
	</Directory>
*/

#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_request.h"

#include <netinet/ip.h>

typedef struct {
	unsigned tos;
} iptos_config;

module MODULE_VAR_EXPORT iptos_module;


static const char *iptos_parse_iptos(char *word, unsigned *ptos)
{
	unsigned tos;
	char *bar;

	tos = 0;
	if (!*word) {
usage:
		return "unrecognized TOS.  use none, lowdelay, throughput, reliability, or lowcost, possibly bitwise-or'd together by using '|'.";
	}
	for (;;) {
		bar = strchr(word, '|');
		if (!bar) {
			bar = word + strlen(word);
		}
		if (bar - word == 8 && strncasecmp(word, "lowdelay", 8) == 0) {
			tos |= IPTOS_LOWDELAY;
		}
		else if (bar - word == 10 && strncasecmp(word, "throughput", 10) == 0) {
			tos |= IPTOS_THROUGHPUT;
		}
		else if (bar - word == 11 && strncasecmp(word, "reliability", 11) == 0) {
			tos |= IPTOS_RELIABILITY;
		}
		else if (bar - word == 7 && strncasecmp(word, "lowcost", 7) == 0) {
			tos |= IPTOS_LOWCOST;
		}
		else if (bar - word == 4 && strncasecmp(word, "none", 4) == 0) {
			tos = 0;
		}
		else {
			goto usage;
		}
		if (*bar) {
			word = bar + 1;
		}
		else {
			break;
		}
	}
	*ptos = tos;
	return NULL;
}


static const char *iptos_cmd_iptos(cmd_parms *cmd, void *mconfig, char *word)
{
	iptos_config *config = mconfig;

	return iptos_parse_iptos(word, &config->tos);
}


static void *iptos_create_dir_config(pool *p, char *dirspec)
{
	iptos_config *config;

	config = ap_pcalloc(p, sizeof(iptos_config));
	config->tos = 0;
	return config;
}


static int iptos_fixup(request_rec *r)
{
	iptos_config *config = ap_get_module_config(r->per_dir_config, &iptos_module);
	unsigned tos;

	/* don't want to do this for subrequests */
	if (!ap_is_initial_req(r)) {
		return DECLINED;
	}
	tos = config->tos;	/* not sure if copy is necessary, but make one to be safe */
	if (setsockopt(r->connection->client->fd, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(tos))) {
		ap_log_rerror(APLOG_MARK, APLOG_INFO, r, "setsockopt(IP_TOS, %u)", tos);
	}
	return DECLINED;
}


static const command_rec iptos_cmds[] =
{
	{ "IPTOS", iptos_cmd_iptos, NULL, OR_OPTIONS, TAKE1,
	    "arg is none, lowdelay, throughput, reliability, or lowcost -- and can be bitwise-or'd together using '|'"
	},
	{NULL}
};


module MODULE_VAR_EXPORT iptos_module =
{
    STANDARD_MODULE_STUFF,
    NULL,			/* module initializer */
    iptos_create_dir_config,	/* per-directory config creator */
    NULL,			/* dir config merger */
    NULL,			/* server config creator */
    NULL,			/* server config merger */
    iptos_cmds,			/* command table */
    NULL,			/* [9] list of handlers */
    NULL,			/* [2] filename-to-URI translation */
    NULL,			/* [5] check/validate user_id */
    NULL,			/* [6] check user_id is valid *here* */
    NULL,			/* [4] check access by host address */
    NULL,			/* [7] MIME type checker/setter */
    iptos_fixup,		/* [8] fixups */
    NULL			/* [10] logger */
#if MODULE_MAGIC_NUMBER >= 19970103
    ,NULL			/* [3] header parser */
#endif
#if MODULE_MAGIC_NUMBER >= 19970719
    ,NULL			/* process initializer */
#endif
#if MODULE_MAGIC_NUMBER >= 19970728
    ,NULL			/* process exit/cleanup */
#endif
#if MODULE_MAGIC_NUMBER >= 19970902
    ,NULL			/* [1] post read_request handling */
#endif
};
