/*
 *  output-tarball.c
 *  mod_musicindex
 *
 *  $Id: output-tarball.c 812 2008-12-21 01:34:29Z varenet $
 *
 *  Created by Thibaut VARENE on Mon Jun 26 2006.
 *  Copyright (c) 2006-2008 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Tarballs generation
 *
 * @author Thibaut Varene
 * @version $Revision: 812 $
 * @date 2006-2008
 *
 * This file handles on the fly tarball generation and associated operations
 *
 * @todo use apr_file I/O
 */

/* build flag: -D_FILE_OFFSET_BITS=64 - libarchive goes AWOL without it */
#ifndef _FILE_OFFSET_BITS
#define _FILE_OFFSET_BITS 64
#endif

#include "mod_musicindex.h"

#include <http_core.h>
#include <archive.h>	/* libarchive */
#include <archive_entry.h>
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>	/* stat */
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>	/* stat */
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* stat */
#endif
#include <stdio.h>	/* fopen... */

#define RWBUFSIZE	8192	/**< Size of our r/w buffer */
#define QSKIP_CHECKS(q)	(!(q->flags & EF_ALLOWTARBALL))
#define ARCHIVE_SET_FORMAT(a)	\
	archive_write_set_compression_none(a); \
	archive_write_set_format_ustar(a)

/** Holds our client data for libarchive */
struct cdata {
	request_rec *r;	/**< Apache request_rec */
};

static int
wrapopen(struct archive *a, void *client_data)
{
	return (ARCHIVE_OK);
}

static ssize_t
wrapwrite(struct archive *a, void *client_data, void *buff, size_t n)
{
	struct cdata *mydata = client_data;

	return (ap_rwrite(buff, n, mydata->r));
}

/**
 * Sends tarball.
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param pack A pack of music entries.
 *
 * It is a known "feature" that when tar'ing a custom playlist, the generated
 * tarball will contain the full path as published on the webserver, including
 * '/'. We could get rid of the leading '/' if needed, but there's one good
 * reason not to put all the files at the root of one big tarball: in case there
 * are name doublons.
 */
void send_tarball(request_rec *r, const mu_pack *const pack)
{
	const mu_ent *q = pack->fhead;
	struct cdata *mydata;
	struct archive *a;
	struct archive_entry *entry;
	struct stat st;
	static char buff[RWBUFSIZE];
	int len;
	FILE *file;

	if (!q)
		return;

	if (!(mydata = malloc(sizeof(struct cdata))))
		return;

	if (!(a = archive_write_new()))
		goto fail1;
	
	if (!(entry = archive_entry_new()))
		goto fail2;

	mydata->r = r;
	ARCHIVE_SET_FORMAT(a);
	archive_write_open(a, mydata, wrapopen, wrapwrite, NULL);

	for (; q; q = q->next) {
		if (unlikely(QSKIP_CHECKS(q)))
			continue;

		stat(q->filename, &st);
		archive_entry_clear(entry);
		archive_entry_copy_stat(entry, &st);
		archive_entry_set_pathname(entry, q->file);
		archive_write_header(a, entry);

		file = fopen(q->filename, "r");
		len = fread(buff, 1, sizeof(buff), file);

		while (len > 0) {
			archive_write_data(a, buff, len);
			len = fread(buff, 1, sizeof(buff), file);
		}
		fclose(file);
	}

	archive_entry_free(entry);
fail2:
	archive_write_finish(a);
fail1:
	free(mydata);
}

static ssize_t
wrapwritesize(struct archive *a, void *client_data, void *buff, size_t n)
{
	ssize_t *mysize = client_data;
	*mysize += n;
	return n;
}

/**
 * Compute tarball size
 *
 * @param r Apache request_rec struct to handle connection details.
 * @param pack A pack of music entries.
 *
 * This function must be kept coherent with send_tarball()
 */
ssize_t
tarball_size(request_rec *r, const mu_pack *const pack)
{
	const mu_ent *q = pack->fhead;
	ssize_t tbsize = -1;
	struct archive *a;
	struct archive_entry *ae;
	struct stat st;
	static char buff[RWBUFSIZE];
	int len;
	off_t fsize;

	if (!q)
		goto fail1;

	if (!(a = archive_write_new()))
		goto fail1;

	if (!(ae = archive_entry_new()))
		goto fail2;
	
	ARCHIVE_SET_FORMAT(a);
	archive_write_open(a, &tbsize, NULL, wrapwritesize, NULL);

	for (; q; q = q->next) {
		if (QSKIP_CHECKS(q))
			continue;

		stat(q->filename, &st);
		archive_entry_clear(ae);
		archive_entry_copy_stat(ae, &st);
		archive_entry_set_pathname(ae, q->file);
		archive_write_header(a, ae);

		fsize = q->size;
		len = (likely(fsize > sizeof(buff)) ? sizeof(buff) : fsize);
		while (len > 0) {
			fsize -= archive_write_data(a, buff, len);
			len = (likely(fsize > sizeof(buff)) ? sizeof(buff) : fsize);
		}
	}
	archive_entry_free(ae);
fail2:
	archive_write_finish(a);
fail1:
	return tbsize;
}
