/***************************************************************************
 $RCSfile: medium.h,v $
                             -------------------
    cvs         : $Id: medium.h,v 1.17 2005/03/08 00:55:52 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifndef AH_MEDIUM_H
#define AH_MEDIUM_H

#include <gwenhywfar/misc.h>
#include <gwenhywfar/inherit.h>
#include <gwenhywfar/libloader.h>
#include <gwenhywfar/buffer.h>

#include <aqbanking/banking.h>
#include <aqhbci/objectref.h> /* for AQHBCI_API */

/**
 * If this flag is set then signing with an RSA key will not choose the
 * smaller signature according iso9796-appendix. The default is to allow it.
 */
#define AH_MEDIUM_FLAGS_DISABLE_SMALLER_SIGNATURE 0x00000001


#ifdef __cplusplus
extern "C" {
#endif
typedef struct AH_MEDIUM AH_MEDIUM;

typedef enum {
  AH_Medium_UIActionInsertMedium=0,
  AH_Medium_UIActionInsertCorrectMedium,
  AH_Medium_UIActionStartInputPinViaKeypad,
  AH_Medium_UIActionFinishedInputPinViaKeypad
} AH_MEDIUM_UIACTION;


typedef enum {
  AH_MediumTypeDDV=0,
  AH_MediumTypeRDH,
  AH_MediumTypePINTAN
} AH_MEDIUMTYPE;


typedef enum {
  AH_MediumDeviceFile=0,
  AH_MediumDeviceChipcard,
  AH_MediumDevicePseudo,
  AH_MediumDeviceAny=999
} AH_MEDIUMDEVICE;



typedef enum {
  AH_MediumResultOk=0,
  AH_MediumResultNoKey,
  AH_MediumResultBadKey,
  AH_MediumResultSignSeq,
  AH_MediumResultInvalidSignature,
  AH_MediumResultGenericError,
  AH_MediumResultNotSupported
} AH_MEDIUM_RESULT;


typedef enum {
  /** Medium is supported by this plugin */
  AH_MediumCheckResultOk=0,
  /** Function is not supported by this medium */
  AH_MediumCheckResultNotSupported,
  /** Medium might be supported, but the name is wrong */
  AH_MediumCheckResultWrongName,
  /** Medium is not supported by this plugin */
  AH_MediumCheckResultWrongMedium,
  /** Generic error (leads to aborting the check) */
  AH_MediumCheckResultGenericError,
} AH_MEDIUM_CHECKRESULT;


GWEN_LIST_FUNCTION_LIB_DEFS(AH_MEDIUM, AH_Medium, AQHBCI_API);
GWEN_INHERIT_FUNCTION_LIB_DEFS(AH_MEDIUM, AQHBCI_API);
#ifdef __cplusplus
}
#endif


#include <gwenhywfar/buffer.h>
#include <gwenhywfar/crypt.h>
#include <aqhbci/hbci.h>

#ifdef __cplusplus
extern "C" {
#endif


/** @defgroup AQHBCI_MOD_HBCI_MEDIUM HBCI Medium
 * @ingroup AQHBCI_MOD_HBCI
 * @short Security Medium For HBCI
 * @author Martin Preuss<martin@libchipcard.de>
 *
 * <p>
 * There are generally two types of media:
 * <ul>
 *   <li>private media</li>
 *   <li>public media</li>
 * </ul>
 * <b>Private media</b> are those used to decrypt received messages and to
 * sign outbound messages. If the security mode RDH is used then such a medium
 * will use private keys. <br>
 * Private media MUST NOT implement the encrypt and verify functions.<br>
 * <b>Public media</b> are used to encrypt outbound messages and to verify
 * signatures on incoming messages. If the security mode RDH is used then
 * such a medium will use public keys.<br>
 * Public media MUST NOT implement the decrypt and sign functions.
 * </p>
 * <p>
 * Before any medium function can be used the medium must be <i>mounted</i>.
 * This is accomplished by calling @ref AH_Medium_Mount. Upon this call
 * the implementation will make all necessary information and keys available.
 * </p>
 */
/*@{*/


/** @name Prototypes For Virtual Functions
 *
 */
/*@{*/
typedef int (*AH_MEDIUM_MOUNTFN)(AH_MEDIUM *m);
typedef int (*AH_MEDIUM_CREATEFN)(AH_MEDIUM *m);
typedef int (*AH_MEDIUM_UNMOUNTFN)(AH_MEDIUM *m, int force);
typedef int (*AH_MEDIUM_CHANGEPINFN)(AH_MEDIUM *m);

typedef int (*AH_MEDIUM_TODBFN)(const AH_MEDIUM *m,
                                     GWEN_DB_NODE *db);
typedef int (*AH_MEDIUM_FROMDBFN)(AH_MEDIUM *m,
                                       GWEN_DB_NODE *db);

typedef AH_MEDIUM_RESULT (*AH_MEDIUM_SIGNFN)(AH_MEDIUM *m,
                                             GWEN_BUFFER *msgbuf,
                                             GWEN_BUFFER *signbuf);
typedef int (*AH_MEDIUM_GETNEXTSIGNSEQFN)(AH_MEDIUM *m);

typedef int (*AH_MEDIUM_SETLOCALSIGNSEQFN)(AH_MEDIUM *m,
                                           int i);

typedef AH_MEDIUM_RESULT (*AH_MEDIUM_DECRYPTKEYFN)(AH_MEDIUM *m,
                                                   GWEN_BUFFER *srcbuf,
                                                   GWEN_BUFFER *decryptbuf);
typedef AH_MEDIUM_RESULT (*AH_MEDIUM_ENCRYPTKEYFN)(AH_MEDIUM *m,
                                                   GWEN_BUFFER *srcbuf,
                                                   GWEN_BUFFER *encryptbuf);
typedef GWEN_BUFFER* (*AH_MEDIUM_GENERATEMSGKEYFN)(AH_MEDIUM *m);


typedef AH_MEDIUM_RESULT (*AH_MEDIUM_VERIFYFN)(AH_MEDIUM *m,
                                               GWEN_BUFFER *msgbuf,
                                               GWEN_BUFFER *signbuf,
                                               int signseq);

typedef int (*AH_MEDIUM_SELECTCONTEXTFN)(AH_MEDIUM *m, int idx);

typedef int (*AH_MEDIUM_CREATECONTEXTFN)(AH_MEDIUM *m,
					 int country,
					 const char *bankId,
					 const char *userId);

typedef int (*AH_MEDIUM_REMOVECONTEXTFN)(AH_MEDIUM *m, int idx);

typedef int (*AH_MEDIUM_READCONTEXTFN)(AH_MEDIUM *m,
				       int idx,
				       int *country,
				       GWEN_BUFFER *bankId,
				       GWEN_BUFFER *userId,
				       GWEN_BUFFER *server,
				       int *port);

typedef int (*AH_MEDIUM_WRITECONTEXTFN)(AH_MEDIUM *m,
					int idx,
					int country,
					const char *bankId,
					const char *userId,
					const char *server,
					int port);

/*@}*/




/** @name Constructor And Destructor
 *
 */
/*@{*/
/**
 * @param mtn medium type name (like "RDHFile", "DDVCard" etc)
 * @param bankId bank code (German "Bankleitzahl")
 * @param userId id of this medium's owner
 */
AQHBCI_API
AH_MEDIUM *AH_Medium_new(AH_HBCI *hbci,
                         AH_MEDIUMTYPE mt,
                         const char *mtn,
                         const char *mediumName);

/**
 * Destroys the medium if the internal usage counter reaches zero.
 * That counter is incremented by @ref AH_Medium_new and
 * @ref AH_Medium_Attach and decremented by this function.
 */
AQHBCI_API
void AH_Medium_free(AH_MEDIUM *m);

/**
 * Attaches to the medium. This means that the usage count of this medium
 * gets incremented so a following free will not really destroy it until
 * the usage counter reaches zero.
 * So after finishing using the medium attached to you should simply call
 * @ref AH_Medium_free and that function will take care of the usage
 * counter stuff.
 * Please note that @ref AH_Medium_new also increments the usage counter
 * so you do not have to call this function after creating the medium.
 */
AQHBCI_API
void AH_Medium_Attach(AH_MEDIUM *m);

/*@}*/


/** @name Informational Functions
 *
 */
/*@{*/

/**
 * This flag only tells whether the medium name or device type have changed
 */
AQHBCI_API
int AH_Medium_IsDirty(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetIsDirty(AH_MEDIUM *m, int i);

AQHBCI_API
AH_HBCI *AH_Medium_GetHBCI(const AH_MEDIUM *m);
AQHBCI_API
AB_BANKING *AH_Medium_GetBankingApi(const AH_MEDIUM *m);

AQHBCI_API
AH_MEDIUMTYPE AH_Medium_GetMediumType(const AH_MEDIUM *m);
AQHBCI_API
const char *AH_Medium_GetMediumTypeName(const AH_MEDIUM *m);
AQHBCI_API
const char *AH_Medium_GetMediumName(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetMediumName(AH_MEDIUM *m, const char *s);

/**
 * This string is presented to the user to allow him to identify the
 * medium an interactive function refers to (such as GetPin).
 * This string should be stored by the medium upon unmount and restored
 * when mounting the medium.
 */
AQHBCI_API
const char *AH_Medium_GetDescriptiveName(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetDescriptiveName(AH_MEDIUM *m, const char *s);


AQHBCI_API
AH_MEDIUMDEVICE AH_Medium_GetDeviceType(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetDeviceType(AH_MEDIUM *m, AH_MEDIUMDEVICE md);

/**
 * For RDH media this id is assigned by the server (you will need to
 * perform a special job in order to retrieve the system id).
 * For DDV media this id is taken from the DDV card.
 * The implementation of the mount function MUST update this field if
 * possible (e.g. on DDV media this is taken from the card and stored here).
 * This field is automatically stored upon @ref AH_Medium_ToDB and
 * restored upon @ref AH_Medium_FromDB.
 */
AQHBCI_API
const char *AH_Medium_GetSecurityIdPtr(const AH_MEDIUM *m);
AQHBCI_API
unsigned int AH_Medium_GetSecurityIdSize(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetSecurityId(AH_MEDIUM *m,
                             const void *p, unsigned int l);

/**
 * The peer's address. This field is only used by clients to get the
 * address to connect to.
 * This field should be updated upon mount.
 */
AQHBCI_API
const char *AH_Medium_GetPeerAddr(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetPeerAddr(AH_MEDIUM *m,
                           const char *s);

/**
 * The peer's port.
 * This field should be updated upon mount.
 */
AQHBCI_API
int AH_Medium_GetPeerPort(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetPeerPort(AH_MEDIUM *m, int i);

AQHBCI_API
GWEN_TYPE_UINT32 AH_Medium_GetFlags(const AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetFlags(AH_MEDIUM *m, GWEN_TYPE_UINT32 fl);
AQHBCI_API
void AH_Medium_AddFlags(AH_MEDIUM *m, GWEN_TYPE_UINT32 fl);
AQHBCI_API
void AH_Medium_SubFlags(AH_MEDIUM *m, GWEN_TYPE_UINT32 fl);

AQHBCI_API
int AH_Medium_IsMounted(AH_MEDIUM *m);

AQHBCI_API
const char *AH_Medium_TypeToName(AH_MEDIUMTYPE t);

/*@}*/


/** @name Context Selection, Creation and Removal
 *
 */
/*@{*/

AQHBCI_API
int AH_Medium_SelectContext(AH_MEDIUM *m, int idx);

AQHBCI_API
  int AH_Medium_FindContext(AH_MEDIUM *m,
			    int country,
			    const char *bankId,
			    const char *userId);

AQHBCI_API
int AH_Medium_CreateContext(AH_MEDIUM *m,
			    int country,
			    const char *bankId,
			    const char *userId);

AQHBCI_API
int AH_Medium_RemoveContext(AH_MEDIUM *m, int idx);

AQHBCI_API
int AH_Medium_ReadContext(AH_MEDIUM *m,
			  int idx,
			  int *country,
			  GWEN_BUFFER *bankId,
			  GWEN_BUFFER *userId,
			  GWEN_BUFFER *server,
			  int *port);

AQHBCI_API
int AH_Medium_WriteContext(AH_MEDIUM *m,
			   int idx,
			   int country,
			   const char *bankId,
			   const char *userId,
			   const char *server,
			   int port);


/*@}*/



/** @name Encryption And Decryption
 *
 */
/*@{*/
/**
 */
AQHBCI_API
AH_MEDIUM_RESULT AH_Medium_Encrypt(AH_MEDIUM *m,
                                   GWEN_BUFFER *msgbuf,
                                   GWEN_BUFFER *encryptbuf,
                                   GWEN_BUFFER *msgKeyBuffer);

/**
 */
AQHBCI_API
AH_MEDIUM_RESULT AH_Medium_Decrypt(AH_MEDIUM *m,
                                   GWEN_BUFFER *msgbuf,
                                   GWEN_BUFFER *decryptbuf,
                                   GWEN_BUFFER *msgKeyBuffer);
/*@}*/



/** @name Keyspecs
 *
 * The KeySpecs are only available if the medium has been <i>mounted</i>
 * at least once in its lifetime.
 */
/*@{*/
AQHBCI_API
const GWEN_KEYSPEC*
  AH_Medium_GetLocalSignKeySpec(AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetLocalSignKeySpec(AH_MEDIUM *m,
                                   const GWEN_KEYSPEC *ks);

AQHBCI_API
const GWEN_KEYSPEC*
  AH_Medium_GetLocalCryptKeySpec(AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetLocalCryptKeySpec(AH_MEDIUM *m,
				    const GWEN_KEYSPEC *ks);

AQHBCI_API
const GWEN_KEYSPEC*
  AH_Medium_GetRemoteSignKeySpec(AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetRemoteSignKeySpec(AH_MEDIUM *m,
				    const GWEN_KEYSPEC *ks);

AQHBCI_API
const GWEN_KEYSPEC*
  AH_Medium_GetRemoteCryptKeySpec(AH_MEDIUM *m);
AQHBCI_API
void AH_Medium_SetRemoteCryptKeySpec(AH_MEDIUM *m,
				     const GWEN_KEYSPEC *ks);
/*@}*/


/** @name Virtual Functions
 *
 * The functions in this group are wrappers which in most cases directly
 * call the implementations of the functions (exceptions:
 * @ref AH_Medium_Mount and @ref AH_Medium_Unmount).
 */
/*@{*/
/**
 * <p>
 * The implementation of this function should make the KeySpecs available
 * (using @ref AH_Medium_SetRemoteCryptKeySpec and others as
 * appropriate), at least id the medium is mounted for the first time in it's
 * lifetime. These keyspecs are then preserved upon
 * @ref AH_Medium_ToDB and @ref AH_Medium_fromDB.<br>
 * </p>
 * <p>
 * It generally is a good idea to update the keyspecs upon mount.
 * </p>
 *
 * <p>
 * Please note that this call only get's through to the virtual function of
 * inheriting classes if the mountCount is 0. Otherwise the mountCount will
 * simply be incremented without calling the implementation.
 * </p>
 *
 * <p>
 *  The following functions should be called by the implementation of this
 *  function:
 *  <ul>
 *   <li>@ref AH_Medium_SetLocalSignKeySpec   </li>
 *   <li>@ref AH_Medium_SetLocalCryptKeySpec  </li>
 *   <li>@ref AH_Medium_SetRemoteSignKeySpec  </li>
 *   <li>@ref AH_Medium_SetRemoteCryptKeySpec </li>
 *   <li>@ref AH_Medium_SetSecurityId         </li>
 *  </ul>
 * </p>
 */
AQHBCI_API
int AH_Medium_Mount(AH_MEDIUM *m);

/**
 * Creates the medium. A KeyFile medium would prepare the file upon this.
 * Upon successful return this medium is mounted so you need to unmount it
 * in order to really write a keyfile.
 */
AQHBCI_API
int AH_Medium_Create(AH_MEDIUM *m);

/**
 *
 * Please note that this call only get's through to the virtual function of
 * inheriting classes if the mountCount is 1 or <i>force</i> is !=0.
 * Otherwise the mountCount will simply be decremented without calling the
 * implementation.
 */
AQHBCI_API
int AH_Medium_Unmount(AH_MEDIUM *m, int force);



/**
 * Makes the medium change the PIN.
 */
AQHBCI_API
int AH_Medium_ChangePin(AH_MEDIUM *m);


/**
 * Stores all interesting data about this medium to the given DB (but not the
 * keys!) in order to make the medium restoreable via
 * @ref AH_Medium_FromDB.
 * This function stores the information it can access (like the keyspecs)
 * and then calls the implementation with a subgroup of the given DB.
 */
AQHBCI_API
int AH_Medium_ToDB(const AH_MEDIUM *m, GWEN_DB_NODE *db);

/**
 * Restores the medium from the given DB.
 * This function restores the information it can access (like the keyspecs)
 * and then calls the implementation with a subgroup of the given DB.
 */
AQHBCI_API
int AH_Medium_FromDB(AH_MEDIUM *m, GWEN_DB_NODE *db);

/**
 */
AQHBCI_API
AH_MEDIUM_RESULT AH_Medium_Sign(AH_MEDIUM *m,
                                GWEN_BUFFER *msgbuf,
                                GWEN_BUFFER *signbuf);
/**
 */
AQHBCI_API
int AH_Medium_SetLocalSignSeq(AH_MEDIUM *m, int i);


/**
 */
AQHBCI_API
int AH_Medium_GetNextSignSeq(AH_MEDIUM *m);


/**
 */
AQHBCI_API
AH_MEDIUM_RESULT AH_Medium_EncryptKey(AH_MEDIUM *m,
                                      GWEN_BUFFER *srckey,
                                      GWEN_BUFFER *encKey);

/**
 */
AQHBCI_API
AH_MEDIUM_RESULT AH_Medium_DecryptKey(AH_MEDIUM *m,
                                      GWEN_BUFFER *srckey,
                                      GWEN_BUFFER *decKey);

AQHBCI_API
GWEN_BUFFER *AH_Medium_GenerateMsgKey(AH_MEDIUM *m);


/**
 * @param signseq Signature sequence counter received. If 0 then this
 * will not be verified, therefore the caller MUST make sure that a "0"
 * is never accepted from a signature inside a message !
 */
AQHBCI_API
AH_MEDIUM_RESULT AH_Medium_Verify(AH_MEDIUM *m,
                                  GWEN_BUFFER *msgbuf,
                                  GWEN_BUFFER *signbuf,
                                  int signseq);
/*@}*/



/** @name Setters For Virtual Functions
 *
 */
/*@{*/
AQHBCI_API
void AH_Medium_SetChangePinFn(AH_MEDIUM *m,
			      AH_MEDIUM_CHANGEPINFN f);
AQHBCI_API
void AH_Medium_SetMountFn(AH_MEDIUM *m,
			  AH_MEDIUM_MOUNTFN f);
AQHBCI_API
void AH_Medium_SetCreateFn(AH_MEDIUM *m,
			   AH_MEDIUM_CREATEFN f);
AQHBCI_API
void AH_Medium_SetUnmountFn(AH_MEDIUM *m,
			    AH_MEDIUM_UNMOUNTFN f);
AQHBCI_API
void AH_Medium_SetFromDbFn(AH_MEDIUM *m,
			   AH_MEDIUM_FROMDBFN f);
AQHBCI_API
void AH_Medium_SetToDbFn(AH_MEDIUM *m,
			 AH_MEDIUM_TODBFN f);

AQHBCI_API
void AH_Medium_SetGetNextSignSeqFn(AH_MEDIUM *m,
				   AH_MEDIUM_GETNEXTSIGNSEQFN f);
AQHBCI_API
void AH_Medium_SetSetLocalSignSeqFn(AH_MEDIUM *m,
				    AH_MEDIUM_SETLOCALSIGNSEQFN f);

AQHBCI_API
void AH_Medium_SetEncryptKeyFn(AH_MEDIUM *m,
                               AH_MEDIUM_ENCRYPTKEYFN f);
AQHBCI_API
void AH_Medium_SetDecryptKeyFn(AH_MEDIUM *m,
                               AH_MEDIUM_DECRYPTKEYFN f);
AQHBCI_API
void AH_Medium_SetGenerateMsgKeyFn(AH_MEDIUM *m,
                                   AH_MEDIUM_GENERATEMSGKEYFN f);

AQHBCI_API
void AH_Medium_SetVerifyFn(AH_MEDIUM *m,
                           AH_MEDIUM_VERIFYFN f);
AQHBCI_API
void AH_Medium_SetSignFn(AH_MEDIUM *m,
                         AH_MEDIUM_SIGNFN f);

AQHBCI_API
void AH_Medium_SetSelectContextFn(AH_MEDIUM *m,
				  AH_MEDIUM_SELECTCONTEXTFN f);
AQHBCI_API
void AH_Medium_SetCreateContextFn(AH_MEDIUM *m,
				  AH_MEDIUM_CREATECONTEXTFN f);
AQHBCI_API
void AH_Medium_SetRemoveContextFn(AH_MEDIUM *m,
				  AH_MEDIUM_REMOVECONTEXTFN f);

AQHBCI_API
void AH_Medium_SetReadContextFn(AH_MEDIUM *m,
				AH_MEDIUM_READCONTEXTFN f);
AQHBCI_API
void AH_Medium_SetWriteContextFn(AH_MEDIUM *m,
				 AH_MEDIUM_WRITECONTEXTFN f);

/*@}*/

/** @name Helper Functions
 *
 */
/*@{*/

AQHBCI_API
int AH_Medium_InputPin(AH_MEDIUM *m,
                       char *buffer,
                       int minLen,
                       int maxLen,
                       int flags);

AQHBCI_API
int AH_Medium_SetPinStatus(AH_MEDIUM *m,
                           const char *pin,
                           AB_BANKING_PINSTATUS status);

AQHBCI_API
int AH_Medium_InputTan(AH_MEDIUM *m,
                       char *buffer,
                       int minLen,
                       int maxLen);

AQHBCI_API
int AH_Medium_SetTanStatus(AH_MEDIUM *m,
                           const char *tan,
                           AB_BANKING_TANSTATUS status);

AQHBCI_API
int AH_Medium_UI(AH_MEDIUM *m, AH_MEDIUM_UIACTION act);


/*@}*/



/*@}*/ /* defgroup */






/** @defgroup AQHBCI_MOD_HBCI_MEDIUMPROVIDER HBCI Medium Provider
 * @ingroup AQHBCI_MOD_HBCI_MEDIUM
 * @short Security Medium Provider For HBCI
 * @author Martin Preuss<martin@libchipcard.de>
 */
/*@{*/

typedef struct AH_MEDIUMPROVIDER AH_MEDIUMPROVIDER;


GWEN_LIST_FUNCTION_LIB_DEFS(AH_MEDIUMPROVIDER, AH_MediumProvider, AQHBCI_API);
GWEN_INHERIT_FUNCTION_LIB_DEFS(AH_MEDIUMPROVIDER, AQHBCI_API);


/** @name Prototypes For Virtual Functions
 *
 * The functions in this group are wrappers which in most cases directly
 * call the implementations of the functions.
 */
/*@{*/

typedef AH_MEDIUMPROVIDER* (*AH_MEDIUMPROVIDER_NEWFN)(AH_HBCI *hbci);

typedef AH_MEDIUM*
  (*AH_MEDIUMPROVIDER_FACTORYFN)(AH_MEDIUMPROVIDER *mp,
                                 AH_HBCI *hbci,
                                 const char *mediumName);

typedef AH_MEDIUM_CHECKRESULT
  (*AH_MEDIUMPROVIDER_CHECKFN)(AH_MEDIUMPROVIDER *mp,
                               AH_HBCI *hbci,
                               GWEN_BUFFER *mediumName);


/*@}*/


AQHBCI_API
AH_MEDIUMPROVIDER *AH_MediumProvider_new(AH_HBCI *hbci,
					 const char *typeName);
AQHBCI_API
void AH_MediumProvider_free(AH_MEDIUMPROVIDER *mp);

AQHBCI_API
GWEN_LIBLOADER*
  AH_MediumProvider_GetLibLoader(const AH_MEDIUMPROVIDER *mp);
AQHBCI_API
void AH_MediumProvider_SetLibLoader(AH_MEDIUMPROVIDER *mp,
				    GWEN_LIBLOADER *ll);

AQHBCI_API
AH_HBCI *AH_MediumProvider_GetHBCI(const AH_MEDIUMPROVIDER *mp);


/** @name Virtual Functions
 *
 * The functions in this group are wrappers which in most cases directly
 * call the implementations of the functions (exceptions:
 * @ref AH_Medium_Mount and @ref AH_Medium_Unmount).
 */
/*@{*/
AQHBCI_API
AH_MEDIUM *AH_MediumProvider_Factory(AH_MEDIUMPROVIDER *mp,
                                     AH_HBCI *hbci,
                                     const char *mediumName);

/**
 * <p>
 * This method checks whether the given name resolves to a medium
 * of this type. A file medium might check whether the file has the
 * correct format, a card plugin might check whether the inserted card
 * is of the correct type.
 * </p>
 *
 * <p>
 * If the returned error code indicates success, then the medium is
 * a valid medium with regard to this plugin.
 * </p>
 *
 * <p>
 * If the returned error code is AH_MediumCheckResultWrongMedium
 * then it is suggested that this medium is not supported by the plugin.
 * All other error codes indicate an error, but only
 * AH_MediumCheckResultWrongMedium will be treated as a definitive NACK.
 * </p>
 *
 * <p>
 * This means any other error code should not lead to the conclusion that
 * the medium is not supported.
 * In such a case it would be best for the application to retry
 * </p>
 *
 * <p>
 * If this medium plugin supports card media, then the following rules
 * additionally apply:
 * <ul>
 * <li>
 *  If the parameter "name" is empty, then on success this method may
 *  set the name of this medium (in most cases this is the serial
 *  number of the card)
 * </li>
 * <li>
 *  If the parameter "name" is NOT empty, then the plugin MUST check
 *  whether the name is correct (i.e. if the serial number matches the
 *  given card). If the name does not match this method MUST return
 *  AH_MediumCheckResultWrongName.
 * </li>
 * </ul>
 * </p>
 *
 * <p>
 * However, if the plugin does not support this method at all it MUST
 * return AH_MediumCheckResultNotSupported.
 * </p>
 */
AQHBCI_API
AH_MEDIUM_CHECKRESULT AH_MediumProvider_Check(AH_MEDIUMPROVIDER *mp,
                                              AH_HBCI *hbci,
                                              GWEN_BUFFER *mediumName);

AQHBCI_API
const char*
  AH_MediumProvider_GetMediumTypeName(const AH_MEDIUMPROVIDER *mp);
AQHBCI_API
AH_MEDIUMDEVICE AH_MediumProvider_GetDeviceType(const AH_MEDIUMPROVIDER *mp);
AQHBCI_API
const char*
  AH_MediumProvider_GetDescription(const AH_MEDIUMPROVIDER *mp);


AQHBCI_API
void AH_MediumProvider_SetDeviceType(AH_MEDIUMPROVIDER *mp,
                                     AH_MEDIUMDEVICE md);

AQHBCI_API
void AH_MediumProvider_SetDescription(AH_MEDIUMPROVIDER *mp,
                                      const char *s);


/** @name Setters For Virtual Functions
 *
 */
/*@{*/
AQHBCI_API
void AH_MediumProvider_SetFactoryFn(AH_MEDIUMPROVIDER *mp,
                                    AH_MEDIUMPROVIDER_FACTORYFN f);
AQHBCI_API
void AH_MediumProvider_SetCheckFn(AH_MEDIUMPROVIDER *mp,
                                  AH_MEDIUMPROVIDER_CHECKFN f);
/*@}*/ /* defgroup */


#ifdef __cplusplus
}
#endif

#endif /* AH_MEDIUM_H */


