//
//  libavg - Media Playback Engine. 
//  Copyright (C) 2003-2008 Ulrich von Zadow
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//  Current versions can be found at www.libavg.de
//

#include "FileHelper.h"
#include "ConfigMgr.h"
#include "Exception.h"

#ifndef _WIN32
#include <libgen.h>
#else
#include <direct.h>
#endif
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>

#include <map>
#include <cstring>
#include <iostream>
#include <fstream>

using namespace std;

namespace avg {

string getPath(const string& Filename)
{
    if (Filename.length() > 0 && Filename.at(Filename.length()-1) == '/') {
        return Filename;
    }
#ifdef _WIN32
    int pos = int(Filename.find_last_of("\\/"));
    string DirName;
    if (pos >= 0) {
        DirName = Filename.substr(0, pos+1);
    } else {
        DirName = Filename;
    }
#else
    char * pszBuffer = strdup(Filename.c_str());

    string DirName(dirname(pszBuffer));
    free(pszBuffer);
    DirName += "/";
#endif

    return DirName;
}

string getFilenamePart(const string& Filename)
{
    if (Filename.find_last_of("\\/") == 0) {
        return Filename;
    }
#ifdef _WIN32
    int pos = int(Filename.find_last_of("\\/"));
    string BaseName(Filename.substr(pos+1));
#else
    char * pszBuffer = strdup(Filename.c_str());

    string BaseName(basename(pszBuffer));
    free(pszBuffer);
#endif

    return BaseName;
}

std::string getCWD()
{

    char szBuf[1024];
#ifdef _WIN32
    char * pBuf = _getcwd(szBuf, 1024);
#else
    char * pBuf = getcwd(szBuf, 1024);
#endif
    return string(pBuf)+"/";
}

bool isAbsPath(const std::string& path)
{
#ifdef _WIN32
    return ((path.length() != 0) && path[1] == ':') || path[0] == '\\' || path[0] == '/';
#else
    return path[0] == '/';
#endif
    
}

bool fileExists(const string& sFilename)
{
    struct stat myStat;
    return stat(sFilename.c_str(), &myStat) != -1;
}

void readWholeFile(const string& sFilename, string& sContent)
{
    ifstream File(sFilename.c_str());
    if (!File) {
        throw Exception(AVG_ERR_FILEIO, "Opening "+sFilename+
                " for reading failed.");
    }
    vector<char> buffer(65536);
    sContent.resize(0);
    while (File) {
        File.read(&(*buffer.begin()), (streamsize)(buffer.size()));
        sContent.append(&(*buffer.begin()), (unsigned)File.gcount());
    }
    if (!File.eof() || File.bad()) {
        throw Exception(AVG_ERR_FILEIO, "Reading "+sFilename+
                " failed.");
    }
}

void writeWholeFile(const string& sFilename, const string& sContent)
{
    ofstream outFile(sFilename.c_str());
    if (!outFile) {
        throw Exception(AVG_ERR_FILEIO, "Opening "+sFilename+
                " for writing failed.");
    }
    outFile << sContent;
}


void copyFile(const string& sSourceFile, const string& sDestFile)
{
    string sData;
            
    readWholeFile(sSourceFile, sData);
    writeWholeFile(sDestFile, sData);
}

}
