/**
 * Copyright 2011-2015
 *
 * Modified from https://github.com/BoostGSoC13/boost.trie
 *
 * Distributed under the Boost Software License, Version 1.0.
 *
 * Boost Software License - Version 1.0 - August 17th, 2003
 *
 * Permission is hereby granted, free of charge, to any person or organization
 * obtaining a copy of the software and accompanying documentation covered by
 * this license (the "Software") to use, reproduce, display, distribute,
 * execute, and transmit the Software, and to prepare derivative works of the
 * Software, and to permit third-parties to whom the Software is furnished to
 * do so, all subject to the following:
 *
 * The copyright notices in the Software and this entire statement, including
 * the above license grant, this restriction and the following disclaimer,
 * must be included in all copies of the Software, in whole or in part, and
 * all derivative works of the Software, unless such copies or derivative
 * works are solely in the form of machine-executable object code generated by
 * a source language processor.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 * FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifndef LIBBITCOIN_TRIE_ITERATOR_IPP
#define LIBBITCOIN_TRIE_ITERATOR_IPP

namespace libbitcoin {

// trie_iterator implementation
template<typename Value, typename Reference, typename Pointer>
trie_iterator<Value, Reference, Pointer>::trie_iterator()
    : trie_node_(nullptr), value_node_(nullptr), cached_key_()
{
}

template<typename Value, typename Reference, typename Pointer>
trie_iterator<Value, Reference, Pointer>::trie_iterator(
    structure_node_type* node)
    : trie_node_(node), value_node_(node->value_head), cached_key_()
{
}

template<typename Value, typename Reference, typename Pointer>
trie_iterator<Value, Reference, Pointer>::trie_iterator(value_node_type* value)
    : trie_node_(value->anchor), value_node_(value), cached_key_()
{
}

template<typename Value, typename Reference, typename Pointer>
trie_iterator<Value, Reference, Pointer>::trie_iterator(
    structure_node_type* node, value_node_type* value)
    : trie_node_(node), value_node_(value), cached_key_()
{
}

template<typename Value, typename Reference, typename Pointer>
trie_iterator<Value, Reference, Pointer>::trie_iterator(const iterator& it)
    : trie_node_(it.trie_node_), value_node_(it.value_node_),
        cached_key_(it.cached_key_)
{
}

template<typename Value, typename Reference, typename Pointer>
typename trie_iterator<Value, Reference, Pointer>::reference trie_iterator<
    Value, Reference, Pointer>::operator*() const
{
    return value_node_->value;
}

template<typename Value, typename Reference, typename Pointer>
typename trie_iterator<Value, Reference, Pointer>::pointer trie_iterator<Value,
    Reference, Pointer>::operator->() const
{
    return &(operator*());
}

template<typename Value, typename Reference, typename Pointer>
bool trie_iterator<Value, Reference, Pointer>::operator==(
    const trie_iterator& other) const
{
    return (trie_node_ == other.trie_node_) && 
        (value_node_ == other.value_node_);
}

template<typename Value, typename Reference, typename Pointer>
bool trie_iterator<Value, Reference, Pointer>::operator!=(
    const trie_iterator& other) const
{
    return !(*this == other);
}

template<typename Value, typename Reference, typename Pointer>
typename trie_iterator<Value, Reference, Pointer>::iter_type&
trie_iterator<Value, Reference, Pointer>::operator++()
{
    increment();
    return *this;
}

template<typename Value, typename Reference, typename Pointer>
typename trie_iterator<Value, Reference, Pointer>::iter_type trie_iterator<
    Value, Reference, Pointer>::operator++(int)
{
    auto it = *this;
    increment();
    return it;
}

template<typename Value, typename Reference, typename Pointer>
typename trie_iterator<Value, Reference, Pointer>::iter_type&
trie_iterator<Value, Reference, Pointer>::operator--()
{
    decrement();
    return *this;
}

template<typename Value, typename Reference, typename Pointer>
typename trie_iterator<Value, Reference, Pointer>::iter_type trie_iterator<
    Value, Reference, Pointer>::operator--(int)
{
    auto it = *this;
    decrement();
    return it;
}

template<typename Value, typename Reference, typename Pointer>
void trie_iterator<Value, Reference, Pointer>::trie_node_increment()
{
    cached_key_.resize(0);
    if (trie_node_->parent != nullptr)
        trie_node_ = trie_node_->next;
}

template<typename Value, typename Reference, typename Pointer>
void trie_iterator<Value, Reference, Pointer>::trie_node_decrement()
{
    cached_key_.resize(0);
    if (trie_node_->previous->parent != nullptr)
        trie_node_ = trie_node_->previous;
}

template<typename Value, typename Reference, typename Pointer>
void trie_iterator<Value, Reference, Pointer>::increment()
{
    // if not at end of current trie node values, continue walk
    if ((value_node_ != nullptr) && (value_node_->next != nullptr))
        value_node_ = value_node_->next;
    else
    {
        // else move to next trie node
        trie_node_increment();
        value_node_ = trie_node_->value_head;
    }
}

template<typename Value, typename Reference, typename Pointer>
void trie_iterator<Value, Reference, Pointer>::decrement()
{
    if ((value_node_ != nullptr) && (value_node_->previous != nullptr))
        value_node_ = value_node_->previous;
    else
    {
        trie_node_decrement();
        value_node_ = trie_node_->value_tail;
    }
}

template<typename Value, typename Reference, typename Pointer>
binary_type trie_iterator<Value, Reference, Pointer>::get_key()
{
    if (cached_key_.size() == 0)
    {
        std::vector<structure_node_type*> nodes;

        auto current = trie_node_;
        while (current != nullptr)
        {
            nodes.push_back(current);
            current = current->parent;
        }

        for (auto node = nodes.rbegin(); node != nodes.rend(); node++)
            cached_key_.append((*node)->label);
    }

    return cached_key_;
}

} // namespace libbitcoin

#endif
