/* This file is part of libccc
 *
 * AUTHORS
 *     Sven Herzberg  <herzi@gnome-de.org>
 *
 * Copyright (C) 2006  Sven Herzberg <herzi@gnome-de.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "cc-pixbuf.h"

#include <gdk/gdkcairo.h>
#include <ccc/cc-utils.h>

CcItem*
cc_pixbuf_new(void) {
	return g_object_new(CC_TYPE_PIXBUF, NULL);
}

void
cc_pixbuf_set_pixbuf(CcPixbuf* self, GdkPixbuf* pixbuf) {
	g_return_if_fail(CC_IS_PIXBUF(self));
	g_return_if_fail(!pixbuf || GDK_IS_PIXBUF(pixbuf));

	if(pixbuf == self->pixbuf) {
		return;
	}

	// FIXME: redraw on pixbuf changes

	if(self->pixbuf) {
		g_object_unref(self->pixbuf);
		self->pixbuf = NULL;
	}

	if(pixbuf) {
		self->pixbuf = g_object_ref(pixbuf);
	}

	cc_item_update_bounds(CC_ITEM(self), NULL);
	g_object_notify(G_OBJECT(self), "pixbuf");
}

void
cc_pixbuf_set_position(CcPixbuf* self, gdouble x, gdouble y, gdouble w, gdouble h) {
	g_return_if_fail(CC_IS_PIXBUF(self));

	g_object_set(self,
		     //"position-x", x,
		     //"position-x", x,
		     "position-w", w,
		     "position-h", h,
		     NULL);
}

/* GType */
G_DEFINE_TYPE(CcPixbuf, cc_pixbuf, CC_TYPE_ITEM);

enum {
	PROP_0,
	PROP_PIXBUF,
	PROP_POS_W,
	PROP_POS_H
};

static void
cc_pixbuf_init(CcPixbuf* self) {
	cc_pixbuf_set_position(self, 0.0, 0.0, 0.0, 0.0);
}

static void
cp_dispose(GObject* object) {
	CcPixbuf* self;

	if(CC_ITEM_DISPOSED(object)) {
		return;
	}

	self = CC_PIXBUF(object);
	if(self->pixbuf) {
		g_object_unref(self->pixbuf);
		self->pixbuf = NULL;
	}

	G_OBJECT_CLASS(cc_pixbuf_parent_class)->dispose(object);
}

static void
cp_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
	CcPixbuf* self = CC_PIXBUF(object);

	switch(prop_id) {
	case PROP_PIXBUF:
		g_value_set_object(value, self->pixbuf);
		break;
	case PROP_POS_W:
		g_value_set_double(value, self->position.x2 - self->position.x1);
		break;
	case PROP_POS_H:
		g_value_set_double(value, self->position.y2 - self->position.y1);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
cp_set_width(CcPixbuf* self, gdouble width) {
	if(width == self->position.x2 - self->position.x1) {
		return;
	}

	self->position.x2 = self->position.x1 + width;

	cc_item_update_bounds(CC_ITEM(self), NULL);
	g_object_notify(G_OBJECT(self), "position-w");
}

static void
cp_set_height(CcPixbuf* self, gdouble height) {
	if(height == self->position.y2 - self->position.y1) {
		return;
	}

	self->position.y2 = self->position.y1 + height;

	cc_item_update_bounds(CC_ITEM(self), NULL);
	g_object_notify(G_OBJECT(self), "position-h");
}

static void
cp_set_property(GObject* object, guint prop_id, GValue const* value, GParamSpec* pspec) {
	CcPixbuf* self = CC_PIXBUF(object);

	switch(prop_id) {
	case PROP_PIXBUF:
		cc_pixbuf_set_pixbuf(self, g_value_get_object(value));
		break;
	case PROP_POS_W:
		cp_set_width(self, g_value_get_double(value));
		break;
	case PROP_POS_H:
		cp_set_height(self, g_value_get_double(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
cp_render(CcItem* item, CcView* view, cairo_t* cr) {
	CcPixbuf* self = CC_PIXBUF(item);

	if(self->pixbuf) {
		CcDRect coordinates = self->position;
		cc_view_world_to_window(view, &coordinates.x1, &coordinates.y1);
		cc_view_world_to_window(view, &coordinates.x2, &coordinates.y2);
		if(CC_ITEM_GRID_ALIGNED(item)) {
			gdouble w1 = 1.0,
				w2 = w1;
			cc_point_grid_align(&coordinates.x1, &coordinates.y1, &w1);
			cc_point_grid_align(&coordinates.x2, &coordinates.y2, &w2);
		}
		cairo_save(cr);
		cairo_rectangle(cr, coordinates.x1, coordinates.y1, coordinates.x2 - coordinates.x1, coordinates.y2 - coordinates.y1);
		cairo_translate(cr, coordinates.x1, coordinates.y1);
		gdouble scale_x = (coordinates.x2 - coordinates.x1) / gdk_pixbuf_get_width(self->pixbuf);
		gdouble scale_y = (coordinates.y2 - coordinates.y1) / gdk_pixbuf_get_height(self->pixbuf);
		cairo_scale(cr, scale_x, scale_y);
		gdk_cairo_set_source_pixbuf(cr, self->pixbuf, 0.0, 0.0);
		cairo_fill(cr);
		cairo_restore(cr);
	}
}

static void
cp_update_bounds(CcItem* item, CcView const* view, gpointer user_data) {
	CcPixbuf* self = CC_PIXBUF(item);
	CcDRect* new_bounds = NULL;
	CcDRect* bounds = cc_hash_map_lookup(item->bounds, view);

	if(self->pixbuf) {
		new_bounds = &self->position;
	}

	if(new_bounds != bounds && (!new_bounds || !bounds || !cc_d_rect_equal(*new_bounds, *bounds))) {
		if(bounds) {
			cc_item_dirty(item, view, *bounds);
			cc_hash_map_remove(item->bounds, view);
		}

		if(new_bounds) {
			cc_hash_map_insert(item->bounds, (gpointer)view, cc_d_rect_copy(new_bounds));
			cc_item_dirty(item, view, *new_bounds);
			cc_item_bounds_changed(item, view);
		}
	}
}

static void
cc_pixbuf_class_init(CcPixbufClass* self_class) {
	GObjectClass* go_class;
	CcItemClass * ci_class;

	/* GObjectClass */
	go_class = G_OBJECT_CLASS(self_class);
	go_class->dispose      = cp_dispose;
	go_class->get_property = cp_get_property;
	go_class->set_property = cp_set_property;

	g_object_class_install_property(go_class,
					PROP_PIXBUF,
					g_param_spec_object("pixbuf",
							    "pixbuf",
							    "pixbuf",
							    GDK_TYPE_PIXBUF,
							    G_PARAM_READWRITE));
	g_object_class_install_property(go_class,
					PROP_POS_W,
					g_param_spec_double("position-w",
							    "position-w",
							    "The width of the image",
							    -G_MAXDOUBLE, G_MAXDOUBLE,
							    0.0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(go_class,
					PROP_POS_H,
					g_param_spec_double("position-h",
							    "position-h",
							    "The height of the image",
							    -G_MAXDOUBLE, G_MAXDOUBLE,
							    0.0,
							    G_PARAM_READWRITE));

	/* CcItemClass */
	ci_class = CC_ITEM_CLASS(self_class);
	// FIXME: implement distance
	ci_class->render        = cp_render;
	ci_class->update_bounds = cp_update_bounds;
}

