/* Terminology
 *
 *     - create/destroy       - the X resource is created/destroyed
 *     - new/free/ref/unref   - the local proxy is created/destroyed
 *
 * Generally local proxies can exist even if the other resource doesn't
 * anymore. In that case they don't have an associated XID.
 *
 * And of course we are being async.
 *
 */

#include <X11/X.h>
#include <X11/Xutil.h>
#include <X11/extensions/Xcomposite.h>
#include <X11/extensions/Xdamage.h>
#include <X11/extensions/XTest.h>

#include <glib-object.h>

#include "ws.h"
#include "wsint.h"
#include "watch.h"

G_DEFINE_TYPE (WsDisplay, ws_display, G_TYPE_OBJECT);

/*
 * Window system object
 */
struct Trap
{
    int begin_serial;
    int end_serial;
    gboolean caught_something;
    
    Trap *next;
};

static void
ws_display_finalize (GObject *object)
{
    WsDisplay *display = WS_DISPLAY (object);
    
    g_hash_table_destroy (display->damage_table);
    
    G_OBJECT_CLASS (ws_display_parent_class)->finalize (object);
}

static void
ws_display_class_init (WsDisplayClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    object_class->finalize = ws_display_finalize;
}

static void
ws_display_init (WsDisplay *display)
{
    display->damage_table = g_hash_table_new_full (
	g_direct_hash, g_direct_equal, NULL, g_free);
}

typedef struct DamageInfo
{
    XID			damage;
    DamageCallback	cb;
    gpointer		data;
} DamageInfo;

/*
 * These two functions basically exist to work around freedesktop bug 5730
 */
void
_ws_display_register_damage (WsDisplay      *display,
			     XID             damage,
			     DamageCallback  cb,
			     gpointer        data)
{
    DamageInfo *info;
    
    g_return_if_fail (
	g_hash_table_lookup (display->damage_table, (gpointer)damage) == NULL);

    info = g_new (DamageInfo, 1);
    
    info->damage = damage;
    info->cb = cb;
    info->data = data;
    
    g_hash_table_insert (display->damage_table, (gpointer)damage, info);
}

void
_ws_display_unregister_damage (WsDisplay	 *display,
			       XID		  damage)
{
    DamageInfo *damage_info = g_hash_table_lookup (
	display->damage_table, (gpointer)damage);
    
    g_return_if_fail (damage_info != NULL);
    
    g_hash_table_remove (display->damage_table, (gpointer)damage);
}

static void
process_damage_event (WsDisplay		  *display,
		      XDamageNotifyEvent *damage_event)
{
    XID damage = damage_event->damage;
    DamageInfo *info =
	g_hash_table_lookup (display->damage_table, (gpointer)damage);
    
    if (info)
	info->cb (display, damage, info->data);
}

static void
process_event (WsDisplay *display,
	       XEvent    *xevent)
{
#if 0
    if (xevent->type == XMotionNotify)
	g_print ("motion\n");
    else
	g_print ("not motion\n");
#endif
    
    if (display->damage.available &&
	xevent->type == display->damage.event_base + XDamageNotify)
    {
	XDamageNotifyEvent *de = (XDamageNotifyEvent *)xevent;
	
	process_damage_event (display, de);
    }
    else if (xevent->type == ConfigureNotify)
    {
	Window xwindow = ((XConfigureEvent *)xevent)->window;
	WsWindow *window = g_hash_table_lookup (
	    display->xresources, (gpointer)xwindow);
	
	if (window)
	    _ws_window_process_event (window, xevent);
    }
}

static void
process_events (gpointer data)
{
    WsDisplay *display = data;
    Display *xdisplay = display->xdisplay;
    
    while (XPending (display->xdisplay))
    {
	XEvent event;
	
	XNextEvent (xdisplay, &event);
	
	process_event (display, &event);
    }
}

static void
init_display (WsDisplay *display)
{
    int i;
    Display *xdisplay = display->xdisplay;
    
    display->n_screens = ScreenCount (xdisplay);
    display->screens = g_new0 (WsScreen *, display->n_screens);
    
    for (i = 0; i < display->n_screens; ++i)
	display->screens[i] = ws_display_get_screen_from_number (display, i);
    
    display->xresources = g_hash_table_new (g_direct_hash, g_direct_equal);
    
    display->composite.available = FALSE;
    display->damage.available = FALSE;
    display->shm.available = FALSE;
    display->test.available = FALSE;
    display->fixes.available = FALSE;
}

static ErrorHandlerFunc old_x_error_handler;
static GList *all_displays;

static void
pop_obsolete_handlers (WsDisplay *display,
		       int serial)
{
    Trap *new_top;
    
    new_top = display->traps;
    while (new_top != NULL &&
	   new_top->end_serial != -1 &&
	   new_top->end_serial <= serial)
    {
	new_top = new_top->next;
    }
    
    while (display->traps != new_top)
    {
	Trap *trap;
	
	trap = display->traps;
	display->traps = trap->next;
	
#if 0
	g_print ("popping: %d %d (%p)\n",
		 trap->begin_serial, trap->end_serial, display->xdisplay);
#endif
	
	g_free (trap);
    }
    
    if (!display->traps)
    {
	XSetErrorHandler (display->old_handler);
	display->old_handler = NULL;
    }
}

static void
process_error (WsDisplay *display,
	       XErrorEvent *error)
{
    Trap *trap;
    char buf[64];
    
    if (!display->traps)
	g_print ("no traps before popping\n");
    
    pop_obsolete_handlers (display, error->serial);
    
    if (!display->traps)
	g_print ("no traps after popping\n");
    
    for (trap = display->traps; trap; trap = trap->next)
    {
#if 0
	g_print ("trap: %d %d\n", trap->begin_serial, trap->end_serial);
#endif
	if (trap->begin_serial <= error->serial &&
	    (trap->end_serial > error->serial || trap->end_serial == -1))
	{
	    trap->caught_something = TRUE;
	    return;
	}
    }
    
    XGetErrorText (display->xdisplay, error->error_code, buf, 63);  
    
    g_error ("Unexpected X error: %s serial %ld error_code "
	     "%d request_code %d minor_code %d)\n",
	     buf,
	     error->serial, 
	     error->error_code, 
	     error->request_code,
	     error->minor_code);
}

static int
xerror_handler (Display		*xdisplay,
		XErrorEvent	*error_event)
{
    GList *list;
    WsDisplay *error_display = NULL;
    
    for (list = all_displays; list != NULL; list = list->next)
    {
	WsDisplay *display = list->data;
	
	if (display->xdisplay == xdisplay)
	{
	    error_display = display;
	    break;
	}
    }
    
    if (error_display)
    {
	process_error (error_display, error_event);
    }
    else if (old_x_error_handler)
    {
	(* old_x_error_handler) (xdisplay, error_event);
    }
    
    return 42;
}

WsDisplay *
ws_display_new (const char *dpy)
{
    Display *xdisplay = XOpenDisplay (dpy);
    WsDisplay *display;
    int fd;
    
    if (!xdisplay)
	return NULL;
    
    display = g_object_new (WS_TYPE_DISPLAY, NULL);
    
    display->xdisplay = xdisplay;
    
    init_display (display);
    
    fd = ConnectionNumber (xdisplay);
    
    /* process events that are already read off the wire */
    process_events (display);
    
    fd_add_watch (fd, display);
    fd_set_read_callback (fd, process_events);
    fd_set_poll_callback (fd, process_events);
    
    if (!all_displays)
	old_x_error_handler = XSetErrorHandler (xerror_handler);
    
    all_displays = g_list_prepend (all_displays, display);
    
    return display;
}

void
ws_display_process_xevent (WsDisplay *ws,
			   XEvent *xevent)
{
    pop_obsolete_handlers (ws, xevent->xany.serial);
    process_event (ws, xevent);
}

void
ws_display_begin_error_trap  (WsDisplay *display)
{
    struct Trap *trap = g_new0 (Trap, 1);
    
    trap->begin_serial = NextRequest (display->xdisplay);
    trap->end_serial = -1;
    trap->next = display->traps;
    trap->caught_something = FALSE;
    
    display->traps = trap;
    
    display->old_handler = XSetErrorHandler (xerror_handler);
    
#if 0
    g_print ("beginning: %d (%p)\n", trap->begin_serial, display->xdisplay);
#endif
    g_assert (display->traps);
}

static Trap *
ws_display_end_error_trap_internal (WsDisplay *display)
{
    Trap *trap;
    
    /* Find first open trap */
    trap = display->traps;
    while (trap && trap->end_serial != -1)
	trap = trap->next;
    
    if (!trap)
    {
	g_warning ("ws_end_trap_errors() called without corresponding "
		   "ws_begin_trap_errors()");
	return NULL;
    }
    
    trap->end_serial = NextRequest (display->xdisplay);
    return trap;
}


void
ws_display_end_error_trap (WsDisplay *display)
{
    ws_display_end_error_trap_internal (display);
}

gboolean
ws_display_end_error_trap_with_return (WsDisplay *display)
{
    Trap *trap = ws_display_end_error_trap_internal (display);
    
    ws_display_sync (display);
    
    return trap->caught_something;
}

gboolean
ws_display_init_composite (WsDisplay *display)
{
    if (XCompositeQueryExtension (
	    display->xdisplay,
	    &display->composite.event_base,
	    &display->composite.error_base))
    {
	display->composite.available = TRUE;
	
	return TRUE;
    }
    
    return FALSE;
}

gboolean
ws_display_init_fixes (WsDisplay *display)
{
    if (XFixesQueryExtension (
	    display->xdisplay,
	    &display->composite.event_base,
	    &display->composite.error_base))
    {
	display->fixes.available = TRUE;
	
	return TRUE;
    }
    
    return FALSE;
}

gboolean
ws_display_init_damage (WsDisplay *display)
{
    if (XDamageQueryExtension (
	    display->xdisplay,
	    &display->damage.event_base,
	    &display->damage.error_base))
    {
	display->damage.available = TRUE;
	return TRUE;
    }
    
    return FALSE;
}

gboolean
ws_display_init_test (WsDisplay *display)
{
    int dummy1, dummy2;
    
    if (XTestQueryExtension (
	    display->xdisplay,
	    &display->test.event_base,
	    &display->test.error_base,
	    &dummy1, &dummy2))
    {
	display->test.available = TRUE;
	return TRUE;
    }
    
    return FALSE;
}

void
ws_display_flush (WsDisplay *display)
{
    g_return_if_fail (display != NULL);
    
    XFlush (display->xdisplay);
}

void
ws_display_sync (WsDisplay *display)
{
    g_return_if_fail (display != NULL);
    
    XSync (display->xdisplay, 0);
}

void
ws_display_set_synchronize (WsDisplay *display,
			    gboolean sync)
{
    XSynchronize (display->xdisplay, sync);
}

WsWindow *
_ws_display_lookup_window (WsDisplay *display, XID xid)
{
    g_return_val_if_fail (display != NULL, NULL);
    
    return g_hash_table_lookup (display->xresources, (gpointer)xid);
}

void
ws_display_set_ignore_grabs (WsDisplay *display,
			     gboolean  ignore)
{
    g_return_if_fail (display->test.available);
    
    XTestGrabControl (display->xdisplay, ignore);
}

void
_ws_display_add_resource (WsDisplay *display, XID xid, WsResource *resource)
{
    g_hash_table_insert (display->xresources, (gpointer)xid, resource);
}

WsScreen *
ws_display_get_default_screen (WsDisplay *display)
{
    Screen *xscreen;
    int screen_no;
    int i;
    
    g_return_val_if_fail (display != NULL, NULL);
    
    screen_no = DefaultScreen (display->xdisplay);
    xscreen = ScreenOfDisplay (display->xdisplay, screen_no);
    
    for (i = 0; i < display->n_screens; ++i)
    {
	if (display->screens[i]->xscreen == xscreen)
	    return display->screens[i];
    }
    
    return NULL;
}

WsScreen *
ws_display_get_screen_from_number (WsDisplay *display,
				   int number)
{
    if (!display->screens[number])
    {
	display->screens[number] = _ws_screen_new (
	    display, ScreenOfDisplay (display->xdisplay, number));
    }
    
    return display->screens[number];
    
}

WsScreen *
_ws_display_lookup_screen (WsDisplay    *display,
			   Screen       *xscreen)
{
    int i;
    
    for (i = 0; i < display->n_screens; ++i)
    {
	if (display->screens[i]->xscreen == xscreen)
	    return display->screens[i];
    }
    
    return NULL;
}

void
ws_display_grab (WsDisplay *display)
{
    XGrabServer (display->xdisplay);
}

void
ws_display_ungrab (WsDisplay *display)
{
    XUngrabServer (display->xdisplay);
}

#if 0
typedef enum
{
    CORE,
    DAMAGE,
    SYNC
} Extension;

typedef struct
{
    guint	xtype;
    const char *name;
    int		drawable_offset;
    int		extenstion_event_offset;
} EventTranslation;

const EventTranslation translation_table[] =
{
    { KeyPress,   "KeyPress",   G_STRUCT_OFFSET (XKeyPressedEvent, window) } ,
    { KeyRelease, "KeyRelease", G_STRUCT_OFFSET (XKeyReleasedEvent, window) } ,
    { ButtonPress, "ButtonPress", G_STRUCT_OFFSET (XButtonEvent, window) } ,
    { ButtonRelease, "ButtonRelease", G_STRUCT_OFFSET (XButtonEvent, window) } ,
    { MotionNotify, "MotionNotify", G_STRUCT_OFFSET (XMotionEvent, window) } ,
    { EnterNotify, "EnterNotify", G_STRUCT_OFFSET (XEnterWindowEvent, window) } ,
    { LeaveNotify, "LeaveNotify", G_STRUCT_OFFSET (XLeaveWindowEvent, window) } ,
    { FocusIn, "FocusIn", G_STRUCT_OFFSET (XFocusInEvent, window) } ,
    { FocusOut, "FocusOut", G_STRUCT_OFFSET (XFocusOutEvent, window) } ,
    { KeymapNotify, "KeymapNotify", G_STRUCT_OFFSET (XKeymapEvent, window) } ,
    { Expose,           "Expose", G_STRUCT_OFFSET (XExposeEvent, window) } ,
    { GraphicsExpose,   "GraphicsExpose", G_STRUCT_OFFSET (XGraphicsExposeEvent, drawable) } ,
    { NoExpose,         "NoExpose", G_STRUCT_OFFSET (XNoExposeEvent, drawable) } ,
    { VisibilityNotify, "VisibilityNotify", G_STRUCT_OFFSET (XVisibilityEvent, window) } ,
    { CreateNotify, "CreateNotify", G_STRUCT_OFFSET (XCreateWindowEvent, window) } ,
    { DestroyNotify, "DestroyNotify", G_STRUCT_OFFSET (XDestroyWindowEvent, window) } ,
    { UnmapNotify, "UnmapNotify", G_STRUCT_OFFSET (XUnmapEvent, window) } ,
    { MapNotify, "MapNotify", G_STRUCT_OFFSET (XMapEvent, window) } ,
    { MapRequest, "MapRequest", G_STRUCT_OFFSET (XMapRequestEvent, window) } ,
    { ReparentNotify, "ReparentNotify", G_STRUCT_OFFSET (XReparentEvent, window) } ,
    { ConfigureNotify, "ConfigureNotify", G_STRUCT_OFFSET (XConfigureEvent, window) } ,
    { ConfigureRequest, "ConfigureRequest", G_STRUCT_OFFSET (XConfigureRequestEvent, window) } ,
    { GravityNotify, "GravityNotify", G_STRUCT_OFFSET (XGravityEvent, window) } ,
    { ResizeRequest, "ResizeRequest", G_STRUCT_OFFSET (XResizeRequestEvent, window) } ,
    { CirculateNotify, "CirculateNotify", G_STRUCT_OFFSET (XCirculateEvent, window) } ,
    { CirculateRequest, "CirculateRequest", G_STRUCT_OFFSET (XCirculateRequestEvent, window) } ,
    { PropertyNotify, "PropertyNotify", G_STRUCT_OFFSET (XPropertyEvent, window) } ,
    { SelectionClear, "SelectionClear", G_STRUCT_OFFSET (XSelectionClearEvent, window) } ,
    { SelectionRequest, "SelectionRequest", G_STRUCT_OFFSET (XSelectionRequestEvent, window) } ,
    { SelectionNotify, "SelectionNotify", G_STRUCT_OFFSET (XSelectionEvent, window) } ,
    { ColormapNotify, "ColormapNotify", G_STRUCT_OFFSET (XColormapEvent, window) } ,
    { ClientMessage, "ClientMessage", G_STRUCT_OFFSET (XClientMessageEvent, window) } ,
    { MappingNotify, "MappingNotify", G_STRUCT_OFFSET (XMappingNotifyEvent, window) },
};
#endif

