use Data::Dumper;
use IO::File;
use Path::Tiny;
use Lintian::Data;
use Lintian::Profile;

use strict;
use warnings;

our $conf_file_name = "control";
our $conf_dir       = 'debian';
our $model_to_test  = "Dpkg::Control";

my $profile = Lintian::Profile->new ('debian');
Lintian::Data->set_vendor($profile);
my $standards = Lintian::Data->new('standards-version/release-dates', qr/\s+/);
my @std_list = $standards->all();

my $t3_description = "This is an extension of Dist::Zilla::Plugin::InlineFiles, 
providing the following file:

 - xt/release/pod-spell.t - a standard Test::Spelling test" ;

our @tests = (
    {

        # t0
        check => {
            'source Source',          "libdist-zilla-plugins-cjm-perl",
            'source Standards-Version' => $std_list[0],
            'source Build-Depends:0', "debhelper (>= 7)",
            # not really happy about the versioned dep of libmeta-cpan-perl created ex-nihilo.
            # this is a side effect of other compromises in dual dependency handling which should be harmless
            'source Build-Depends-Indep:0', "libcpan-meta-perl (>= 2.110440) | perl (>= 5.13.10)",     # fixed
            'source Build-Depends-Indep:1', "libdist-zilla-perl",    # fixed
            'source Build-Depends-Indep:5', "libpath-class-perl",
            'source Build-Depends-Indep:6', "libmodule-build-perl (>= 0.360000) | perl (<< 5.19.0)", # fixed
            'source Build-Depends-Indep:7', "udev [linux-any] | makedev [linux-any]",
            'source X-Comment' => qr/Debian #810023/,
            'source Priority' => 'optional',
            'binary:libdist-zilla-plugins-cjm-perl X-Comment' => qr/Debian #810023/,

            'binary:libdist-zilla-plugins-cjm-perl Depends:0',
            '${misc:Depends}',
            'source Vcs-Browser' ,'https://salsa.debian.org/perl-team/modules/packages/libdist-zilla-plugins-cjm-perl',
            'source Vcs-Git', 'https://salsa.debian.org/perl-team/modules/packages/libdist-zilla-plugins-cjm-perl.git',
        },
        log4perl_load_warnings => [ [ User => map { (warn => $_ ) } (
            qr/standards version/,
            qr/dependency/, qr/dual life/, (qr/dependency/) x 2,
            qr/libmodule-build-perl \(>= 0.360000\) \| perl \(>= 5.8.8\)/,
            qr/should be 'libmodule-build-perl \(>= 0.360000\)/,
            qr/dependency/, qr/dual life/, (qr/dependency/) x 2
        ) ] ],
        apply_fix => 1,
        file_contents_like => {
            'debian/control' => [
                # check that write order is tweaked for Standards-Version
                qr!Standards-Version: [\d.]+\nVcs-Browser!
            ]
        },
    },
    {

        # t1
        check => { 'binary:seaview Recommends:0', 'clustalw', },
        apply_fix => 1,
        load => 'binary:seaview Synopsis="multiplatform interface for sequence alignment"',
    },
    {

        # t2
        check => {
            'binary:xserver-xorg-video-all Architecture' => 'any',
            'binary:xserver-xorg-video-all Depends:0'    => '${F:XServer-Xorg-Video-Depends}',
            'binary:xserver-xorg-video-all Depends:1'    => '${misc:Depends}',
            'binary:xserver-xorg-video-all Replaces:0'   => 'xserver-xorg-driver-all',
            'binary:xserver-xorg-video-all Conflicts:0'  => 'xserver-xorg-driver-all',
        },
        apply_fix => 1,
    },
    {

        # t3
        check => {
            # no longer mess up synopsis with lcfirst
            'binary:libdist-zilla-plugin-podspellingtests-perl Synopsis' =>
              "Release tests for POD spelling",
            'binary:libdist-zilla-plugin-podspellingtests-perl Description' => $t3_description ,
        },
        load => 'binary:libdist-zilla-plugin-podspellingtests-perl '
            . 'Description="'.$t3_description.'"',
        apply_fix => 1,
    },
    {

        # t4, also checks XS-Python-Version deprecation
        check => {
            'source Priority' => 'extra',
            'source X-Python-Version' => ">= 2.3, << 2.5",
            'source Standards-Version', "3.9.8",
        },
        log4perl_load_warnings => [ [
            User =>
            warn => qr/Standards-Version/,
            warn => qr/deprecated/
        ] ],
    },
    {

        # t5
        check => { 'source X-Python-Version' => ">= 2.3, << 2.6" },
        log4perl_load_warnings => [[ User => warn => qr/deprecated/ ]],
    },
    {

        # t6
        check => { 'source X-Python-Version' => ">= 2.3" },
        log4perl_load_warnings => [[ User => warn => qr/deprecated/ ]],
    },
    {
        name => 'sdlperl',
        load => 'source Uploaders:2="Sam Hocevar (Debian packages) <sam@zoy.org>"',
        load_check => 'skip',
        check => {
            'binary:libsdl-perl Depends:2' => '${misc:Depends}',
            'binary:libsdl-perl Conflicts:2' => undef,
        },
        apply_fix => 1,
    },
    {
        name => 'libpango-perl',
        verify_annotation => {
            'source Build-Depends' => "do NOT add libgtk2-perl to build-deps (see bug #554704)",
            'source Maintainer'    => "what a fine\nteam this one is",
        },
        apply_fix => 1,
    },
    {
        name => 'libwx-scintilla-perl',
        apply_fix => 1,
    },
    {
        # test for #683861 -- dependency version check and epochs
        name => 'libmodule-metadata-perl',
        apply_fix => 1,
    },
    {
        # test for #682730 (reduces libclass-isa-perl | perl (<< 5.10.1-13) to perl)
        name => 'libclass-meta-perl',
        check => { 'source Build-Depends-Indep:1' => 'libclass-isa-perl | perl (<< 5.11.0)' },
        apply_fix => 1,
    },
    {
        # test for #692849, must not warn about missing libfoo dependency
        name => 'dbg-dep',
    },
    {
        # test for #696768, Built-Using field
        name => 'built-using',
        apply_fix => 1,
    },
    {
        # test for #719753, XS-Autobuild field
        name => 'non-free',
        check => {
            'source Section' => 'non-free/libs',
            'source XS-Autobuild' => 'yes',
        },
    },
    {
        # test for #713053,  XS-Ruby-Versions and XB-Ruby-Versions fields
        name => 'ruby',
        apply_fix => 1, # to fix pkg-testsuite param
        check => {
            'source XS-Ruby-Versions' => 'all',
            'binary:libfast-xs-ruby XB-Ruby-Versions' => '${ruby:Versions}',
        },
    },
    {
        # test for #903905,  XS-Ruby-Versions in a package not maintained
        # by ruby team
        name => 'ruby-in-med-team',
        check => {
            'source XS-Ruby-Versions' => 'all',
            'binary:ruby-rgfa XB-Ruby-Versions' => '${ruby:Versions}',
        },
    },
    {
        # test for XS-Testsuite field
        name => 'xs-testsuite',
        log4perl_load_warnings => [ [
            User => warn => qr/source Standards-Version/,
            warn => qr/deprecated/
        ] ],
        apply_fix => 1,
        check => {
            'source Testsuite' => 'autopkgtest-pkg-ruby',
        },
    },
    {
        name => 'gnu-r-stuff',
        # last warning is about line too long.
        log4perl_load_warnings => [[
            User => map { (warn => $_)} qr/standards version/ , (qr/Vcs/) x 2 , qr/Description/
        ]],
        apply_fix => 1,
        check => [
            'source Section' => 'gnu-r',
            'binary:gnu-r-view Section' => 'gnu-r',
        ]
    },
    {
        name => 'build-profiles',
        apply_fix => 1,
        check => {
            'binary:pkg-config Build-Profiles' => '<!stage1>',
            'binary:pkg-config-stage1 Build-Profiles' => '<stage1>',
            'source Build-Depends:3' => 'libglib2.0-dev <!stage1>'
        },
    },

    {
        name => 'comments-in-dep-list',
        file_contents_like => {
            "debian/control" => qr/# Python/,
        }
    },

    {
        name => 'tricky-comment',
        verify_annotation => {
            'binary:libmoosex-types-iso8601-perl Synopsis' => "not yet packaged\nRecommends",
        }
    },

    {
        name => 'med-team',
        log4perl_load_warnings => [ [
            User => warn => qr/source Standards-Version/,
            ( warn => qr/canonical/) x 2
        ]],
        apply_fix => 1,
        check => {
            'source Vcs-Git' => 'https://salsa.debian.org/med-team/abacas.git',
            'source Vcs-Browser' => 'https://salsa.debian.org/med-team/abacas',
        },
    },
    {
        name => 'neurodebian-team',
        apply_fix => 1,
        check => {
            'source Vcs-Git' => 'https://salsa.debian.org/neurodebian-team/caret-data.git',
            'source Vcs-Browser' => 'https://salsa.debian.org/neurodebian-team/caret-data',
        },
    },
    {
        name => 'bcftools',
        apply_fix => 1,
        verify_annotation => {
            'binary:bcftools Suggests:0' => "These are needed for plot-vcfstats"
        },
        check => {
            'binary:bcftools Suggests:0' => "python"
        },
    },
);

my $cache_file = 't/model_tests.d/dependency-cache.txt';

$Config::Model::Dpkg::Dependency::use_test_cache = 1;

my $ch = new IO::File "$cache_file";
foreach ($ch->getlines) {
    chomp;
    my ($k,$v) = split m/ => / ;
    $Config::Model::Dpkg::Dependency::cache{$k} = time . ' '. $v ;
}
$ch -> close ;

END {
    return if $::DebianDependencyCacheWritten ;
    my %h = %Config::Model::Dpkg::Dependency::cache ;
    map { s/^\d+ //;} values %h ; # remove time stamp
    my $str = join ("\n", map { "$_ => $h{$_}" ;} sort keys %h) ;

    my $fh = new IO::File "> $cache_file";
    print "writing back cache file\n";
    if ( defined $fh ) {
        # not a big deal if cache cannot be written back
        $fh->print($str);
        $fh->close;
        $::DebianDependencyCacheWritten=1;
    }
}

1;
