/*--------------------------------------------------------------------------+
$Id: ProcessUtils.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.io;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;

/**
 * Executes a system process. Takes care of reading stdout and stderr of the
 * process in separate threads to avoid blocking.
 * 
 * @author juergens
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: 4D8721B16BD1F29DDCC9C144FC6E0BCC
 */
public class ProcessUtils {

	/**
	 * Executes a process in a thread-safe way.
	 * 
	 * @param completeArguments
	 *            Array of command line arguments to start the process
	 * 
	 * @return result of the execution
	 */
	public static ExecutionResult execute(String[] completeArguments)
			throws IOException {
		return execute(completeArguments, null);
	}

	/**
	 * Executes a process in a thread-safe way.
	 * 
	 * @param completeArguments
	 *            Array of command line arguments to start the process
	 * @param input
	 *            String that gets written to stdin
	 * 
	 * @return result of the execution
	 */
	public static ExecutionResult execute(String[] completeArguments,
			String input) throws IOException {
		ProcessBuilder builder = new ProcessBuilder(completeArguments);
		return execute(builder, input);
	}

	/**
	 * Executes a process in a thread-safe way.
	 * 
	 * @param builder
	 *            builder that gets executed
	 * @return result of the execution
	 */
	public static ExecutionResult execute(ProcessBuilder builder)
			throws IOException {
		return execute(builder, null);
	}

	/**
	 * Executes a process in a thread-safe way.
	 * 
	 * @param builder
	 *            builder that gets executed
	 * @param input
	 *            String that gets written to stdin
	 * @return result of the execution
	 */
	public static ExecutionResult execute(ProcessBuilder builder, String input)
			throws IOException {
		// start process
		Process process = builder.start();

		// read error for later use
		StreamReaderThread stderrReader = new StreamReaderThread(process
				.getErrorStream());
		StreamReaderThread stdoutReader = new StreamReaderThread(process
				.getInputStream());

		// write input to process
		if (input != null) {
			Writer stdIn = new OutputStreamWriter(process.getOutputStream());
			stdIn.write(input);
			stdIn.close();
		}

		// wait for process
		try {
			process.waitFor();

			// It is important to wait for the threads, so the output is
			// completely stored.
			stderrReader.join();
			stdoutReader.join();

		} catch (InterruptedException e) {
			// ignore this one
		}

		return new ExecutionResult(stdoutReader.getContent(), stderrReader
				.getContent(), process.exitValue());
	}

	/**
	 * Parameter object that encapsulates the result of a process execution.
	 * This object is immutable.
	 */
	public static class ExecutionResult {

		/** Output on stdout of the process */
		private final String stdout;

		/** Output on stderr of the process */
		private final String stderr;

		/** Return code of the process */
		private final int returnCode;

		/** Constructor */
		private ExecutionResult(String stdout, String stderr, int returnCode) {
			this.stdout = stdout;
			this.stderr = stderr;
			this.returnCode = returnCode;
		}

		/** Returns stdout. */
		public String getStdout() {
			return stdout;
		}

		/** Returns stderr. */
		public String getStderr() {
			return stderr;
		}

		/** Returns returnCode. */
		public int getReturnCode() {
			return returnCode;
		}
	}
}