/***************************************************************************
                          cstring.cpp  -  description
                             -------------------
    begin                : Fri Sep 21 2001
    copyright            : (C) 2001-2003 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include <dclib/dcos.h>
#include "cstring.h"

CString::CString()
{
	Init();
}

CString::CString( const char * sz )
{
	Init();

	set(sz);
}

CString::CString( const char ch )
{
	Init();

	set(&ch,1);
}

CString::CString( const CString & stringSrc ) : CObject()
{
	Init();

	set(stringSrc.m_szBuffer,stringSrc.m_nStringLength);
}

CString::~CString()
{
	if (m_szBuffer)
	{
		free(m_szBuffer);
		m_szBuffer = 0;
	}
}

/** */
void CString::Init()
{
	m_szBuffer      = 0;
	m_nStringLength = 0;
	m_nBufferSize   = 0;
}

/** */
void CString::Empty()
{
	if (m_szBuffer)
	{
		free(m_szBuffer);
	}

	Init();
}

/** */
unsigned char CString::GetHash( long value ) const
{
	if ( (!IsEmpty()) && (value < m_nStringLength) )
		return ((unsigned char)(m_szBuffer[value]&0xFF));
	else
		return 0;
}

/** */
long CString::Find( const char ch, long nStart ) const
{
	long i;

	if ( IsEmpty() )
	{
		return -1;
	}

	if ( nStart == -1 )
		nStart = 0;

	if ( nStart > m_nStringLength )
		return -1;

	for(i=nStart;i<m_nStringLength;i++)
	{
		if ( m_szBuffer[i] == ch )
			return i;
	}

	return -1;
}

/** */
long CString::Find( const char * sz, long nStart, bool cs ) const
{
	long i;
	long len;
	long ret;

	if ( !cs )
	{
		return FindCase(sz,nStart);
	}

	if ( IsEmpty() )
	{
		if ( sz != 0 )
			return -1;
		else
			return 0;
	}

	if ( !sz )
		return -1;

	if ( nStart == -1 )
		nStart = 0;

	len = strlen(sz);

	if ( (nStart+len) > m_nStringLength )
		return -1;

	for(i=nStart,ret=-1;i<=(m_nStringLength-len);i++)
	{
		if ( strncmp(sz,m_szBuffer+i,len) == 0 )
		{
			ret = i; //-index;
			break;
		}
	}

	return ret;
}

/** */
long CString::Find( const CString & string, long nStart, bool cs ) const
{
	long i;
	long len;
	long ret;
	char * sz;

	if ( !cs )
	{
		return FindCase(string,nStart);
	}

	sz  = string.Data();
	len = string.Length();

	if ( IsEmpty() )
	{
		if ( sz != 0 )
			return -1;
		else
			return 0;
	}

	if ( !sz )
		return -1;

	if ( nStart == -1 )
		nStart = 0;

	if ( (nStart+len) > m_nStringLength )
		return -1;

	for(i=nStart,ret=-1;i<=(m_nStringLength-len);i++)
	{
		if ( strncmp(sz,m_szBuffer+i,len) == 0 )
		{
			ret = i; //-index;
			break;
		}
	}

	return ret;
}

/** */
long CString::FindCase( const char * sz, long nStart ) const
{
	long i;
	long len;
	long ret;

	if ( IsEmpty() )
	{
		if ( sz != 0 )
			return -1;
		else
			return 0;
	}

	if ( !sz )
		return -1;

	if ( nStart == -1 )
		nStart = 0;

	len = strlen(sz);

	if ( (nStart+len) > m_nStringLength )
		return -1;

	for(i=nStart,ret=-1;i<=(m_nStringLength-len);i++)
	{
#ifdef WIN32
		if ( _strnicmp(sz,m_szBuffer+i,len) == 0 )
#else
		if ( strncasecmp(sz,m_szBuffer+i,len) == 0 )
#endif
		{
			ret = i-nStart;
			break;
		}
	}

	return ret;
}

/** */
long CString::FindCase( const CString & string, long nStart ) const
{
	long i;
	long len;
	long ret;
	char * sz;

	sz  = string.Data();
	len = string.Length();

	if ( IsEmpty() )
	{
		if ( sz != 0 )
			return -1;
		else
			return 0;
	}

	if ( !sz )
		return -1;

	if ( nStart == -1 )
		nStart = 0;

	if ( (nStart+len) > m_nStringLength )
		return -1;

	for(i=nStart,ret=-1;i<=(m_nStringLength-len);i++)
	{
#ifdef WIN32
		if ( _strnicmp(sz,m_szBuffer+i,len) == 0 )
#else
		if ( strncasecmp(sz,m_szBuffer+i,len) == 0 )
#endif
		{
			ret = i-nStart;
			break;
		}
	}

	return ret;
}

/** */
long CString::FindRev( char ch, long nStart ) const
{
	long i;

	if ( IsEmpty() )
		return -1;

	if ( nStart == -1 )
		nStart = m_nStringLength;

	if ( nStart > (m_nStringLength+1) )
		return -1;

	for(i=nStart;i>=0;i--)
	{
		if (m_szBuffer[i]==ch)
			return i;
	}

	return -1;
}

/** */
long CString::FindRev( const CString & string ) const
{
	long i;
	long len;
	long ret=-1;
	char * sz;

	sz  = string.Data();
	len = string.Length();

	if ( IsEmpty() )
	{
		if ( sz != 0 )
			return -1;
		else
			return 0;
	}

	if ( !sz )
		return -1;

	for(i=(m_nStringLength-len);i>=0;i--)
	{
		if ( string == CString(m_szBuffer).Mid(i,len) )
		{
			ret = i; //-index;
			break;
		}
	}

	return ret;
}

/** */
CString CString::Mid( long nFirst, long nCount ) const
{
	CString tmp = "";

	if ( IsEmpty() )
		return tmp;
	if ( nCount == -1 )
		nCount = m_nStringLength-nFirst;
	if ( nCount <= 0 )
		return tmp;
	if( nFirst > m_nStringLength )
		return tmp;

	if( (nFirst+nCount) > m_nStringLength )
		return tmp;

	tmp.set(m_szBuffer+nFirst,nCount);

	return tmp;
}

/** */
CString CString::Section( const char sep, int start, int end ) const
{
	int i,si,ei;
	CString tmp = "";

	if ( IsEmpty() )
		return tmp;

	for(i=0,si=0;i<start&&si!=-1;i++,si++)
	{
		si = Find(sep,si);
	}

	if ( si == -1 )
		return tmp;

	for(ei=si;i<=end&&ei!=-1;i++,ei++)
	{
		ei = Find(sep,ei);
	}

	if ( (si==-1) || (ei==-1) )
		return tmp;

	return Mid(si,ei-si-1);
}

/** */
void CString::Append( const CString & string )
{
	CString tmp(string);

	add(tmp.Data(),tmp.Length());
}

/** */
void CString::Append( const char ch )
{
	char d[2];

	d[0]=ch;
	d[1]=0;
	add(d,1);
}

/** */
bool CString::IsEmpty() const
{
	return ( (m_nStringLength==0) || (m_szBuffer==0) );
}

/** */
CString CString::setNum( const int n )
{
	char c[50];

#ifdef WIN32
	_snprintf(c,50,"%d",n);
#else
	snprintf(c,50,"%d",n);
#endif

	return CString(c);
}

/** */
CString CString::setNum( const unsigned int n )
{
	char c[50];

#ifdef WIN32
	_snprintf(c,50,"%u",n);
#else
	snprintf(c,50,"%u",n);
#endif

	return CString(c);
}

/** */
CString CString::setNum( const long n )
{
	char c[50];

#ifdef WIN32
	_snprintf(c,50,"%ld",n);
#else
	snprintf(c,50,"%ld",n);
#endif

	return CString(c);
}

/** */
CString CString::setNum( const ulonglong n )
{
	char c[50];

#ifdef WIN32
	_ui64toa( n, c, 10 );
#else
	snprintf(c,50,"%llu",n);
#endif

	return CString(c);
}

/** */
CString CString::setNum( const double n, const int p )
{
	char c[50];

#ifdef WIN32
	_snprintf(c,50,"%.*f",p,n);
#else
	snprintf(c,50,"%.*f",p,n);
#endif
	return CString(c);
}

/** */
ulonglong CString::asULL( int base ) const
{
	ulonglong res;
	char *tail=0;

	if ( IsEmpty() )
	{
		return 0;
	}

#ifdef WIN32
	res=strtoull(Data(),&tail,base);
#else
	#ifdef HAVE_STRTOULL
	res=strtoull(Data(),&tail,base);
	#elif HAVE_STRTOUQ
	res=strtouq(Data(),&tail,base);
	#elif defined(__arch64__) 
 	res=strtoul(Data(),&tail,base);
	#else
	#error "You don't have strtoull or strtouq function !"
	#endif
#endif

	if((res==0)&&(errno!=0)) {}

	return res;
}

/** */
unsigned int CString::asUINT( int base ) const
{
	return (unsigned int)asLONG(base);
}

/** */
int CString::asINT( int base ) const
{
	return (int)asLONG(base);
}

/** */
long CString::asLONG( int base ) const
{
	unsigned int res;
	char *tail=0;

	if ( IsEmpty() )
	{
		return 0;
	}

	res=strtol(Data(),&tail,base);

	if((res==0)&&(errno!=0)) {}

	return res;
}

/** */
double CString::asDOUBLE() const
{
	char *endptr = 0;

	if ( IsEmpty() )
	{
		return 0;
	}

	return strtod(Data(),&endptr);
}

/** */
CString CString::ToUpper()
{
	CString ret="";
	long i=0;

	if ( IsEmpty() )
		return ret;

	for(;i<m_nStringLength;i++)
	{
		ret += (const char)(toupper(m_szBuffer[i])&0xFF);
	}

	return ret;
}

/** */
CString CString::Replace( CString src, CString string ) const
{
	long i,t;
	CString tmp = "";

	t = i = 0;
	while ( (i=Find(src.Data(),i)) != -1 )
	{
		tmp += Mid(t,i-t);
		tmp += string;
		i+=src.Length();
		t = i;
	}

	tmp += Mid(t,m_nStringLength-t);

	return tmp;
}

/** */
void CString::set( const char * sz, long nLength )
{
	long l;

	Empty();

	if ( !sz )
		return;

	if( nLength == -1 )
		l = strlen(sz);
	else
		l = nLength;

	if (l<=0)
		return;

	m_szBuffer = (char*) malloc(l+1); //new char[l+1];

	if ( m_szBuffer == 0 )
	{
		printf("CString::set malloc [%ld]: %s\n",l+1,strerror(errno));
		return;
	}

	memcpy(m_szBuffer,sz,l);
	m_szBuffer[l]   = 0;
	m_nStringLength = l;
	m_nBufferSize   = l+1;
}

/** */
void CString::add( const char * sz, long nLength )
{
	long l,len,i;
	char *p;

	if (!sz)
		return;

	if( nLength == -1 )
		l = strlen(sz);
	else
		l = nLength;

	if (l<=0)
		return;

	if(!m_szBuffer)
	{
		set(sz,l);
	}
	else
	{
		len = m_nStringLength;

		if((l+len+1)>m_nBufferSize)
		{
			i = 1000+1+l;
			p = (char*) realloc(m_szBuffer,m_nBufferSize+i);

			if ( p == 0 )
			{
				perror("CString::add realloc");
				return;
			}

			m_szBuffer = p;
			m_nBufferSize += i;
		}

		memset(m_szBuffer+len+l,0,1);
		memcpy(m_szBuffer+len,sz,l);
		m_nStringLength+=l;
	}
}

#ifdef WIN32
// original from dietlibc
ulonglong CString::strtoull(const char *nptr, char **endptr, int base) const
{
  longlong v=0;

  while(isspace(*nptr)) ++nptr;

  if (*nptr == '+') ++nptr;
  if (!base) {
    if (*nptr=='0') {
      base=8;
      if ((*(nptr+1)=='x')||(*(nptr+1)=='X')) {
	nptr+=2;
	base=16;
      }
    }
    else
      base=10;
  }
  while(*nptr) {
    register unsigned char c=*nptr;
    c=(c>='a'?c-'a'+10:c>='A'?c-'A'+10:c-'0');
    if (c>=base) break;
    v=v*base+c;
    ++nptr;
  }
  if (endptr) *endptr=(char *)nptr;
  return v;
}
#endif

/** */
bool operator == ( const CString & string1, const CString & string2 )
{
	if ( string1.IsEmpty() && string2.IsEmpty() )
		return TRUE;

	if ( string1.Length() != string2.Length() )
		return FALSE;

	if ( memcmp( string1.Data(), string2.Data(), string1.Length() ) != 0 )
		return FALSE;

	return TRUE;
}

/** */
CString CString::RightJustify( long nNewLength, char fill , bool /*truncate*/ )
{
	CString s="";

	while( (s.m_nStringLength+m_nStringLength) < nNewLength )
		s+=fill;

	return(s+*this);
}

/** */
bool operator == ( const char * sz, const CString & string )
{
	return ( CString(sz) == string );
}

/** */
bool operator == ( const CString & string, const char * sz )
{
	return ( CString(sz) == string );
}

/** */
bool operator != ( const CString & string1, const CString & string2 )
{
	return (!(string1 == string2));
}

/** */
bool operator != ( const char * sz, const CString & string )
{
	return (!(CString(sz) == string));
}

/** */
bool operator != ( const CString & string, const char * sz )
{
	return (!(CString(sz) == string));
}
