/***************************************************************************
                          cconnection.cpp  -  description
                             -------------------
    begin                : Sat Oct 6 2001
    copyright            : (C) 2001-2005 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#ifndef WIN32
#include <stdlib.h>
#include <unistd.h>
#else
#include <wtypes.h>
#include <winbase.h>
#include <stdlib.h>
#endif

#include <dclib/dcos.h>
#include <dclib/core/cbytearray.h>

#include "cconnection.h"

#define BUFFER_SIZE		1024*50
#define DEFAULT_DC_PORT		411
#define CONNECTION_TIMEOUT	60
#define DATA_TIMEOUT		60
#define NOTIFY_TIMEOUT		1

CConnection::CConnection()
{
	m_sIP   = "";
	m_nPort = DEFAULT_DC_PORT;

	m_bForceDisconnect = FALSE;

	// set connection timeout
	m_nConnectTimeout = CONNECTION_TIMEOUT;
	m_timeNotify      = 0;
	m_timeConnection  = 0;

	m_eState = estNONE;

	m_pBuffer = new CByteArray(BUFFER_SIZE);

	m_pSendList = new CThreadList<CObject>();

	m_pConnMutex = new CThread();
}

CConnection::~CConnection()
{
	CSocket::Disconnect();

	if ( m_pBuffer != NULL )
	{
		delete m_pBuffer;
		m_pBuffer = 0;
	}

	if (m_pSendList)
	{
		delete m_pSendList;
		m_pSendList = 0;
	}

	if ( m_pConnMutex )
	{
		delete m_pConnMutex;
		m_pConnMutex = 0;
	}
}

/** */
int CConnection::Connect()
{
	m_pConnMutex->Lock();

	int err = -1;

	if ( m_eState != estNONE )
	{
		StateDisconnect();
	}

	m_eState = estCONNECT;

	m_bForceDisconnect = FALSE;

	err = 0;

	m_pConnMutex->UnLock();

	return err;
}

/** */
int CConnection::Connect( CString ip, eSocketType sockettype )
{
	unsigned int port;
	CString s;

	ParseHost( ip, s, port );

	if ( port == 0 )
	{
		port = DEFAULT_DC_PORT;
	}

	return Connect( s, port, sockettype );
}

/** */
int CConnection::Connect( CString ip, int port, eSocketType sockettype )
{
	m_pConnMutex->Lock();

	m_sIP   = ip;
	m_nPort = port;

	SocketType = sockettype;

	m_pConnMutex->UnLock();

	return Connect();
}

/** */
int CConnection::Disconnect( bool force )
{
	int err = -1;

	if ( force )
	{
		err = 0;

		m_bForceDisconnect = TRUE;
	}
	else
	{
		m_pConnMutex->Lock();

		if ( m_eState != estNONE )
		{
			m_eState = estDISCONNECTING;

			err = 0;
		}

		m_pConnMutex->UnLock();
	}

	return err;
}

/** */
int CConnection::SetSocket( int handle, eSocketType sockettype )
{
	CString ip;
	int port;

	if ( m_eState != estNONE )
	{
		return -1;
	}

	m_pConnMutex->Lock();

	m_sIP   = "";
	m_nPort = 0;

	if ( CSocket::SetSocket(handle,sockettype) == -1 )
	{
		m_pConnMutex->UnLock();
		return -1;
	}

	// get remote host & port
	if ( GetPeerName( &ip, &port ) == FALSE )
	{
		m_pConnMutex->UnLock();
		return -1;
	}

	// set remote host & port
	SetHost( ip, port );

	m_bForceDisconnect = FALSE;

	// init data timeout
	m_timeConnection = time(0);
	// init notify timer
	m_timeNotify = time(0);

	m_eState = estCONNECTED;

	connectionState(estCONNECTED);

	m_pConnMutex->UnLock();

	return 0;
}

/** */
CString CConnection::GetHost( bool peername )
{
	CString s,host;
	int port;

	if ( peername )
	{
		if ( !GetPeerName(&host,&port) )
		{
			return s;
		}
	}
	else
	{
		port = GetPort();
		host = GetIP();
	}

	s = host + ':' + CString().setNum(port);

	return s;
}

/** */
bool CConnection::ChangeSocketMode( eSocketMode mode, CString cert, CString key )
{
	bool res = FALSE;

	m_pConnMutex->Lock();

	if ( m_eState == estCONNECTED )
	{
		// flush all data
		StateSend();

		if ( m_eState == estCONNECTED )
		{
			res = CSocket::ChangeSocketMode(mode, cert, key );

			if ( (res == TRUE) && (mode != esmSOCKET) )
			{
				m_eState = estCONNECTING;
			}
		}
	}

	m_pConnMutex->UnLock();

	return res;
}

/** */
bool CConnection::IsSendQueueEmpty()
{
	bool res = TRUE;

	if ( m_pSendList != 0 )
	{
		m_pSendList->Lock();

		res = (m_pSendList->Count() == 0);

		m_pSendList->UnLock();
	}

	return res;
}

/** */
void CConnection::Thread( CObject * )
{
	int wait = 4;

	m_pConnMutex->Lock();

	if (m_bForceDisconnect)
	{
		// first we flush all messages
		if ( m_eState == estCONNECTED )
			StateSend();
		// set disconnect state
		if ( m_eState != estNONE )
			m_eState = estDISCONNECTING;
		m_bForceDisconnect = FALSE;
	}

	switch(m_eState)
	{
		case estNONE:
			break;
		case estCONNECT:
			StateConnect();
			if ( m_eState == estCONNECT )
				wait = 100;
			break;
		case estCONNECTING:
			StateConnecting();
			break;
		case estCONNECTED:
			// read data
			StateRead();
			// send data
			if ( m_eState == estCONNECTED )
				StateSend();
			// send data from higher level
			if ( m_eState == estCONNECTED )
				DataSend();
			// check data timeout
			if ( m_eState == estCONNECTED )
			{
				if ( (time(0)-m_timeConnection) >= DATA_TIMEOUT )
				{
					DataTimeout();
					m_timeConnection = time(0);
				}
			}
			break;
		case estDISCONNECTING:
			StateDisconnect();
			break;
		case estDISCONNECTED:
			break;
		default:
			break;
	}

	if ( (time(0)-m_timeNotify) >= NOTIFY_TIMEOUT )
	{
		m_pConnMutex->UnLock();
		Notify();
		m_pConnMutex->Lock();
		m_timeNotify = time(0);
	}

	m_pConnMutex->UnLock();

	if ( iRun == 1 )
	{
		NanoSleep(wait);
	}
}

/** */
void CConnection::connectionState( eConnectionState e )
{
	m_pConnMutex->UnLock();

	ConnectionState(e);

	m_pConnMutex->Lock();
}

/** */
void CConnection::StateConnect()
{
	eConnectState ecs;

	m_timeConnection = time(0);

	ecs = CSocket::Connect( m_sIP, m_nPort, TRUE );

	if ( ecs == ecsERROR )
	{
		m_eState = estDISCONNECTING;

		connectionState(estSOCKETERROR);
	}
	else if ( ecs == ecsSUCCESS )
	{
		m_eState  = estCONNECTING;
	}
}

/** */
void CConnection::StateConnecting()
{
	int err=0;

	err = CSocket::IsConnect();

	if ( err < 0 )
	{
		m_eState = estDISCONNECTING;

		connectionState(estSOCKETERROR);
	}
	else if ( err == 1 )
	{
		// init data timeout
		m_timeConnection = time(0);
		// init notify timer
		m_timeNotify = time(0);

		m_eState = estCONNECTED;

		if ( m_eSocketMode == esmSOCKET )
			connectionState(estCONNECTED);
		else
			connectionState(estSSLCONNECTED);
	}
	// check connection timeout
	else if ( (time(0) - m_timeConnection) >= m_nConnectTimeout )
	{
		m_eState = estDISCONNECTING;

		connectionState(estCONNECTIONTIMEOUT);
	}
}

/** */
void CConnection::StateRead()
{
	int len=1;
	int i;

	if ( m_pBuffer == 0 )
	{
		return;
	}

	for (i=0;(i<25)&&(m_eState==estCONNECTED)&&(m_bForceDisconnect==FALSE)&&(len>0);i++)
	{
		len = CSocket::Read((char*)m_pBuffer->Data(),BUFFER_SIZE-1,0,1);

		if ( len < 0 )
		{
			m_eState = estDISCONNECTING;

			connectionState(estSOCKETERROR);

			break;
		}
		else if ( len > 0 )
		{
			// update timeout
			m_timeConnection = time(0);

			// fix end
			m_pBuffer->Data()[len] = 0;

			// hack
			m_pConnMutex->UnLock();
			DataAvailable((char*)m_pBuffer->Data(),len);
			m_pConnMutex->Lock();
		}
		else
		{
			break;
		}
	}
}

/** */
void CConnection::StateSend()
{
	int err = 0;
	CByteArray *s,*s1;
	CObject *obj=0,*obj1;

	if ( m_pSendList != 0 )
	{
		m_pSendList->Lock();

		// build one big package
		if ( (obj=m_pSendList->Next(0)) != 0 )
		{
			s = ((CByteArray*)obj);

			while( (s->Size() < 1024) && ((obj1=m_pSendList->Next(obj)) != 0) )
			{
				s1 = ((CByteArray*)obj1);
				s->Append(s1->Data(),s1->Size());
				m_pSendList->Del(obj1);
			}
		}

		if ( obj )
		{
			s = ((CByteArray*)obj);

			if ( s->Size() > 0 ) 
			{
				err = CSocket::Write( s->Data(), s->Size(), 0, 1 );

				if ( (err > 0) && (err != s->Size()) )
				{
					// TODO: make this better ;-)
					CByteArray b;
					printf("CConnection: warning send %d %ld\n",err,s->Size());
					b.SetSize(0);
					b.Append(s->Data()+err,s->Size()-err);
					s->SetSize(0);
					s->Append(b.Data(),b.Size());
					// add traffic control
					CSocket::m_Traffic.AddTraffic(ettCONTROLTX,err);
					// update timeout
					m_timeConnection = time(0);
				}
				else if ( err == s->Size() )
				{
					// remove object from the list
					m_pSendList->Del(obj);
					// add traffic control
					CSocket::m_Traffic.AddTraffic(ettCONTROLTX,err);
					// update timeout
					m_timeConnection = time(0);
				}
			}
		}

		m_pSendList->UnLock();
	}

	if ( err == -1 )
	{
		m_eState = estDISCONNECTING;

		connectionState(estSOCKETERROR);
	}
}

/** */
void CConnection::StateDisconnect()
{
	CObject *obj;

	CSocket::Disconnect();

	if (m_pSendList!=0)
	{
		m_pSendList->Lock();

		while( (obj=m_pSendList->Next(0)) != 0 )
			m_pSendList->Del(obj);

		m_pSendList->UnLock();
	}

	m_eState = estNONE;

	connectionState(estDISCONNECTED);
}

/** */
int CConnection::Write( const unsigned char * buffer, int len, bool direct )
{
	int err = 0;

	if ( !direct )
	{
		if ( (m_pSendList != 0) &&
	    	    ((m_eState == estCONNECTED) ||
	     	     (m_eState == estCONNECTING)) )
		{
			m_pSendList->Lock();

			if ( len > 0 )
			{
				CByteArray *s = new CByteArray();

				s->Append(buffer,len);

				m_pSendList->Add((CObject*)s);
			}

			m_pSendList->UnLock();
		}
	}
	else
	{
		err = CSocket::Write( buffer, len, 0, 1 );
	}

	if ( err == -1 )
	{
		m_eState = estDISCONNECTING;

		ConnectionState(estSOCKETERROR);
	}

	return err;
}
