/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <libdrizzle/drizzle_client.h>
#include <libdrizzle/drizzle_server.h>

#include "common.h"

#define DRIZZLE_RETURN_CHECK(__ret, __function, __drizzle) { \
  if ((__ret) != DRIZZLE_RETURN_OK) \
  { \
    printf(__function ":%s\n", drizzle_error(__drizzle)); \
    return; \
  } }

static void proxy(drizzle_st *drizzle, drizzle_con_st *client,
                  drizzle_con_st *server, drizzle_result_st *client_result,
                  drizzle_result_st *server_result, drizzle_column_st *column);

static void usage(char *name);

int main(int argc, char *argv[])
{
  uint32_t count= 0;
  bool mysql_client= false;
  bool mysql_server= false;
  int c;
  int listen_fd;
  char *server_host;
  in_port_t server_port;
  drizzle_st drizzle;
  int client_fd;
  struct sockaddr_in sa;
  socklen_t sa_len;
  drizzle_return_t ret;
  drizzle_con_st client;
  drizzle_con_st server;
  drizzle_result_st client_result;
  drizzle_result_st server_result;
  drizzle_column_st column;

  while((c = getopt(argc, argv, "c:mM")) != -1)
  {
    switch(c)
    {
    case 'c':
      count= (uint32_t)atoi(optarg);
      break;

    case 'm':
      mysql_client= true;
      break;

    case 'M':
      mysql_server= true;
      break;

    default:
      usage(argv[0]);
      return 1;
    }
  }

  if (argc != (optind + 3))
  {
    usage(argv[0]);
    return 1;
  }

  listen_fd= listen_init((in_port_t)atoi(argv[optind]));
  if (listen_fd == -1)
    return 1;

  server_host= argv[optind + 1];
  server_port= (in_port_t)atoi(argv[optind + 2]);

  if (drizzle_create(&drizzle) == NULL)
  {
    printf("drizzle_create: memory allocation error\n");
    return 1;
  }

  while (1)
  {
    sa_len= sizeof(sa);
    client_fd= accept(listen_fd, (struct sockaddr *)(&sa), &sa_len);
    if (client_fd == -1)
    {
      printf("accept:%d\n", errno);
      return 1;
    }

    if (drizzle_con_create(&drizzle, &client) == NULL)
    {
      printf("drizzle_con_create: memory allocation error\n");
      return 1;
    }

    ret= drizzle_con_set_fd(&client, client_fd);
    if (ret != DRIZZLE_RETURN_OK)
    {
      printf("drizzle_con_connect: %s\n", drizzle_error(&drizzle));
      return 1;
    }

    if (mysql_client)
      drizzle_con_add_options(&client, DRIZZLE_CON_MYSQL);

    if (drizzle_con_create(&drizzle, &server) == NULL)
    {
      printf("drizzle_con_create: memory allocation error\n");
      return 1;
    }

    drizzle_con_add_options(&server,
                            DRIZZLE_CON_RAW_PACKET | DRIZZLE_CON_RAW_SCRAMBLE);
    if (mysql_server)
      drizzle_con_add_options(&server, DRIZZLE_CON_MYSQL);
    drizzle_con_set_tcp(&server, server_host, server_port);

    ret= drizzle_con_connect(&server);
    if (ret != DRIZZLE_RETURN_OK)
    {
      printf("drizzle_con_connect: %s\n", drizzle_error(&drizzle));
      return 1;
    }

    printf("Connect: %s:%u\n", inet_ntoa(sa.sin_addr), ntohs(sa.sin_port));

    proxy(&drizzle, &client, &server, &client_result, &server_result, &column);

    printf("Disconnect\n");

    drizzle_con_free(&server);
    drizzle_con_free(&client);

    if (count > 0)
    {
      count--;

      if (count == 0)
        break;
    }
  }

  drizzle_free(&drizzle);

  return 0;
}

static void proxy(drizzle_st *drizzle, drizzle_con_st *client,
                  drizzle_con_st *server, drizzle_result_st *client_result,
                  drizzle_result_st *server_result, drizzle_column_st *column)
{
  drizzle_return_t ret;
  drizzle_command_t command;
  const uint8_t *data;
  size_t offset;
  size_t size;
  size_t total;
  uint64_t row;
  bool row_break;
  drizzle_field_t field;

  /* Handshake packets. */
  ret= drizzle_server_handshake_read(server);
  DRIZZLE_RETURN_CHECK(ret, "drizzle_server_handshake_read", drizzle)

  drizzle_con_copy_handshake(client, server);

  ret= drizzle_server_handshake_write(client);
  DRIZZLE_RETURN_CHECK(ret, "drizzle_server_handshake_write", drizzle)

  ret= drizzle_client_handshake_read(client);
  DRIZZLE_RETURN_CHECK(ret, "drizzle_client_handshake_read", drizzle)

  drizzle_con_copy_handshake(server, client);

  ret= drizzle_client_handshake_write(server);
  DRIZZLE_RETURN_CHECK(ret, "drizzle_client_handshake_write", drizzle)

  (void)drizzle_result_read(server, server_result, &ret);
  DRIZZLE_RETURN_CHECK(ret, "drizzle_result_read", drizzle)

  drizzle_con_set_status(client, drizzle_con_status(server));
  (void)drizzle_result_clone(client, client_result, server_result);

  ret= drizzle_result_write(client, client_result, true);
  DRIZZLE_RETURN_CHECK(ret, "drizzle_result_write", drizzle)

  if (drizzle_result_error_code(server_result) != 0 ||
      drizzle_result_eof(server_result))
  {
    /* There was a handshake or authentication error. */
    return;
  }

  drizzle_con_add_options(server, DRIZZLE_CON_READY);

  /* Command loop. */
  while (1)
  {
    drizzle_result_free(client_result);
    drizzle_result_free(server_result);

    while (1)
    {
      data= drizzle_command_read(client, &command, &offset, &size, &total,
                                 &ret);
      if (ret == DRIZZLE_RETURN_LOST_CONNECTION)
        return;

      DRIZZLE_RETURN_CHECK(ret, "drizzle_command_read", drizzle)

      (void)drizzle_command_write(server, NULL, command, data, size, total,
                                  &ret);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_command_write", drizzle)

      if ((offset + size) == total)
        break;
    }

    if (command == DRIZZLE_COMMAND_QUIT)
      return;
    else if (command == DRIZZLE_COMMAND_FIELD_LIST)
    {
      (void)drizzle_result_create(server, server_result);
      (void)drizzle_result_create(client, client_result);
    }
    else
    {
      (void)drizzle_result_read(server, server_result, &ret);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_result_read", drizzle)

      drizzle_con_set_status(client, drizzle_con_status(server));
      (void)drizzle_result_clone(client, client_result, server_result);

      if (drizzle_result_column_count(server_result) == 0)
      {
        /* Simple result with no column, row, or field data. */
        ret= drizzle_result_write(client, client_result, true);
        DRIZZLE_RETURN_CHECK(ret, "drizzle_result_write", drizzle)

        continue;
      }

      ret= drizzle_result_write(client, client_result, false);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_result_write", drizzle)
    }

    /* Columns. */
    while (1)
    {
      if (drizzle_column_read(server_result, column, &ret) == NULL)
      {
        DRIZZLE_RETURN_CHECK(ret, "drizzle_column_read", drizzle)
        break;
      }

      DRIZZLE_RETURN_CHECK(ret, "drizzle_column_read", drizzle)

      ret= drizzle_column_write(client_result, column);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_column_write", drizzle)

      drizzle_column_free(column);
    }

    drizzle_con_set_status(client, drizzle_con_status(server));
    drizzle_result_set_eof(client_result, true);

    if (command == DRIZZLE_COMMAND_FIELD_LIST)
    {
      ret= drizzle_result_write(client, client_result, true);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_result_write", drizzle)
      continue;
    }
    else
    {
      ret= drizzle_result_write(client, client_result, false);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_result_write", drizzle)
    }

    /* Rows. */
    while (1)
    {
      row= drizzle_row_read(server_result, &ret);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_row_read", drizzle)

      if (row == 0)
        break;

      drizzle_result_set_row_size(client_result,
                                  drizzle_result_row_size(server_result));

      ret= drizzle_row_write(client_result);
      DRIZZLE_RETURN_CHECK(ret, "drizzle_row_write", drizzle)

      /* Fields. */
      row_break= false;
      while (row_break == false)
      {
        field= drizzle_field_read(server_result, &offset, &size, &total, &ret);
        if (ret == DRIZZLE_RETURN_ROW_END)
          break;
        else if (ret == DRIZZLE_RETURN_ROW_BREAK)
        {
          if (size == 0)
            break;

          row_break= true;
        }
        else
          DRIZZLE_RETURN_CHECK(ret, "drizzle_field_read", drizzle)

        ret= drizzle_field_write(client_result, field, size, total);
        DRIZZLE_RETURN_CHECK(ret, "drizzle_field_write", drizzle)
      }
    }

    ret= drizzle_result_write(client, client_result, true);
    DRIZZLE_RETURN_CHECK(ret, "drizzle_result_write", drizzle)
  }
}

static void usage(char *name)
{
  printf("\nusage: %s [-c <count>] [-m] [-M] <listen port> <server host> "
         "<server port>\n", name);
  printf("\t-c <count> - number of connections to accept before exiting\n");
  printf("\t-m         - use the MySQL protocol for incoming connections\n");
  printf("\t-M         - use the MySQL protocol for outgoing connectionsn\n");
}
