/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Command definitions
 */

#include "common.h"

/*
 * Private variables.
 */

static drizzle_command_drizzle_t _command_drizzle_map[]=
{
 DRIZZLE_COMMAND_DRIZZLE_SLEEP,
 DRIZZLE_COMMAND_DRIZZLE_QUIT,
 DRIZZLE_COMMAND_DRIZZLE_INIT_DB,
 DRIZZLE_COMMAND_DRIZZLE_QUERY,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_SHUTDOWN,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_CONNECT,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_PING,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END,
 DRIZZLE_COMMAND_DRIZZLE_END
};

/*
 * Client definitions
 */

drizzle_result_st *drizzle_quit(drizzle_con_st *con, drizzle_result_st *result,
                                drizzle_return_t *ret_ptr)
{
  return drizzle_command_write(con, result, DRIZZLE_COMMAND_QUIT, NULL, 0, 0,
                               ret_ptr);
}

drizzle_result_st *drizzle_select_db(drizzle_con_st *con,
                                     drizzle_result_st *result, const char *db,
                                     drizzle_return_t *ret_ptr)
{
  drizzle_con_set_db(con, db);
  return drizzle_command_write(con, result, DRIZZLE_COMMAND_INIT_DB,
                               (uint8_t *)db, strlen(db), strlen(db), ret_ptr);
}

drizzle_result_st *drizzle_refresh(drizzle_con_st *con,
                                   drizzle_result_st *result,
                                   drizzle_refresh_options_t options,
                                   drizzle_return_t *ret_ptr)
{
  return drizzle_command_write(con, result, DRIZZLE_COMMAND_REFRESH,
                               (uint8_t *)&options, 1, 1, ret_ptr);
}

drizzle_result_st *drizzle_shutdown(drizzle_con_st *con,
                                    drizzle_result_st *result,
                                    drizzle_shutdown_level_t level,
                                    drizzle_return_t *ret_ptr)
{
  if (con->options & DRIZZLE_CON_MYSQL)
  {
    return drizzle_command_write(con, result, DRIZZLE_COMMAND_SHUTDOWN,
                                 (uint8_t *)&level, 1, 1, ret_ptr);
  }

  return drizzle_command_write(con, result, DRIZZLE_COMMAND_SHUTDOWN, NULL, 0,
                               0, ret_ptr);
}

drizzle_result_st *drizzle_stat(drizzle_con_st *con, drizzle_result_st *result,
                                drizzle_return_t *ret_ptr)
{
  return drizzle_command_write(con, result, DRIZZLE_COMMAND_STATISTICS, NULL, 0,
                               0, ret_ptr);
}

drizzle_result_st *drizzle_debug_info(drizzle_con_st *con,
                                      drizzle_result_st *result,
                                      drizzle_return_t *ret_ptr)
{
  return drizzle_command_write(con, result, DRIZZLE_COMMAND_DEBUG, NULL, 0, 0,
                               ret_ptr);
}

drizzle_result_st *drizzle_ping(drizzle_con_st *con, drizzle_result_st *result,
                                drizzle_return_t *ret_ptr)
{
  return drizzle_command_write(con, result, DRIZZLE_COMMAND_PING, NULL, 0, 0,
                               ret_ptr);
}

drizzle_result_st *drizzle_change_user(drizzle_con_st *con,
                                       drizzle_result_st *result,
                                       const char *user, const char *password,
                                       const char *db,
                                       drizzle_return_t *ret_ptr)
{
  drizzle_con_set_auth(con, user, password);
  drizzle_con_set_db(con, db);
  return drizzle_command_write(con, result, DRIZZLE_COMMAND_CHANGE_USER, NULL,
                               0, 0, ret_ptr);
}

drizzle_result_st *drizzle_command_write(drizzle_con_st *con,
                                         drizzle_result_st *result,
                                         drizzle_command_t command,
                                         const uint8_t *data, size_t size,
                                         size_t total,
                                         drizzle_return_t *ret_ptr)
{
  if (!(con->options & DRIZZLE_CON_READY))
  {
    if (con->options & DRIZZLE_CON_RAW_PACKET)
    {
      DRIZZLE_ERROR_SET(con->drizzle, "drizzle_command_write",
                        "connection not ready")
      *ret_ptr= DRIZZLE_RETURN_NOT_READY;
      return result;
    }

    *ret_ptr= drizzle_con_connect(con);
    if (*ret_ptr != DRIZZLE_RETURN_OK)
      return result;
  }

  if (DRIZZLE_STATE_NONE(con))
  {
    if (con->options & (DRIZZLE_CON_RAW_PACKET | DRIZZLE_CON_NO_RESULT_READ))
      con->result= NULL;
    else
    {
      con->result= drizzle_result_create(con, result);
      if (con->result == NULL)
      {
        *ret_ptr= DRIZZLE_RETURN_MEMORY;
        return NULL;
      }
    }

    con->command= command;
    con->command_data= (uint8_t *)data;
    con->command_size= size;
    con->command_offset= 0;
    con->command_total= total;

    DRIZZLE_STATE_PUSH(con, drizzle_state_command_write)
  }
  else if (con->command_data == NULL)
  {
    con->command_data= (uint8_t *)data;
    con->command_size= size;
  }

  *ret_ptr= drizzle_state_loop(con);
  if (*ret_ptr == DRIZZLE_RETURN_PAUSE)
    *ret_ptr= DRIZZLE_RETURN_OK;
  else if (*ret_ptr != DRIZZLE_RETURN_OK &&
           *ret_ptr != DRIZZLE_RETURN_IO_WAIT &&
           *ret_ptr != DRIZZLE_RETURN_ERROR_CODE)
  {
    drizzle_result_free(con->result);
    con->result= result;
  }

  return con->result;
}

/*
 * Server definitions
 */

uint8_t *drizzle_command_read(drizzle_con_st *con, drizzle_command_t *command,
                              size_t *offset, size_t *size, size_t *total,
                              drizzle_return_t *ret_ptr)
{
  if (DRIZZLE_STATE_NONE(con))
  {
    con->packet_number= 0;
    con->command_offset= 0;
    con->command_total= 0;

    DRIZZLE_STATE_PUSH(con, drizzle_state_command_read)
    DRIZZLE_STATE_PUSH(con, drizzle_state_packet_read)
  }
  
  *offset= con->command_offset;

  *ret_ptr= drizzle_state_loop(con);
  if (*ret_ptr == DRIZZLE_RETURN_PAUSE)
    *ret_ptr= DRIZZLE_RETURN_OK;

  *command= con->command;
  *size= con->command_size;
  *total= con->command_total;

  return con->command_data;
}

uint8_t *drizzle_command_buffer(drizzle_con_st *con, drizzle_command_t *command,
                                size_t *total, drizzle_return_t *ret_ptr)
{
  uint8_t *command_data;
  size_t offset= 0;
  size_t size= 0;

  command_data= drizzle_command_read(con, command, &offset, &size, total,
                                     ret_ptr);
  if (*ret_ptr != DRIZZLE_RETURN_OK)
    return NULL;

  if (command_data == NULL)
  {
    *total= 0;
    return NULL;
  }

  if (con->command_buffer == NULL)
  {
    con->command_buffer= malloc((*total) + 1);
    if (con->command_buffer == NULL)
    {
      DRIZZLE_ERROR_SET(con->drizzle, "drizzle_command_buffer", "malloc");
      *ret_ptr= DRIZZLE_RETURN_MEMORY;
      return NULL;
    }
  }

  memcpy(con->command_buffer + offset, command_data, size);

  while ((offset + size) != (*total))
  {
    command_data= drizzle_command_read(con, command, &offset, &size, total,
                                       ret_ptr);
    if (*ret_ptr != DRIZZLE_RETURN_OK)
      return NULL;

    memcpy(con->command_buffer + offset, command_data, size);
  }

  command_data= con->command_buffer;
  con->command_buffer= NULL;
  command_data[*total]= 0;

  return command_data;
}

/*
 * Internal state functions.
 */

drizzle_return_t drizzle_state_command_read(drizzle_con_st *con)
{
  PDEBUG("drizzle_state_command_read", "%5zu %5zu", con->buffer_size,
         con->packet_size)

  if (con->buffer_size == 0)
  {
    DRIZZLE_STATE_PUSH(con, drizzle_state_read)
    return DRIZZLE_RETURN_OK;
  }

  if (con->command_total == 0)
  {
    con->command= (drizzle_command_t)(con->buffer_ptr[0]);
    con->buffer_ptr++;
    con->buffer_size--;

    con->command_total= (con->packet_size - 1);
  }

  if (con->buffer_size < (con->command_total - con->command_offset))
  {
    con->command_size= con->buffer_size;
    con->command_offset+= con->command_size;
  }
  else
  {
    con->command_size= (con->command_total - con->command_offset);
    con->command_offset= con->command_total;
  }

  con->command_data= con->buffer_ptr;
  con->buffer_ptr+= con->command_size;
  con->buffer_size-= con->command_size;

  if (con->command_offset == con->command_total)
    DRIZZLE_STATE_POP(con)
  else
    return DRIZZLE_RETURN_PAUSE;

  return DRIZZLE_RETURN_OK;
}

drizzle_return_t drizzle_state_command_write(drizzle_con_st *con)
{
  uint8_t *start;
  uint8_t *ptr;
  size_t free_size;
  drizzle_return_t ret;

  if (con->command_data == NULL && con->command_total != 0 &&
      con->command != DRIZZLE_COMMAND_CHANGE_USER)
  {
    return DRIZZLE_RETURN_PAUSE;
  }

  if (con->buffer_size == 0)
  {
    con->buffer_ptr= con->buffer;
    start= con->buffer;
  }
  else
    start= con->buffer_ptr + con->buffer_size;

  if (con->command_offset == 0)
  {
    /* Make sure we can fit the largest non-streaming packet, currently a
       DRIZZLE_COMMAND_CHANGE_USER command. */

    con->packet_size= 1  /* Command */
                    + strlen(con->user) + 1
                    + 1  /* Scramble size */
                    + DRIZZLE_MAX_SCRAMBLE_SIZE
                    + strlen(con->db) + 1;

    /* Flush buffer if there is not enough room. */
    free_size= (size_t)DRIZZLE_MAX_BUFFER_SIZE - (size_t)(start - con->buffer);
    if (free_size < con->packet_size)
    {
      DRIZZLE_STATE_PUSH(con, drizzle_state_write)
      return DRIZZLE_RETURN_OK;
    }

    /* Store packet size at the end since it may change. */
    con->packet_number= 1;
    ptr= start;
    ptr[3]= 0;
    if (con->options & DRIZZLE_CON_MYSQL)
      ptr[4]= (uint8_t)(con->command);
    else
      ptr[4]= (uint8_t)(_command_drizzle_map[con->command]);
    ptr+= 5;

    if (con->command == DRIZZLE_COMMAND_CHANGE_USER)
    {
      ptr= drizzle_pack_auth(con, ptr, &ret);
      if (ret != DRIZZLE_RETURN_OK)
        return ret;

      con->buffer_size+= (4 + con->packet_size);
    }
    else if (con->command_total == 0)
    {
      con->packet_size= 1;
      con->buffer_size+= 5;
    }
    else
    {
      con->packet_size= 1 + con->command_total;
      free_size-= 5;

      /* Copy as much of the data in as we can into the write buffer. */
      if (con->command_size <= free_size)
      {
        memcpy(ptr, con->command_data, con->command_size);
        con->command_offset= con->command_size;
        con->command_data= NULL;
        con->buffer_size+= 5 + con->command_size;
      }
      else
      {
        memcpy(ptr, con->command_data, free_size);
        con->command_offset= free_size;
        con->command_data+= free_size;
        con->command_size-= free_size;
        con->buffer_size+= 5 + free_size;
      }
    }

    /* Store packet size now. */
    DRIZZLE_SET_BYTE3(start, con->packet_size)
  }
  else
  {
    /* Write directly from the caller buffer for the rest. */
    con->buffer_ptr= (uint8_t *)con->command_data;
    con->buffer_size= con->command_size;
    con->command_offset+= con->command_size;
    con->command_data= NULL;
  }

  if (con->command_offset == con->command_total)
  {
    DRIZZLE_STATE_POP(con)

    if (!(con->options & (DRIZZLE_CON_RAW_PACKET |
                          DRIZZLE_CON_NO_RESULT_READ)) &&
        con->command != DRIZZLE_COMMAND_FIELD_LIST)
    {
      DRIZZLE_STATE_PUSH(con, drizzle_state_result_read)
      DRIZZLE_STATE_PUSH(con, drizzle_state_packet_read)
    }
  }

  DRIZZLE_STATE_PUSH(con, drizzle_state_write)

  return DRIZZLE_RETURN_OK;
}
