/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Connection definitions
 */

#include "common.h"

 /*
 * Private declarations
 */

/**
 * @addtogroup drizzle_con_private Private Connection Functions
 * @ingroup drizzle_con
 * @{
 */

/**
 * Set socket options for a connection.
 */
static drizzle_return_t _con_setsockopt(drizzle_con_st *con);

/** @} */

/*
 * Common definitions
 */

drizzle_con_st *drizzle_con_create(drizzle_st *drizzle, drizzle_con_st *con)
{
  if (con == NULL)
  {
    con= malloc(sizeof(drizzle_con_st));
    if (con == NULL)
    {
      if (drizzle != NULL)
        DRIZZLE_ERROR_SET(drizzle, "drizzle_con_create", "malloc")
      return NULL;
    }

    con->options= DRIZZLE_CON_ALLOCATED;
  }
  else
    con->options= 0;

  if (drizzle == NULL)
  {
    drizzle= drizzle_create(NULL);
    if (drizzle == NULL)
    {
      if (con->options & DRIZZLE_CON_ALLOCATED)
        free(con);
      return NULL;
    }

    drizzle->options|= DRIZZLE_AUTO_ALLOCATED;
  }

  con->drizzle= drizzle;

  if (drizzle->con_list)
    drizzle->con_list->prev= con;
  con->next= drizzle->con_list;
  con->prev= NULL;
  drizzle->con_list= con;
  drizzle->con_count++;

  con->result_list= NULL;
  con->result_count= 0;
  con->data= NULL;
  con->data_free_fn= NULL;
  con->query= NULL;
  strncpy(con->user, DRIZZLE_DEFAULT_USER, DRIZZLE_MAX_USER_SIZE);
  con->user[DRIZZLE_MAX_USER_SIZE - 1]= 0;
  con->password[0]= 0;
  con->db[0]= 0;
  con->host[0]= 0;
  con->port= 0;
  con->tcp_addrinfo= NULL;
  con->uds_addrinfo.ai_addr= NULL;
  con->addrinfo_next= NULL;
  drizzle_con_set_tcp(con, NULL, 0);
  DRIZZLE_STATE_RESET(con)
  con->fd= -1;
  con->events= 0;
  con->revents= 0;
  con->buffer_ptr= con->buffer;
  con->buffer_size= 0;
  con->packet_size= 0;
  con->packet_number= 0;
  con->protocol_version= 0;
  con->server_version[0]= 0;
  con->thread_id= 0;
  con->scramble= NULL;
  con->capabilities= DRIZZLE_CAPABILITIES_NONE;
  con->charset= 0;
  con->status= DRIZZLE_CON_STATUS_NONE;
  con->max_packet_size= DRIZZLE_MAX_PACKET_SIZE;
  con->command= 0;
  con->command_data= NULL;
  con->command_buffer= NULL;
  con->command_size= 0;
  con->command_offset= 0;
  con->command_total= 0;

  return con;
}

drizzle_con_st *drizzle_con_clone(drizzle_st *drizzle, drizzle_con_st *con,
                                  drizzle_con_st *from)
{
  if (drizzle == NULL)
    drizzle= from->drizzle;

  con= drizzle_con_create(drizzle, con);
  if (con == NULL)
    return NULL;

  con->options|= (from->options &
                  (drizzle_con_options_t)~DRIZZLE_CON_ALLOCATED);
  strcpy(con->user, from->user);
  strcpy(con->password, from->password);
  strcpy(con->db, from->db);

  if (drizzle_con_uses_uds(con))
    drizzle_con_clone_uds(con, from);
  else
  {
    strcpy(con->host, from->host);
    con->port= from->port;
  }

  return con;
}

void drizzle_con_free(drizzle_con_st *con)
{
  drizzle_result_st *result;

  if (con->data != NULL && con->data_free_fn != NULL)
    (*(con->data_free_fn))(con, (void *)(con->data));

  for (result= con->result_list; result != NULL; result= con->result_list)
    drizzle_result_free(result);

  if (con->fd != -1)
    drizzle_con_close(con);

  drizzle_con_reset_addrinfo(con);

  if (con->drizzle->con_list == con)
    con->drizzle->con_list= con->next;
  if (con->prev)
    con->prev->next= con->next;
  if (con->next)
    con->next->prev= con->prev;
  con->drizzle->con_count--;

  if (con->drizzle->con_count == 0 &&
      con->drizzle->options & DRIZZLE_AUTO_ALLOCATED)
  {
    drizzle_free(con->drizzle);
  }

  if (con->options & DRIZZLE_CON_ALLOCATED)
    free(con);
}

int drizzle_con_fd(drizzle_con_st *con)
{
  return con->fd;
}

drizzle_return_t drizzle_con_set_fd(drizzle_con_st *con, int fd)
{
  drizzle_return_t ret;

  con->fd= fd;

  ret= _con_setsockopt(con);
  if (ret != DRIZZLE_RETURN_OK)
    con->drizzle->last_errno= errno;

  return ret;
}

void drizzle_con_close(drizzle_con_st *con)
{
  if (con->fd == -1)
    return;

  (void)close(con->fd);
  con->fd= -1;

  con->options&= (drizzle_con_options_t)~DRIZZLE_CON_READY;
  con->packet_number= 0;
  con->buffer_ptr= con->buffer;
  con->buffer_size= 0;
  con->events= 0;
  con->revents= 0;

  DRIZZLE_STATE_RESET(con)
}

drizzle_return_t drizzle_con_wait(drizzle_st *drizzle)
{
  drizzle_con_st *con;
  struct pollfd *pfds;
  uint32_t x;
  int ret;

  if (drizzle->pfds_size < drizzle->con_count)
  {
    pfds= realloc(drizzle->pfds, drizzle->con_count * sizeof(struct pollfd));
    if (pfds == NULL)
    {
      DRIZZLE_ERROR_SET(drizzle, "drizzle_con_wait", "realloc")
      return DRIZZLE_RETURN_MEMORY;
    }

    drizzle->pfds= pfds;
    drizzle->pfds_size= drizzle->con_count;
  }
  else
    pfds= drizzle->pfds;

  x= 0;
  for (con= drizzle->con_list; con != NULL; con= con->next)
  {
    if (con->events == 0)
      continue;

    pfds[x].fd= con->fd;
    pfds[x].events= con->events;
    pfds[x].revents= 0;
    x++;
  }

  if (x == 0)
  {
    DRIZZLE_ERROR_SET(drizzle, "drizzle_con_wait", "no active file descriptors")
    return DRIZZLE_RETURN_NO_ACTIVE_CONNECTIONS;
  }

  while (1)
  {
    PDEBUG("drizzle_con_wait", "%5d", x)
    ret= poll(pfds, x, -1);
    PDEBUG("drizzle_con_wait", "%5d %5d", ret, errno)
    if (ret == -1)
    {
      if (errno == EINTR)
        continue;

      DRIZZLE_ERROR_SET(drizzle, "drizzle_con_wait", "poll:%d", errno)
      drizzle->last_errno= errno;
      return DRIZZLE_RETURN_ERRNO;
    }

    break;
  }

  x= 0;
  for (con= drizzle->con_list; con != NULL; con= con->next)
  {
    if (con->events == 0)
      continue;

    drizzle_con_set_revents(con, pfds[x].revents);
    x++;
  }

  return DRIZZLE_RETURN_OK;
}

drizzle_return_t drizzle_con_set_events(drizzle_con_st *con, short events)
{
  drizzle_return_t ret;

  if ((con->events | events) == con->events)
    return DRIZZLE_RETURN_OK;

  con->events|= events;

  if (con->drizzle->event_watch != NULL)
  { 
    ret= (con->drizzle->event_watch)(con, con->events,
                                     con->drizzle->event_watch_arg);
    if (ret != DRIZZLE_RETURN_OK)
    {
      drizzle_con_close(con);
      return ret;
    }
  }

  return DRIZZLE_RETURN_OK;
}

void drizzle_con_set_revents(drizzle_con_st *con, short revents)
{
  if (revents != 0)
    con->options|= DRIZZLE_CON_IO_READY;

  con->revents= revents;
  con->events&= (short)~revents;
}

drizzle_con_st *drizzle_con_ready(drizzle_st *drizzle)
{
  drizzle_con_st *con;

  /* We can't keep state between calls since connections may be removed during
     processing. If this list ever gets big, we may want something faster. */

  for (con= drizzle->con_list; con != NULL; con= con->next)
  { 
    if (con->options & DRIZZLE_CON_IO_READY)
    {
      con->options&= (drizzle_con_options_t)~DRIZZLE_CON_IO_READY;
      return con;
    }
  }
  
  return NULL;
}

drizzle_st *drizzle_con_drizzle(drizzle_con_st *con)
{
  return con->drizzle;
}

const char *drizzle_con_error(drizzle_con_st *con)
{
  return drizzle_error(con->drizzle);
}

int drizzle_con_errno(drizzle_con_st *con)
{
  return drizzle_errno(con->drizzle);
}

uint16_t drizzle_con_error_code(drizzle_con_st *con)
{
  return drizzle_error_code(con->drizzle);
}

const char *drizzle_con_sqlstate(drizzle_con_st *con)
{
  return drizzle_sqlstate(con->drizzle);
}

drizzle_con_options_t drizzle_con_options(drizzle_con_st *con)
{
  return con->options;
}

void drizzle_con_set_options(drizzle_con_st *con,
                             drizzle_con_options_t options)
{
  con->options= options;
}

void drizzle_con_add_options(drizzle_con_st *con,
                             drizzle_con_options_t options)
{
  con->options|= options;
}

void drizzle_con_remove_options(drizzle_con_st *con,
                                drizzle_con_options_t options)
{
  con->options&= ~options;
}

const char *drizzle_con_host(drizzle_con_st *con)
{
  return con->host;
}

in_port_t drizzle_con_port(drizzle_con_st *con)
{
  if (con->port != 0)
    return con->port;

  if (con->options & DRIZZLE_CON_MYSQL)
    return DRIZZLE_DEFAULT_TCP_PORT_MYSQL;

  return DRIZZLE_DEFAULT_TCP_PORT;
}

void drizzle_con_set_tcp(drizzle_con_st *con, const char *host, in_port_t port)
{
  drizzle_con_reset_addrinfo(con);

  if (host == NULL || host[0] == 0)
    host= DRIZZLE_DEFAULT_TCP_HOST;

  strncpy(con->host, host, NI_MAXHOST);
  con->host[NI_MAXHOST - 1]= 0;

  con->port= port;
}


const char *drizzle_con_user(drizzle_con_st *con)
{
  return con->user;
}

const char *drizzle_con_password(drizzle_con_st *con)
{
  return con->password;
}

void drizzle_con_set_auth(drizzle_con_st *con, const char *user,
                          const char *password)
{
  if (user == NULL)
    strncpy(con->user, DRIZZLE_DEFAULT_USER, DRIZZLE_MAX_USER_SIZE);
  else
    strncpy(con->user, user, DRIZZLE_MAX_USER_SIZE);
  con->user[DRIZZLE_MAX_USER_SIZE - 1]= 0;

  if (password == NULL)
    con->password[0]= 0;
  else
  {
    strncpy(con->password, password, DRIZZLE_MAX_PASSWORD_SIZE);
    con->password[DRIZZLE_MAX_PASSWORD_SIZE - 1]= 0;
  }
}

const char *drizzle_con_db(drizzle_con_st *con)
{
  return con->db;
}

void drizzle_con_set_db(drizzle_con_st *con, const char *db)
{
  if (db == NULL)
    con->db[0]= 0;
  else
  {
    strncpy(con->db, db, DRIZZLE_MAX_DB_SIZE);
    con->db[DRIZZLE_MAX_DB_SIZE - 1]= 0;
  }
}

#if 0
const char *drizzle_con_ssl_cipher(drizzle_con_st *con)
{
  return "";
}

drizzle_return_t drizzle_con_ssl(drizzle_con_st *con, char *key, char *cert,
                                 char *ca_file, char *ca_path, char *cipher)
{
  return DRIZZLE_RETURN_OK;
}
#endif

void *drizzle_con_data(drizzle_con_st *con)
{
  return (void *)(con->data);
}

void drizzle_con_set_data(drizzle_con_st *con, const void *data)
{
  con->data= data;
}

void drizzle_con_set_data_free(drizzle_con_st *con,
                               drizzle_con_data_free_fn *free_fn)
{
  con->data_free_fn= free_fn;
}

uint8_t drizzle_con_protocol_version(drizzle_con_st *con)
{
  return con->protocol_version;
}

const char *drizzle_con_server_version(drizzle_con_st *con)
{
  return con->server_version;
}

uint32_t drizzle_con_server_version_number(drizzle_con_st *con)
{
  uint32_t major;
  uint32_t minor;
  uint32_t version;
  char *current;
  char *end;

  current= con->server_version;

  major= (uint32_t)strtoul(current, &end, 10);
  current= end + 1;
  minor= (uint32_t)strtoul(current, &end, 10);
  current= end + 1;
  version= (uint32_t)strtoul(current, &end, 10);

  return (major * 10000) + (minor * 100) + version;
}

uint32_t drizzle_con_thread_id(drizzle_con_st *con)
{
  return con->thread_id;
}

const uint8_t *drizzle_con_scramble(drizzle_con_st *con)
{
  return con->scramble;
}

drizzle_capabilities_t drizzle_con_capabilities(drizzle_con_st *con)
{
  return con->capabilities;
}

uint8_t drizzle_con_charset(drizzle_con_st *con)
{
  return con->charset;
}

drizzle_con_status_t drizzle_con_status(drizzle_con_st *con)
{
  return con->status;
}

uint32_t drizzle_con_max_packet_size(drizzle_con_st *con)
{
  return con->max_packet_size;
}

/*
 * Client definitions
 */

drizzle_con_st *drizzle_con_add_tcp(drizzle_st *drizzle, drizzle_con_st *con,
                                    const char *host, in_port_t port,
                                    const char *user, const char *password,
                                    const char *db,
                                    drizzle_con_options_t options)
{
  con= drizzle_con_create(drizzle, con);
  if (con == NULL)
    return NULL;

  drizzle_con_set_tcp(con, host, port);
  drizzle_con_set_auth(con, user, password);
  drizzle_con_set_db(con, db);
  drizzle_con_add_options(con, options);

  return con;
}

drizzle_con_st *drizzle_con_add_uds(drizzle_st *drizzle, drizzle_con_st *con,
                                    const char *uds, const char *user,
                                    const char *password, const char *db,
                                    drizzle_con_options_t options)
{
  con= drizzle_con_create(drizzle, con);
  if (con == NULL)
    return NULL;

  drizzle_con_set_uds(con, uds);
  drizzle_con_set_auth(con, user, password);
  drizzle_con_set_db(con, db);
  drizzle_con_add_options(con, options);

  return con;
}

drizzle_return_t drizzle_con_connect(drizzle_con_st *con)
{
  if (con->options & DRIZZLE_CON_READY)
    return DRIZZLE_RETURN_OK;

  if (DRIZZLE_STATE_NONE(con))
  {
    if (!(con->options & DRIZZLE_CON_RAW_PACKET))
    {
      DRIZZLE_STATE_PUSH(con, drizzle_state_server_handshake_read)
      DRIZZLE_STATE_PUSH(con, drizzle_state_packet_read)
    }

    DRIZZLE_STATE_PUSH(con, drizzle_state_connect)
    DRIZZLE_STATE_PUSH(con, drizzle_state_addrinfo)
  }

  return drizzle_state_loop(con);
}

/*
 * Server definitions
 */

void drizzle_con_set_protocol_version(drizzle_con_st *con,
                                      uint8_t protocol_version)
{
  con->protocol_version= protocol_version;
}

void drizzle_con_set_server_version(drizzle_con_st *con,
                                    const char *server_version)
{
  if (server_version == NULL)
    con->server_version[0]= 0;
  else
  {
    strncpy(con->server_version, server_version,
            DRIZZLE_MAX_SERVER_VERSION_SIZE);
    con->server_version[DRIZZLE_MAX_SERVER_VERSION_SIZE - 1]= 0;
  }
}

void drizzle_con_set_thread_id(drizzle_con_st *con, uint32_t thread_id)
{
  con->thread_id= thread_id;
}

void drizzle_con_set_scramble(drizzle_con_st *con, const uint8_t *scramble)
{
  if (scramble == NULL)
    con->scramble= NULL;
  else
  {
    con->scramble= con->scramble_buffer;
    memcpy(con->scramble, scramble, DRIZZLE_MAX_SCRAMBLE_SIZE);
  }
}

void drizzle_con_set_capabilities(drizzle_con_st *con,
                                  drizzle_capabilities_t capabilities)
{
  con->capabilities= capabilities;
}

void drizzle_con_set_charset(drizzle_con_st *con, drizzle_charset_t charset)
{
  con->charset= charset;
}

void drizzle_con_set_status(drizzle_con_st *con, drizzle_con_status_t status)
{
  con->status= status;
}

void drizzle_con_set_max_packet_size(drizzle_con_st *con,
                                     uint32_t max_packet_size)
{
  con->max_packet_size= max_packet_size;
}

void drizzle_con_copy_handshake(drizzle_con_st *con, drizzle_con_st *from)
{
  drizzle_con_set_auth(con, from->user, NULL);
  drizzle_con_set_scramble(con, from->scramble);
  drizzle_con_set_db(con, from->db);
  drizzle_con_set_protocol_version(con, from->protocol_version);
  drizzle_con_set_server_version(con, from->server_version);
  drizzle_con_set_thread_id(con, from->thread_id);
  drizzle_con_set_scramble(con, from->scramble);
  drizzle_con_set_capabilities(con, from->capabilities);
  drizzle_con_set_charset(con, from->charset);
  drizzle_con_set_status(con, from->status);
  drizzle_con_set_max_packet_size(con, from->max_packet_size);
}

/*
 * Internal state functions.
 */

drizzle_return_t drizzle_state_addrinfo(drizzle_con_st *con)
{
  char port_str[NI_MAXSERV];
  struct addrinfo ai;
  int ret;

  if (drizzle_con_uses_uds(con))
    con->addrinfo_next= &(con->uds_addrinfo);
  else
  {
    if (con->tcp_addrinfo != NULL)
    {
      freeaddrinfo(con->tcp_addrinfo);
      con->tcp_addrinfo= NULL;
    }

    if (con->port != 0)
      snprintf(port_str, NI_MAXSERV, "%u", con->port);
    else if (con->options & DRIZZLE_CON_MYSQL)
      snprintf(port_str, NI_MAXSERV, "%u", DRIZZLE_DEFAULT_TCP_PORT_MYSQL);
    else
      snprintf(port_str, NI_MAXSERV, "%u", DRIZZLE_DEFAULT_TCP_PORT);

    memset(&ai, 0, sizeof(struct addrinfo));
    ai.ai_flags= (AI_V4MAPPED | AI_ADDRCONFIG);
    ai.ai_family= AF_UNSPEC;
    ai.ai_socktype= SOCK_STREAM;
    ai.ai_protocol= IPPROTO_TCP;

    ret= getaddrinfo(con->host, port_str, &ai, &(con->tcp_addrinfo));
    if (ret != 0)
    {
      DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_addrinfo",
                        "getaddrinfo:%s", gai_strerror(ret))
      return DRIZZLE_RETURN_GETADDRINFO;
    }

    con->addrinfo_next= con->tcp_addrinfo;
  }

  DRIZZLE_STATE_POP(con)
  return DRIZZLE_RETURN_OK;
}

drizzle_return_t drizzle_state_connect(drizzle_con_st *con)
{
  int ret;
  drizzle_return_t dret;

  PDEBUG("drizzle_state_connect", "%5d", con->fd)

  if (con->fd != -1)
  {
    (void)close(con->fd);
    con->fd= -1;
  }

  if (con->addrinfo_next == NULL)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_connect",
                      "could not connect")
    DRIZZLE_STATE_RESET(con)
    return DRIZZLE_RETURN_COULD_NOT_CONNECT;
  }

  con->fd= socket(con->addrinfo_next->ai_family,
                  con->addrinfo_next->ai_socktype,
                  con->addrinfo_next->ai_protocol);
  if (con->fd == -1)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_connect", "socket:%d", errno)
    con->drizzle->last_errno= errno;
    return DRIZZLE_RETURN_ERRNO;
  }

  dret= _con_setsockopt(con);
  if (dret != DRIZZLE_RETURN_OK)
  {
    con->drizzle->last_errno= errno;
    return dret;
  }

  while (1)
  {
    ret= connect(con->fd, con->addrinfo_next->ai_addr,
                 con->addrinfo_next->ai_addrlen);
    PDEBUG("drizzle_state_connect", "%5d %5d", ret, errno)
    if (ret == 0)
    {
      con->addrinfo_next= NULL;
      break;
    }

    if (errno == EAGAIN || errno == EINTR)
      continue;

    if (errno == EINPROGRESS)
    {
      DRIZZLE_STATE_POP(con)
      DRIZZLE_STATE_PUSH(con, drizzle_state_connecting)
      return DRIZZLE_RETURN_OK;
    }

    if (errno == ECONNREFUSED || errno == ENETUNREACH || errno == ETIMEDOUT)
    {
      con->addrinfo_next= con->addrinfo_next->ai_next;
      return DRIZZLE_RETURN_OK;
    }

    DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_connect", "connect:%d",
                      errno)
    con->drizzle->last_errno= errno;
    return DRIZZLE_RETURN_ERRNO;
  }

  DRIZZLE_STATE_POP(con)
  return DRIZZLE_RETURN_OK;
}

drizzle_return_t drizzle_state_connecting(drizzle_con_st *con)
{
  drizzle_return_t ret;

  PDEBUG("drizzle_state_connecting", "%5d", con->fd)

  while (1)
  {
    if (con->revents & POLLOUT)
    {
      DRIZZLE_STATE_POP(con)
      return DRIZZLE_RETURN_OK;
    }
    else if (con->revents & (POLLERR | POLLHUP | POLLNVAL))
    {
      con->revents= 0;
      DRIZZLE_STATE_POP(con)
      DRIZZLE_STATE_PUSH(con, drizzle_state_connect)
      con->addrinfo_next= con->addrinfo_next->ai_next;
      return DRIZZLE_RETURN_OK;
    }

    ret= drizzle_con_set_events(con, POLLOUT);
    if (ret != DRIZZLE_RETURN_OK)
      return ret;

    if (con->drizzle->options & DRIZZLE_NON_BLOCKING)
      return DRIZZLE_RETURN_IO_WAIT;

    ret= drizzle_con_wait(con->drizzle);
    if (ret != DRIZZLE_RETURN_OK)
      return ret;
  }
}

drizzle_return_t drizzle_state_read(drizzle_con_st *con)
{
  drizzle_return_t ret;
  ssize_t read_size;

  if (con->buffer_size == 0)
    con->buffer_ptr= con->buffer;
  else if ((con->buffer_ptr - con->buffer) > (DRIZZLE_MAX_BUFFER_SIZE / 2))
  {
    memmove(con->buffer, con->buffer_ptr, con->buffer_size);
    con->buffer_ptr= con->buffer;
  }

  while (1)
  {
    read_size= read(con->fd, con->buffer_ptr + con->buffer_size,
                    (size_t)DRIZZLE_MAX_BUFFER_SIZE -
                    ((size_t)(con->buffer_ptr - con->buffer) +
                     con->buffer_size));
    PDEBUG("drizzle_state_read", "%5d %5zd %5d", con->fd, read_size, errno)
    if (read_size == 0)
    {
      DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_read",
                        "lost connection to server (EOF)")
      return DRIZZLE_RETURN_LOST_CONNECTION;
    }
    else if (read_size == -1)
    {
      if (errno == EAGAIN)
      {
        ret= drizzle_con_set_events(con, POLLIN);
        if (ret != DRIZZLE_RETURN_OK)
          return 0;

        if (con->drizzle->options & DRIZZLE_NON_BLOCKING)
          return DRIZZLE_RETURN_IO_WAIT;

        ret= drizzle_con_wait(con->drizzle);
        if (ret != DRIZZLE_RETURN_OK)
          return ret;

        continue;
      }
      else if (errno == ECONNREFUSED)
      {
        con->revents= 0;
        DRIZZLE_STATE_POP(con)
        DRIZZLE_STATE_PUSH(con, drizzle_state_connect)
        con->addrinfo_next= con->addrinfo_next->ai_next;
        return DRIZZLE_RETURN_OK;
      }
      else if (errno == EINTR)
        continue;
      else if (errno == EPIPE || errno == ECONNRESET)
      {
        DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_read",
                          "lost connection to server (%d)", errno)
        return DRIZZLE_RETURN_LOST_CONNECTION;
      }

      DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_read", "read:%d", errno)
      con->drizzle->last_errno= errno;
      return DRIZZLE_RETURN_ERRNO;
    }

    con->buffer_size+= (size_t)read_size;
    break;
  }

  DRIZZLE_STATE_POP(con);
  return DRIZZLE_RETURN_OK;
}

drizzle_return_t drizzle_state_write(drizzle_con_st *con)
{
  drizzle_return_t ret;
  ssize_t write_size;

  while (con->buffer_size != 0)
  {
    write_size= write(con->fd, con->buffer_ptr, con->buffer_size);
    PDEBUG("drizzle_state_write", "%5d %5zd %5d", con->fd, write_size, errno)
    if (write_size == 0)
    {
      DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_write",
                        "lost connection to server (EOF)")
      return DRIZZLE_RETURN_LOST_CONNECTION;
    }
    else if (write_size == -1)
    {
      if (errno == EAGAIN)
      {
        ret= drizzle_con_set_events(con, POLLOUT);
        if (ret != DRIZZLE_RETURN_OK)
          return ret;

        if (con->drizzle->options & DRIZZLE_NON_BLOCKING)
          return DRIZZLE_RETURN_IO_WAIT;

        ret= drizzle_con_wait(con->drizzle);
        if (ret != DRIZZLE_RETURN_OK)
          return ret;

        continue;
      }
      else if (errno == EINTR)
        continue;
      else if (errno == EPIPE || errno == ECONNRESET)
      {
        DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_write",
                          "lost connection to server (%d)", errno)
        return DRIZZLE_RETURN_LOST_CONNECTION;
      }

      DRIZZLE_ERROR_SET(con->drizzle, "drizzle_state_write", "write:%d", errno)
      con->drizzle->last_errno= errno;
      return DRIZZLE_RETURN_ERRNO;
    }

    con->buffer_ptr+= write_size;
    con->buffer_size-= (size_t)write_size;
    if (con->buffer_size == 0)
      break;
  }

  con->buffer_ptr= con->buffer;

  DRIZZLE_STATE_POP(con);
  return DRIZZLE_RETURN_OK;
}

/*
 * Private definitions
 */

void drizzle_con_reset_addrinfo(drizzle_con_st *con)
{
  if (con->tcp_addrinfo != NULL)
  {
    freeaddrinfo(con->tcp_addrinfo);
    con->tcp_addrinfo= NULL;
  }

  con->uds_addrinfo.ai_addr= NULL;
  con->addrinfo_next= NULL;
}

static drizzle_return_t _con_setsockopt(drizzle_con_st *con)
{
  int ret;
  struct linger linger;
  struct timeval waittime;

  ret= 1;
  ret= setsockopt(con->fd, IPPROTO_TCP, TCP_NODELAY, &ret,
                  (socklen_t)sizeof(int));
  if (ret == -1 && errno != EOPNOTSUPP)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "setsockopt:TCP_NODELAY:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }

  linger.l_onoff= 1;
  linger.l_linger= DRIZZLE_DEFAULT_SOCKET_TIMEOUT;
  ret= setsockopt(con->fd, SOL_SOCKET, SO_LINGER, &linger,
                  (socklen_t)sizeof(struct linger));
  if (ret == -1)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "setsockopt:SO_LINGER:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }

  waittime.tv_sec= DRIZZLE_DEFAULT_SOCKET_TIMEOUT;
  waittime.tv_usec= 0;
  ret= setsockopt(con->fd, SOL_SOCKET, SO_SNDTIMEO, &waittime,
                  (socklen_t)sizeof(struct timeval));
  if (ret == -1 && errno != ENOPROTOOPT)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "setsockopt:SO_SNDTIMEO:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }

  ret= setsockopt(con->fd, SOL_SOCKET, SO_RCVTIMEO, &waittime,
                  (socklen_t)sizeof(struct timeval));
  if (ret == -1 && errno != ENOPROTOOPT)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "setsockopt:SO_RCVTIMEO:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }

  ret= DRIZZLE_DEFAULT_SOCKET_SEND_SIZE;
  ret= setsockopt(con->fd, SOL_SOCKET, SO_SNDBUF, &ret, (socklen_t)sizeof(int));
  if (ret == -1)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "setsockopt:SO_SNDBUF:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }

  ret= DRIZZLE_DEFAULT_SOCKET_RECV_SIZE;
  ret= setsockopt(con->fd, SOL_SOCKET, SO_RCVBUF, &ret, (socklen_t)sizeof(int));
  if (ret == -1)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "setsockopt:SO_RCVBUF:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }

#ifdef FIONBIO
  ret= 1;
  ret= ioctl(con->fd, FIONBIO, &ret);
  if (ret == -1)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "ioctl:FIONBIO:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }
#else /* !FIONBIO */
  ret= fcntl(con->fd, F_GETFL, 0);
  if (ret == -1)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "fcntl:F_GETFL:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }

  ret= fcntl(con->fd, F_SETFL, ret | O_NONBLOCK);
  if (ret == -1)
  {
    DRIZZLE_ERROR_SET(con->drizzle, "_con_setsockopt",
                      "fcntl:F_SETFL:%d", errno)
    return DRIZZLE_RETURN_ERRNO;
  }
#endif /* FIONBIO */

  return DRIZZLE_RETURN_OK;
}
