/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Drizzle core declarations
 */

#ifndef __DRIZZLE_H
#define __DRIZZLE_H

#if !defined(__cplusplus) && !defined(_WIN32)
# include <stdbool.h>
#endif

#include <inttypes.h>
#include <sys/types.h>

#ifdef _WIN32
# include <winsock2.h>
# include <Ws2tcpip.h>
#else
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <sys/un.h>
# include <netdb.h>
# include <poll.h>
#endif

#include <libdrizzle/visibility.h>
#include <libdrizzle/constants.h>
#include <libdrizzle/structs.h>
#include <libdrizzle/conn.h>
#include <libdrizzle/result.h>
#include <libdrizzle/column.h>

#ifdef  __cplusplus
extern "C" {
#endif

/**
 * @addtogroup drizzle Drizzle Core Interface
 * @ingroup drizzle_client
 * @ingroup drizzle_server
 * This is the core library instance context that other structs (such as
 * connections) are created from.
 *
 * There is no locking within a single drizzle_st structure, so for threaded
 * applications you must either ensure isolation in the application or use
 * multiple drizzle_st structures (for example, one for each thread).
 * @{
 */

/**
 * Get library version string.
 * @return Pointer to static buffer in library that holds the version string.
 */
DRIZZLE_API
const char *drizzle_version(void);

/**
 * Initialize a library instance structure.
 * @param drizzle Caller allocated drizzle structure, or NULL to allocate one.
 * @return Pointer to an allocated drizzle structure if drizzle parameter was
 *         NULL, or the drizzle parameter pointer if it was not NULL.
 */
DRIZZLE_API
drizzle_st *drizzle_create(drizzle_st *drizzle);

/**
 * Clone a library instance structure.
 * @param drizzle Caller allocated drizzle structure, or NULL to allocate one.
 * @param from Drizzle structure to use as a source to clone from.
 * @return Pointer to an allocated drizzle structure if drizzle parameter was
 *         NULL, or the drizzle parameter pointer if it was not NULL.
 */
DRIZZLE_API
drizzle_st *drizzle_clone(drizzle_st *drizzle, drizzle_st *from);

/**
 * Free a library instance structure.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 */
DRIZZLE_API
void drizzle_free(drizzle_st *drizzle);

/**
 * Return an error string for last library error encountered.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @return Pointer to static buffer in library that holds an error string.
 */
DRIZZLE_API
const char *drizzle_error(drizzle_st *drizzle);

/**
 * Value of errno in the case of a DRIZZLE_RETURN_ERRNO return value.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @return An errno value as defined in your system errno.h file.
 */
DRIZZLE_API
int drizzle_errno(drizzle_st *drizzle);

/**
 * Get server defined error code for the last result read.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @return An error code given back in the server response.
 */
DRIZZLE_API
uint16_t drizzle_error_code(drizzle_st *drizzle);

/**
 * Get SQL state code for the last result read.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @return A SQLSTATE code given back in the server response.
 */
DRIZZLE_API
const char *drizzle_sqlstate(drizzle_st *drizzle);

/**
 * Get options for a library instance structure.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @return Options set for drizzle struct.
 */
DRIZZLE_API
drizzle_options_t drizzle_options(drizzle_st *drizzle);

/**
 * Set options for a library instance structure.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @param options Available options for drizzle structs to set.
 */
DRIZZLE_API
void drizzle_set_options(drizzle_st *drizzle, drizzle_options_t options);

/**
 * Add options for a library instance structure.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @param options Available options for drizzle structs that should be added.
 */
DRIZZLE_API
void drizzle_add_options(drizzle_st *drizzle, drizzle_options_t options);

/**
 * Remove options for a library instance structure.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @param options Available options for drizzle structs that should be removed.
 */
DRIZZLE_API
void drizzle_remove_options(drizzle_st *drizzle, drizzle_options_t options);

/**
 * Set custom I/O event callbacks for a drizzle structure.
 * @param drizzle Drizzle structure previously initialized with drizzle_create
 *        or drizzle_clone.
 * @param event_watch Callback function to be called for events.
 * @param event_watch_arg Argument to pass in with callback function.
 */
void drizzle_set_event_watch(drizzle_st *drizzle,
                             drizzle_event_watch_fn *event_watch,
                             void *event_watch_arg);

/** @} */

#ifdef  __cplusplus
}
#endif

#endif /* __DRIZZLE_H */
