/*
 * libexplain - Explain errno values returned by libc functions
 * Copyright (C) 2008, 2009 Peter Miller
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <libexplain/ac/errno.h>
#include <libexplain/ac/sys/stat.h>

#include <libexplain/buffer/eacces.h>
#include <libexplain/buffer/efault.h>
#include <libexplain/buffer/eio.h>
#include <libexplain/buffer/eloop.h>
#include <libexplain/buffer/enametoolong.h>
#include <libexplain/buffer/enoent.h>
#include <libexplain/buffer/enomem.h>
#include <libexplain/buffer/enotdir.h>
#include <libexplain/buffer/errno/generic.h>
#include <libexplain/buffer/errno/path_resolution.h>
#include <libexplain/buffer/errno/readlink.h>
#include <libexplain/buffer/file_type.h>
#include <libexplain/buffer/pathname.h>
#include <libexplain/buffer/pointer.h>
#include <libexplain/explanation.h>
#include <libexplain/path_is_efault.h>


static void
explain_buffer_errno_readlink_system_call(explain_string_buffer_t *sb,
    int errnum, const char *pathname, char *data, int data_size)
{
    (void)errnum;
    explain_string_buffer_puts(sb, "readlink(pathname = ");
    explain_buffer_pathname(sb, pathname);
    explain_string_buffer_puts(sb, ", data = ");
    explain_buffer_pointer(sb, data);
    explain_string_buffer_printf(sb, ", data_size = %d)", data_size);
}


static void
explain_buffer_errno_readlink_explanation(explain_string_buffer_t *sb,
    int errnum, const char *pathname, char *data, int data_size)
{
    explain_final_t final_component;

    (void)data;
    explain_final_init(&final_component);
    final_component.want_to_read = 1;
    final_component.follow_symlink = 0;

    switch (errnum)
    {
    case EACCES:
        explain_buffer_eacces(sb, pathname, "pathname", &final_component);
        break;

    case EFAULT:
        if (explain_path_is_efault(pathname))
        {
            explain_buffer_efault(sb, "pathname");
            break;
        }
        if (explain_path_is_efault(data))
        {
            explain_buffer_efault(sb, "data");
            break;
        }
        break;

    case EINVAL:
        if (data_size <= 0)
        {
            explain_string_buffer_puts
            (
                sb,
                /* FIXME: i18n */
                "data_size is not positive"
            );
        }
        else
        {
            struct stat     st;

            /* FIXME: explain_buffer_wrong_file_type */
            if (lstat(pathname, &st) >= 0)
            {
                explain_string_buffer_puts(sb, "pathname is a ");
                explain_buffer_file_type(sb, st.st_mode);
                explain_string_buffer_puts(sb, ", not a ");
                explain_buffer_file_type(sb, S_IFLNK);
            }
            else
            {
                explain_string_buffer_puts(sb, "pathname is not a ");
                explain_buffer_file_type(sb, S_IFLNK);
            }
        }
        break;

    case EIO:
        explain_buffer_eio_path(sb, pathname);
        break;

    case ELOOP:
    case EMLINK: /* BSD */
        explain_buffer_eloop(sb, pathname, "pathname", &final_component);
        break;

    case ENAMETOOLONG:
        explain_buffer_enametoolong
        (
            sb,
            pathname,
            "pathname",
            &final_component
        );
        break;

    case ENOENT:
        explain_buffer_enoent(sb, pathname, "pathname", &final_component);
        break;

    case ENOMEM:
        explain_buffer_enomem_kernel(sb);
        break;

    case ENOTDIR:
        explain_buffer_enotdir(sb, pathname, "pathname", &final_component);
        break;

    default:
        explain_buffer_errno_generic(sb, errnum);
        break;
    }
}


void
explain_buffer_errno_readlink(explain_string_buffer_t *sb, int errnum,
    const char *pathname, char *data, int data_size)
{
    explain_explanation_t exp;

    explain_explanation_init(&exp, errnum);
    explain_buffer_errno_readlink_system_call
    (
        &exp.system_call_sb,
        errnum,
        pathname,
        data,
        data_size
    );
    explain_buffer_errno_readlink_explanation
    (
        &exp.explanation_sb,
        errnum,
        pathname,
        data,
        data_size
    );
    explain_explanation_assemble(&exp, sb);
}
