/* 

                          Firewall Builder

                 Copyright (C) 2001 NetCitadel, LLC

  Author:  Vadim Zaliva lord@crocodile.org

  $Id: FWObjectDatabase.cc,v 1.26 2003/07/06 06:12:20 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#include <assert.h>

#include <fwbuilder/libfwbuilder-config.h>

#include <fwbuilder/FWObject.hh>
#include <fwbuilder/FWObjectDatabase.hh>

#include <fwbuilder/FWObject.hh>
#include <fwbuilder/Interval.hh>
#include <fwbuilder/ICMPService.hh>
#include <fwbuilder/IPService.hh>
#include <fwbuilder/TCPService.hh>
#include <fwbuilder/UDPService.hh>
#include <fwbuilder/CustomService.hh>
#include <fwbuilder/FWReference.hh>
#include <fwbuilder/FWObjectReference.hh>
#include <fwbuilder/FWServiceReference.hh>
#include <fwbuilder/FWIntervalReference.hh>
#include <fwbuilder/Host.hh>
#include <fwbuilder/Interface.hh>
#include <fwbuilder/IPv4.hh>
#include <fwbuilder/physAddress.hh>
#include <fwbuilder/InterfacePolicy.hh>
#include <fwbuilder/Group.hh>
#include <fwbuilder/Rule.hh>
#include <fwbuilder/RuleElement.hh>
#include <fwbuilder/RuleSet.hh>
#include <fwbuilder/FWOptions.hh>
#include <fwbuilder/Firewall.hh>
#include <fwbuilder/NAT.hh>
#include <fwbuilder/Policy.hh>
#include <fwbuilder/ObjectGroup.hh>
#include <fwbuilder/ServiceGroup.hh>
#include <fwbuilder/IntervalGroup.hh>
#include <fwbuilder/Network.hh>
#include <fwbuilder/AddressRange.hh>
#include <fwbuilder/Management.hh>
#include <fwbuilder/XMLTools.hh>

#include <iostream>

using namespace std;
using namespace libfwbuilder;


FWObjectDatabase*  FWObjectDatabase::db        = NULL;
const char*        FWObjectDatabase::TYPENAME  = {"FWObjectDatabase"};
long               FWObjectDatabase::IDcounter = 0;
//Mutex FWObjectDatabase::IDmutex;
const string FWObjectDatabase::DTD_FILE_NAME  = "fwbuilder.dtd"    ;


FWObjectDatabase::FWObjectDatabase():FWObject()
{
    assert(db==NULL);
    db=this;

    IDcounter = time(NULL);
    data_file = "";

    setName(TYPENAME);

    setId( getRootId() );
    setDirty(false);
}

FWObjectDatabase::FWObjectDatabase(FWObjectDatabase& d):FWObject()
{
    data_file = d.data_file;

    setName(TYPENAME);

    *this = d;  // copies entire tree

    setId( getRootId() );
    setDirty(false);
}

void FWObjectDatabase::setFileName(const string &filename)
{
    data_file = filename;
}

const string &FWObjectDatabase::getFileName() 
{ 
    return data_file; 
}

const string FWObjectDatabase::getFileDir() 
{ 
  char dir_delimiter='/';
#ifdef __MINGW32__
  dir_delimiter = '\\';
#endif

    string::size_type i = data_file.rfind(dir_delimiter);
    if (i==string::npos || i==0) 
	return "";
    else 
	return data_file.substr(0,i);
}

void FWObjectDatabase::load(const string &f, XMLTools::UpgradePredicate *upgrade) throw(FWException)
{
    if(f=="") 
        return;
    
    xmlDocPtr doc=XMLTools::loadFile(f, FWObjectDatabase::TYPENAME, FWObjectDatabase::DTD_FILE_NAME, upgrade);
    
    xmlNodePtr root = xmlDocGetRootElement(doc);
    
    if(!root || !root->name || strcmp(FROMXMLCAST(root->name), FWObjectDatabase::TYPENAME)!=SAME)
    {
	xmlFreeDoc(doc);
        throw FWException("Data file have invalid structure: "+f);
    }
    
    clearChildren();
    
    fromXML(root);
    setDirty(false,true);  // clear dirty flag for all objects, recursive oper.
    setFileName(f);
}

void FWObjectDatabase::saveXML(xmlDocPtr doc) throw(FWException)
{
    doc->children = xmlNewDocNode(doc, NULL, STRTOXMLCAST(getName()), NULL);
    xmlNewNs(doc->children, TOXMLCAST("http://www.fwbuilder.org/1.0/"), NULL);

    toXML(xmlDocGetRootElement(doc));

    XMLTools::setDTD(doc, FWObjectDatabase::TYPENAME, FWObjectDatabase::DTD_FILE_NAME);
}

void FWObjectDatabase::saveFile(const string &filename) throw(FWException)
{
    xmlDocPtr doc = xmlNewDoc(TOXMLCAST("1.0"));
  
    doc->children = xmlNewDocNode(doc, NULL, STRTOXMLCAST(getName()), NULL);
    xmlNewNs(doc->children, TOXMLCAST("http://www.fwbuilder.org/1.0/"), NULL);

    toXML(xmlDocGetRootElement(doc));
    
    XMLTools::saveFile(doc, filename, FWObjectDatabase::TYPENAME, FWObjectDatabase::DTD_FILE_NAME);
    
    xmlFreeDoc(doc);  
    setDirty(false,true);  // clear dirty flag for all objects, recursive oper.
    
}

xmlNodePtr FWObjectDatabase::toXML(xmlNodePtr parent) throw(FWException)
{
    FWObject *o;

    xmlNewProp(parent,NULL,NULL);

    xmlNewProp(parent, 
               TOXMLCAST("version") , 
               TOXMLCAST(LIBFWBUILDER_FORMAT_VERSION));

    string rootid = getId();

    xmlAttrPtr pr = xmlNewProp(parent, 
			       TOXMLCAST("id") , 
			       STRTOXMLCAST(rootid));

    xmlAddID(NULL, parent->doc, STRTOXMLCAST(rootid), pr);


    for(list<FWObject*>::const_iterator j=begin(); j!=end(); ++j) 
        if((o=(*j))!=NULL)   
            o->toXML(parent);
    
    return parent;
}

FWObject *FWObjectDatabase::createFromXML(xmlNodePtr data)
{
    FWObject *nobj;
    const char *n=FROMXMLCAST(data->name);
    if (!n) 
        return NULL;
    
    nobj=create(n);
    
    return(nobj);
}

string FWObjectDatabase::generateUniqueId()
{
    char res[20];

    //TODO: mutex could not be used since threads
    // library is not initialized.

//    IDmutex.lock();
    sprintf(res,"id%lX",IDcounter);
    IDcounter++;
//    IDmutex.unlock();
    
    return res;
}


#define CREATE_OBJ(classname,str) \
if(strcmp(classname::TYPENAME,str)==SAME)\
{ if (create_with_root) nobj=new classname(this); else nobj=new classname();  return nobj; }

FWObject *FWObjectDatabase::create(const string &type_name, bool create_with_root)
{
    const char *n=type_name.c_str();
    FWObject   *nobj;

    if(strcmp("AnyNetwork"  ,n)==SAME)  { nobj = new Network();   nobj->setXMLName("AnyNetwork");   return nobj; }
    if(strcmp("AnyIPService",n)==SAME)  { nobj = new IPService(); nobj->setXMLName("AnyIPService"); return nobj; }
    if(strcmp("AnyInterval" ,n)==SAME)  { nobj = new Interval();  nobj->setXMLName("AnyInterval");  return nobj; }


    CREATE_OBJ(Policy              ,n);
    CREATE_OBJ(InterfacePolicy     ,n);
    CREATE_OBJ(NAT                 ,n);
    CREATE_OBJ(PolicyRule          ,n);
    CREATE_OBJ(NATRule             ,n);
    CREATE_OBJ(RuleElementSrc      ,n);
    CREATE_OBJ(RuleElementDst      ,n);
    CREATE_OBJ(RuleElementSrv      ,n);
    CREATE_OBJ(RuleElementOSrc     ,n);
    CREATE_OBJ(RuleElementODst     ,n);
    CREATE_OBJ(RuleElementOSrv     ,n);
    CREATE_OBJ(RuleElementTSrc     ,n);
    CREATE_OBJ(RuleElementTDst     ,n);
    CREATE_OBJ(RuleElementTSrv     ,n);
    CREATE_OBJ(RuleElementInterval ,n);
    CREATE_OBJ(HostOptions         ,n);
    CREATE_OBJ(FirewallOptions     ,n);
    CREATE_OBJ(PolicyRuleOptions   ,n);
    CREATE_OBJ(NATRuleOptions      ,n);
    CREATE_OBJ(Host                ,n);
    CREATE_OBJ(AddressRange        ,n);
    CREATE_OBJ(Management          ,n);
    CREATE_OBJ(PolicyInstallScript ,n);
    CREATE_OBJ(SNMPManagement      ,n);
    CREATE_OBJ(FWBDManagement      ,n);
    CREATE_OBJ(Firewall            ,n);
    CREATE_OBJ(Network             ,n);
    CREATE_OBJ(Interface           ,n);
    CREATE_OBJ(IPv4                ,n);
    CREATE_OBJ(physAddress         ,n);
    CREATE_OBJ(FWObjectReference   ,n);
    CREATE_OBJ(FWServiceReference  ,n);
    CREATE_OBJ(FWIntervalReference ,n);
    CREATE_OBJ(TCPService          ,n);
    CREATE_OBJ(UDPService          ,n);
    CREATE_OBJ(ICMPService         ,n);
    CREATE_OBJ(IPService           ,n);
    CREATE_OBJ(CustomService       ,n);
    CREATE_OBJ(Interval            ,n);
    CREATE_OBJ(ObjectGroup         ,n);
    CREATE_OBJ(ServiceGroup        ,n);
    CREATE_OBJ(IntervalGroup       ,n);

    cerr << "FWObjectDatabase::create: ERROR: attempt to create object of unknown type '" 
         << type_name << "'" << endl;

    //TODO: throw exception
    nobj = NULL;

    return nobj;
}

Firewall* FWObjectDatabase::_findFirewallByNameRecursive(FWObject* db,
                                         const string &name) throw(FWException)
{
    if (Firewall::isA(db) && db->getName()==name) return Firewall::cast(db);

    list<FWObject*>::iterator j;
    for(j=db->begin(); j!=db->end(); ++j)     
    {
        FWObject *o=*j;

        o=_findFirewallByNameRecursive(o,name);
        if(o) return Firewall::cast(o);
    }
    if (db==this)
        throw FWException("Firewall object '"+name+"' not found");
    return NULL; // not found
}

Firewall* FWObjectDatabase::findFirewallByName(const string &name) throw(FWException)
{
    return _findFirewallByNameRecursive(this,name);
}


