/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: Interface.cc,v 1.24 2002/12/29 07:33:11 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <assert.h>

#include <fwbuilder/libfwbuilder-config.h>

#include <fwbuilder/Interface.hh>
#include <fwbuilder/InterfacePolicy.hh>
#include <fwbuilder/XMLTools.hh>
#include <fwbuilder/IPv4.hh>
#include <fwbuilder/FWObjectDatabase.hh>

using namespace std;
using namespace libfwbuilder;

const char *Interface::TYPENAME={"Interface"};

Interface::Interface(const Interface &i):Address()
{
    FWObject::operator=(i);

    bcast_bits       = i.bcast_bits       ; 
    ostatus          = i.ostatus          ;
    snmp_type        = i.snmp_type        ;
}

Interface::Interface():Address()
{
    setName("unknown");
    setBool("dyn",false);
    setBool("unnum",false);
    setInt("security_level",0);

    bcast_bits       = 1    ;
    ostatus          = true ;
    snmp_type        = -1   ;     
}

Interface::Interface(const FWObject *root) : Address(root)
{
    setName("unknown");
    setBool("dyn",false);
    setBool("unnum",false);
    setInt("security_level",0);

    bcast_bits       = 1    ;
    ostatus          = true ;
    snmp_type        = -1   ;     
}

Interface::~Interface() {}

FWObject& Interface::shallowDuplicate(const FWObject *o, bool preserve_id) throw(FWException)
{
    FWObject::shallowDuplicate(o,preserve_id);

    if (Interface::isA(o))
    {
        bcast_bits       = Interface::constcast(o)->bcast_bits       ; 
        ostatus          = Interface::constcast(o)->ostatus          ;
        snmp_type        = Interface::constcast(o)->snmp_type        ;
    }
    return *this;
}

FWObject& Interface::duplicate(const FWObject *x, bool preserve_id) throw(FWException)
{
    FWObject::duplicate(x,preserve_id);

    const Interface *rx=Interface::constcast(x);
    if (rx!=NULL)
    {
        bcast_bits = rx->bcast_bits;
        ostatus    = rx->ostatus;
        snmp_type  = rx->snmp_type;
    }

    return *this;
}


void Interface::fromXML(xmlNodePtr root) throw(FWException)
{
    FWObject::fromXML(root);

    const char *n;

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("security_level")));
    if (n!=NULL)
    {
        setStr("security_level",n);
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("dyn")));
    if (n!=NULL)
    {
        setStr("dyn",n);
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("unnum")));
    if (n!=NULL)
    {
        setStr("unnum",n);
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("mgmt")));
    if (n!=NULL)
    {
        setStr("mgmt",n);
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("label")));
    if (n!=NULL)
    {
        setStr("label",n);
        FREEXMLBUFF(n);
    }

    n=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("network_zone")));
    if (n!=NULL)
    {
        setStr("network_zone",n);
        FREEXMLBUFF(n);
    }
}

xmlNodePtr Interface::toXML(xmlNodePtr parent) throw(FWException)
{
    xmlNodePtr me = FWObject::toXML(parent, false);
    FWObject *o;

    for(FWObjectTypedChildIterator j=findByType(IPv4::TYPENAME); j!=j.end(); ++j)
        if((o=(*j))!=NULL )
            o->toXML(me);
    
    for(FWObjectTypedChildIterator j=findByType(physAddress::TYPENAME); j!=j.end(); ++j)
        if((o=(*j))!=NULL )
            o->toXML(me);
    
    o=getFirstByType( InterfacePolicy::TYPENAME );
    if (o) o->toXML(me);

    return me;
}



int  Interface::getSecurityLevel() const
{
    return getInt("security_level") ;
}

void Interface::setSecurityLevel(int level)
{
    setInt("security_level",level);
}

bool Interface::isExt() const 
{ 
    return( getInt("security_level")==0 );
}

/**
 *   if parameter value is true, then security level is set to 0 (least secure,
 *   or "outside")
 *   if parameter value is false, then current security level is set to 100
 */
void Interface::setExt(bool external)
{
    setInt("security_level",(external)?0:100);
}

void Interface::setDyn(bool value) { setBool("dyn",value); }
bool Interface::isDyn() const { return(getBool("dyn")); }

void Interface::setUnnumbered(bool value) { setBool("unnum",value); }
bool Interface::isUnnumbered() const { return getBool("unnum"); }

void Interface::setManagement(bool value) { setBool("mgmt",value); }
bool Interface::isManagement() const { return (getBool("mgmt")); }

void Interface::setOStatus(bool value) { ostatus=value; }

void Interface::setInterfaceType(int _snmp_type) { snmp_type=_snmp_type; }

void Interface::setBroadcastBits(int _val) { bcast_bits=_val; }

bool  Interface::validateChild(FWObject *o)
{
    string otype=o->getTypeName();
    return (otype==IPv4::TYPENAME || otype==physAddress::TYPENAME || otype==InterfacePolicy::TYPENAME );
}

bool Interface::isLoopback() const
{
    return (getAddress()==IPAddress("127.0.0.1"));
}

physAddress*  Interface::getPhysicalAddress () const
{
    return physAddress::cast( getFirstByType( physAddress::TYPENAME ) );
}

/*
 * per DTD, there can be 0 or 1 physAddress
 */
void  Interface::setPhysicalAddress(const std::string &paddr)
{
    physAddress *pa=getPhysicalAddress();
    if (pa!=NULL) 
    {
        pa->setPhysAddress(paddr);
    } else
    {
        pa=physAddress::cast( FWObjectDatabase::db->create(physAddress::TYPENAME) );
        pa->setPhysAddress(paddr);
        add(pa);
    }
}

const string &Interface::getLabel() const
{
    return getStr("label");
}

void          Interface::setLabel(const string& n)
{
    setStr("label",n);
}
    
const IPAddress Interface::getIPAddress() const throw(FWException)
{
    IPv4 *ipv4=IPv4::cast( getFirstByType( IPv4::TYPENAME ) );
    if (ipv4!=NULL) return ipv4->getAddress();
    else return IPAddress();
}

IPNetwork Interface::getIPNetwork() const throw(FWException)
{
    IPv4 *ipv4=IPv4::cast( getFirstByType(IPv4::TYPENAME) );
    if (ipv4!=NULL) return ipv4->getIPNetwork();
    else  return IPNetwork( IPAddress() , Netmask() );
}

IPAddress Interface::getAddress() const
{
    IPv4 *ipv4=IPv4::cast( getFirstByType( IPv4::TYPENAME ) );
    if (ipv4!=NULL) return ipv4->getAddress();
    else return IPAddress();
}

Netmask   Interface::getNetmask() const
{
    IPv4 *ipv4=IPv4::cast( getFirstByType( IPv4::TYPENAME ) );
    if (ipv4!=NULL) return ipv4->getNetmask();
    else return Netmask();
}

IPv4*  Interface::addIPv4()
{
    IPv4* ipv4=IPv4::cast( FWObjectDatabase::db->create(IPv4::TYPENAME) );
    add(ipv4);
    return ipv4;
}

void Interface::setAddress(const IPAddress &a)  
{ 
    IPv4 *ipv4=IPv4::cast( getFirstByType( IPv4::TYPENAME ) );
    if (ipv4==NULL)   ipv4=addIPv4();
    ipv4->setAddress(a);
}

void Interface::setNetmask(const Netmask   &nm) 
{ 
    IPv4 *ipv4=IPv4::cast( getFirstByType( IPv4::TYPENAME ) );
    if (ipv4==NULL)   ipv4=addIPv4();
    ipv4->setNetmask(nm);
}

void Interface::setAddress(const std::string &a)  
{
    setAddress(IPAddress(a));
}

void Interface::setNetmask(const std::string &nm) 
{
    setNetmask(Netmask(nm));
}

guint32   Interface::dimension()  const
{
    return 1;
}



