/**
 * @file libgalago/galago-core.c Galago Context API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <libgalago/galago-private.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-log.h>
#include <libgalago/galago-marshal.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <string.h>

#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif

struct _GalagoCore
{
	GalagoObject parent_object;

	char *app_name;
	char *conn_obj_path;
	char *uid;

	DBusConnection *dbus_conn;

	galago_bool filters_added;
	galago_bool watch_all;

	galago_bool daemon;
	galago_bool feed;
	galago_bool registered;

	galago_bool daemon_active;

	galago_bool exit_with_daemon;
	galago_bool registering_connection;
};

typedef struct
{
	GalagoObjectClass parent_class;

} GalagoCoreClass;

#define GALAGO_DAEMON_ID "#galago-daemon#"

static void _galago_core_disconnect(void);
static galago_bool _galago_dbus_register_connection(void);
static void _galago_dbus_unregister_connection(void);
static void _galago_dbus_core_add_service(GalagoService *service);
static void _galago_dbus_core_remove_service(GalagoService *service);
static void _galago_dbus_core_add_person(GalagoPerson *person);
static void _galago_dbus_core_remove_person(GalagoPerson *person);

static GalagoCore *_core = NULL;
static galago_bool first_init = TRUE;

/* D-BUS compatibility */
typedef dbus_bool_t (*StartServiceFunc)(DBusConnection *, const char *,
										dbus_uint32_t, dbus_uint32_t *,
										DBusError *);

#if GALAGO_CHECK_DBUS_VERSION(0, 30)
StartServiceFunc start_service = dbus_bus_start_service_by_name;
#else
StartServiceFunc start_service = dbus_bus_activate_service;
#endif

static GalagoContextOps context_ops =
{
	_galago_dbus_core_add_service,
	_galago_dbus_core_remove_service,
	_galago_dbus_core_add_person,
	_galago_dbus_core_remove_person
};

/**************************************************************************
 * Object/Class support
 **************************************************************************/
#define GALAGO_CLASS_CORE (galago_core_get_class())

GALAGO_REGISTER_CLASS(galago_core, GalagoCore, NULL,
					  GALAGO_DBUS_CORE_INTERFACE);

static void
galago_core_object_init(GalagoCore *core)
{
	_core = core;

	galago_object_set_dbus_path(core, GALAGO_DBUS_CORE_OBJECT);

	galago_signals_init();
}

static void
galago_core_object_finalize(GalagoObject *object)
{
	GalagoContext *context;

	_galago_core_disconnect();

	/*
	 * Hack to prevent this object from beign removed from the
	 * soon-to-be destroyed context after this function ends.
	 */
	galago_object_set_dbus_path(object, NULL);

	context = galago_context_get();
	galago_context_pop();
	galago_context_destroy(context);

	galago_signals_uninit();

	if (_core->app_name != NULL)
		free(_core->app_name);

	if (_core->uid != NULL)
		free(_core->uid);

	if (_core->conn_obj_path != NULL)
		free(_core->conn_obj_path);
}

static void
galago_core_class_init(GalagoObjectClass *klass)
{
	klass->finalize = galago_core_object_finalize;

	galago_signal_register(klass->signal_context, "registered",
						   galago_marshal_VOID, 0);
	galago_signal_register(klass->signal_context, "unregistered",
						   galago_marshal_VOID, 0);
	galago_signal_register(klass->signal_context, "service-added",
						   galago_marshal_VOID__BOXED, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "service-removed",
						   galago_marshal_VOID__BOXED, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "person-added",
						   galago_marshal_VOID__BOXED, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "person-removed",
						   galago_marshal_VOID__BOXED, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "updated",
						   galago_marshal_VOID, 0);
}

/**************************************************************************
 * D-BUS Processing
 **************************************************************************/
static void
reregister_client(void)
{
	const GalagoList *l;

	if (!_galago_dbus_register_connection())
	{
		galago_log_error("Unable to re-establish registration\n");

		return;
	}

	if (!galago_core_is_feed())
		return;

	for (l = galago_core_get_people(TRUE, FALSE); l != NULL; l = l->next)
	{
		GalagoPerson *person = (GalagoPerson *)l->data;

		_galago_dbus_core_add_person(person);
		galago_dbus_object_push_full(person);
	}

	for (l = galago_core_get_services(TRUE, FALSE); l != NULL; l = l->next)
	{
		GalagoService *service = (GalagoService *)l->data;

		_galago_dbus_core_add_service(service);
		galago_dbus_object_push_full(service);
	}
}

static DBusHandlerResult
_handle_dbus_error(DBusConnection *dbus_conn, DBusMessage *message,
				   void *user_data)
{
	const char *error_name = dbus_message_get_error_name(message);

	if (dbus_message_is_error(message, DBUS_ERROR_NO_REPLY))
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	if (dbus_message_is_error(message, DBUS_ERROR_SERVICE_UNKNOWN))
	{
		DBusError error;

		if (!galago_is_daemon())
		{
			galago_context_clear_objects(FALSE);

			galago_core_set_watch_all(FALSE);

			_core->daemon_active = FALSE;
			_core->registered    = FALSE;

			galago_signal_emit(_core, "unregistered");
		}

		dbus_error_init(&error);

		if (!start_service(_core->dbus_conn, GALAGO_DBUS_SERVICE, 0, NULL,
						   &error))
		{
			galago_log_error("Received ServiceDoesNotExist, and cannot "
							 "re-activate daemon. Disconnecting for now.\n");
			dbus_error_free(&error);
		}

		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	else if (dbus_message_is_error(message, DBUS_ERROR_LIMITS_EXCEEDED))
	{
		char *str;
		DBusMessageIter iter;

		dbus_message_iter_init(message, &iter);
		galago_dbus_message_iter_get_string(&iter, str);

		galago_log_error("Received LimitsExceeded: %s\n", str);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
		if (str != NULL)
			dbus_free(str);
#endif

		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	galago_log_error("D-BUS Error: %s\n", error_name);

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static galago_bool
_dbus_fdo_filter_func(DBusConnection *dbus_conn, DBusMessage *message,
					  void *user_data)
{
#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
	DBusMessageIter iter;

	/* D-BUS <= 0.22 */
	if (dbus_message_is_signal(message, DBUS_INTERFACE_DBUS, "ServiceCreated"))
	{
		char *dbus_service;

		dbus_message_iter_init(message, &iter);

		dbus_service = dbus_message_iter_get_string(&iter);

		if (!strcmp(dbus_service, GALAGO_DBUS_SERVICE) && !galago_is_daemon())
			reregister_client();

		dbus_free(dbus_service);
	}
	/* D-BUS <= 0.22 */
	else if (dbus_message_is_signal(message, DBUS_INTERFACE_DBUS,
									"ServiceDeleted"))
	{
		char *dbus_service;

		dbus_message_iter_init(message, &iter);

		dbus_service = dbus_message_iter_get_string(&iter);

		if (!strcmp(dbus_service, GALAGO_DBUS_SERVICE) &&
			!galago_is_daemon())
		{
			galago_context_clear_objects(FALSE);

			galago_core_set_watch_all(FALSE);

			_core->daemon_active = FALSE;
			_core->registered    = FALSE;

			galago_signal_emit(_core, "unregistered");

			if (galago_core_get_exit_with_daemon())
			{
				galago_log_info("Exitting with daemon\n");
				exit(0);
			}
		}

		dbus_free(dbus_service);
	}
	/*
	 * ServiceOwnerChanged: D-BUS == 0.23
	 * NameOwnerChanged:    D-BUS >  0.23
	 */
	else
#endif
	if (dbus_message_is_signal(message, DBUS_INTERFACE_DBUS,
							   "ServiceOwnerChanged") ||
		dbus_message_is_signal(message, DBUS_INTERFACE_DBUS,
							   "NameOwnerChanged"))
	{
		DBusError error;
		char *dbus_service;
		char *old_owner;
		char *new_owner;

		dbus_error_init(&error);

		if (dbus_message_get_args(message, &error,
								  DBUS_TYPE_STRING, &dbus_service,
								  DBUS_TYPE_STRING, &old_owner,
								  DBUS_TYPE_STRING, &new_owner,
								  DBUS_TYPE_INVALID) &&
			!strcmp(dbus_service, GALAGO_DBUS_SERVICE) &&
			!galago_is_daemon())
		{
			galago_bool old_owner_good = (old_owner && *old_owner != '\0');
			galago_bool new_owner_good = (new_owner && *new_owner != '\0');

			galago_log_info("Service owner changed.\n");

			if (!old_owner_good && new_owner_good)
			{
				if (galago_core_get_exit_with_daemon())
				{
					galago_log_info("Exitting with daemon\n");
					exit(0);
				}
				else
				{
					galago_log_info("Reregistering connection.\n");
					reregister_client();
				}
			}

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
			dbus_free(dbus_service);
			dbus_free(old_owner);
			dbus_free(new_owner);
#endif
		}
	}
	else
		return FALSE;

	return TRUE;
}

static DBusHandlerResult
filter_func(DBusConnection *dbus_conn, DBusMessage *message, void *user_data)
{
	DBusMessageIter iter;
	const char *path;

	if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_ERROR)
		return _handle_dbus_error(dbus_conn, message, user_data);

	if (_dbus_fdo_filter_func(dbus_conn, message, user_data))
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	path = dbus_message_get_path(message);

	if (path == NULL ||
		galago_context_get_object(path) == NULL)
	{
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	dbus_message_iter_init(message, &iter);

	/*
	 * Core
	 */
	if (dbus_message_is_signal(message, GALAGO_DBUS_CORE_INTERFACE,
							   "ServiceAdded"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_CLASS_SERVICE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_CORE_INTERFACE,
									"ServiceRemoved"))
	{
		GalagoService *service;

		service = galago_dbus_message_iter_get_object(&iter,
													  GALAGO_CLASS_SERVICE);

		galago_object_unref(service);
	}
	/*
	 * Service
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_SERVICE_INTERFACE,
									"AccountAdded"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_CLASS_ACCOUNT);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_SERVICE_INTERFACE,
									"AccountRemoved"))
	{
		GalagoAccount *account;

		account = galago_dbus_message_iter_get_object(&iter,
													  GALAGO_CLASS_ACCOUNT);

		galago_object_unref(account);
	}
	/*
	 * Account
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"PresenceSet"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_CLASS_PRESENCE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"PresenceUnset"))
	{
		GalagoAccount *account;

		account = galago_context_get_object(dbus_message_get_path(message));

		if (account != NULL)
			galago_account_set_presence(account, NULL);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"Connected"))
	{
		GalagoAccount *account;

		account = galago_context_get_object(dbus_message_get_path(message));

		if (account != NULL)
			galago_account_set_connected(account, TRUE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"Disconnected"))
	{
		GalagoAccount *account;

		account = galago_context_get_object(dbus_message_get_path(message));

		if (account != NULL)
			galago_account_set_connected(account, FALSE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"DisplayNameChanged"))
	{
		GalagoAccount *account;
		char *display_name;

		account = galago_context_get_object(dbus_message_get_path(message));

		if (account != NULL)
		{
			display_name = galago_dbus_message_iter_get_string_or_nil(&iter);

			galago_account_set_display_name(account, display_name);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
			if (display_name != NULL)
				dbus_free(display_name);
#endif
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"AvatarSet"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_CLASS_AVATAR);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"AvatarUnset"))
	{
		GalagoAccount *account;

		account = galago_context_get_object(dbus_message_get_path(message));

		if (account != NULL)
			galago_account_set_avatar(account, NULL);
	}
	/*
	 * Presence
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PRESENCE_INTERFACE,
									"IdleUpdated"))
	{
		GalagoPresence *presence;

		presence = galago_context_get_object(dbus_message_get_path(message));

		if (presence != NULL)
		{
			galago_bool idle;
			dbus_uint32_t idle_time = 0;

			galago_dbus_message_iter_get_boolean(&iter, idle);

			if (idle)
			{
				dbus_message_iter_next(&iter);

				galago_dbus_message_iter_get_uint32(&iter, idle_time);
			}

			galago_presence_set_idle(presence, idle, idle_time);
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PRESENCE_INTERFACE,
									"StatusAdded"))
	{
		GalagoPresence *presence;
		GalagoStatus *status;

		presence = galago_context_get_object(dbus_message_get_path(message));

		if (presence != NULL)
		{
			const char *status_id;

			status = galago_dbus_message_iter_get_object(&iter,
														 GALAGO_CLASS_STATUS);

			status_id = galago_status_get_id(status);

			if (!galago_presence_has_status(presence, status_id))
				galago_presence_add_status(presence, status);
			else
				galago_object_unref(status);
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PRESENCE_INTERFACE,
									"StatusRemoved"))
	{
		GalagoPresence *presence;
		char *status_id;

		presence = galago_context_get_object(dbus_message_get_path(message));

		if (presence != NULL)
		{
			galago_dbus_message_iter_get_string(&iter, status_id);

			galago_presence_remove_status(presence, status_id);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
			dbus_free(status_id);
#endif
		}
	}
	/*
	 * Person
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PERSON_INTERFACE,
									"PropertySet"))
	{
		GalagoPerson *person;
		char *name;

		person = galago_context_get_object(dbus_message_get_path(message));

		if (person != NULL)
		{
			int arg_type;

			galago_dbus_message_iter_get_string(&iter, name);
			dbus_message_iter_next(&iter);

			arg_type = dbus_message_iter_get_arg_type(&iter);

			if (arg_type == DBUS_TYPE_STRING)
			{
				char *value;
				galago_dbus_message_iter_get_string(&iter, value);
				galago_person_set_property_string(person, name, value);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
				dbus_free(value);
#endif
			}
			else if (arg_type == DBUS_TYPE_BOOLEAN)
			{
				galago_bool value;
				galago_dbus_message_iter_get_boolean(&iter, value);
				galago_person_set_property_bool(person, name, value);
			}
			else if (arg_type == DBUS_TYPE_UINT32)
			{
				dbus_uint32_t value;
				galago_dbus_message_iter_get_uint32(&iter, value);
				galago_person_set_property_uint32(person, name, value);
			}

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
			dbus_free(name);
#endif
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PERSON_INTERFACE,
									"PropertyRemoved"))
	{
		GalagoPerson *person;

		person = galago_context_get_object(dbus_message_get_path(message));

		if (person != NULL)
		{
			char *name;

			galago_dbus_message_iter_get_string(&iter, name);

			galago_person_remove_property(person, name);
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PERSON_INTERFACE,
									"PhotoSet"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_CLASS_AVATAR);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PERSON_INTERFACE,
									"PhotoUnset"))
	{
		GalagoPerson *person;

		person = galago_context_get_object(dbus_message_get_path(message));

		if (person != NULL)
			galago_person_set_photo(person, NULL);
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/**************************************************************************
 * Core API
 **************************************************************************/
static galago_bool
_galago_core_connect(void)
{
	galago_bool service_activated = TRUE;
	DBusError error;

	dbus_error_init(&error);

	_core->dbus_conn = dbus_bus_get(DBUS_BUS_SESSION, &error);

	if (_core->dbus_conn == NULL)
	{
		galago_log_error("Error connecting to session bus: %s\n",
						 error.message);
		dbus_error_free(&error);

		return FALSE;
	}

	dbus_connection_set_exit_on_disconnect(_core->dbus_conn, FALSE);

	if (_core->daemon)
	{
		dbus_error_free(&error);

		return TRUE;
	}

	if (!start_service(_core->dbus_conn, GALAGO_DBUS_SERVICE, 0, NULL, &error))
	{
		galago_log_error("Error activating Galago service: %s\n",
						 error.message);
		dbus_error_free(&error);

		dbus_error_init(&error);

		service_activated = FALSE;
	}

	if (!dbus_connection_add_filter(_core->dbus_conn,
									filter_func, NULL, NULL))
	{
		galago_log_error("Error creating core handler\n");
		dbus_error_free(&error);

		return FALSE;
	}

	dbus_bus_add_match(_core->dbus_conn,
					   "type='signal',"
					   "interface='" DBUS_INTERFACE_DBUS "',"
					   "sender='" DBUS_SERVICE_DBUS "'",
					   &error);

	if (dbus_error_is_set(&error))
	{
		galago_log_error("Error subscribing to signals: %s\n", error.message);

		dbus_error_free(&error);

		return FALSE;
	}

	dbus_error_free(&error);

	_core->filters_added = TRUE;

	_galago_dbus_register_connection();

	return TRUE;
}

static void
_galago_core_disconnect(void)
{
	galago_bool unregistered_emitted = FALSE;

	if (_core->dbus_conn != NULL)
	{
		if (!_core->daemon && galago_is_registered())
		{
			_galago_dbus_unregister_connection();

			unregistered_emitted = TRUE;
		}

		if (_core->dbus_conn != NULL)
		{
			if (_core->filters_added)
			{
				dbus_connection_remove_filter(_core->dbus_conn,
											  filter_func, NULL);
			}

			dbus_connection_disconnect(_core->dbus_conn);
			dbus_connection_dispatch(_core->dbus_conn);
			dbus_connection_unref(_core->dbus_conn);
			_core->dbus_conn = NULL;
		}
	}

	_core->filters_added = FALSE;
	_core->registered    = FALSE;

	if (!unregistered_emitted)
		galago_signal_emit(_core, "unregistered");
}

#ifdef HAVE_SIGNAL_H
static void
_sigint_cb(int unused)
{
	signal(SIGINT, SIG_DFL);

	galago_uninit();

	raise(SIGINT);
}
#endif

galago_bool
galago_init(const char *name, galago_bool feed)
{
	GalagoContext *context;

	galago_return_val_if_fail(name  != NULL, FALSE);
	galago_return_val_if_fail(*name != '\0', FALSE);

	if (_core != NULL)
	{
		galago_object_ref(_core);

		return TRUE;
	}

	context = galago_context_new();
	galago_context_set_ops(context, &context_ops);

	galago_context_push(context);

	_core = galago_object_new(GALAGO_CLASS_CORE);

	_core->app_name = strdup(name);
	_core->feed = feed;

	if (!strcmp(name, GALAGO_DAEMON_ID))
		_core->daemon = TRUE;

	if (!_galago_core_connect())
		_galago_core_disconnect();

	if (first_init)
	{
#ifdef HAVE_ATEXIT
		atexit(galago_uninit);
#endif /* HAVE_ATEXIT */

#ifdef HAVE_SIGNAL_H
		if (signal(SIGINT, _sigint_cb) == SIG_ERR)
		{
			galago_log_warning("Unable to register SIGINT "
							   "signal handler\n");
		}
#endif /* HAVE_SIGNAL_H_ */

		first_init = TRUE;
	}

	return TRUE;
}

galago_bool
galago_init_with_mainloop(const char *name, galago_bool feed,
						  void (*func)(DBusConnection *, void *), void *data)
{
	galago_return_val_if_fail(name != NULL, FALSE);
	galago_return_val_if_fail(func != NULL, FALSE);

	if (galago_init(name, feed) && galago_is_connected())
	{
		func(galago_core_get_dbus_conn(), data);

		return TRUE;
	}

	return FALSE;
}

void
galago_uninit(void)
{
	if (!galago_is_initted())
		return;

	_core = galago_object_unref(_core);
}

galago_bool
galago_is_initted(void)
{
	return (_core != NULL);
}

galago_bool
galago_is_connected(void)
{
	return (galago_is_initted() && _core->dbus_conn != NULL &&
			(_core->registering_connection || _core->uid != NULL ||
			 galago_is_daemon()) &&
			dbus_connection_get_is_connected(_core->dbus_conn));
}

galago_bool
galago_is_registered(void)
{
	return _core->registered;
}

galago_bool
galago_is_daemon_active(void)
{
	return (galago_is_daemon() || _core->daemon_active);
}

galago_bool
galago_is_daemon(void)
{
	return (_core->daemon);
}

DBusConnection *
galago_core_get_dbus_conn(void)
{
	galago_return_val_if_fail(galago_is_initted(), NULL);

	return _core->dbus_conn;
}

const char *
galago_core_get_uid(void)
{
	galago_return_val_if_fail(galago_is_initted(),   NULL);
	galago_return_val_if_fail(galago_is_connected(), NULL);

	return _core->uid;
}

const char *
galago_core_get_obj_path(void)
{
	galago_return_val_if_fail(galago_is_initted(),   NULL);
	galago_return_val_if_fail(galago_is_connected(), NULL);

	return _core->conn_obj_path;
}

galago_bool
galago_core_is_feed(void)
{
	galago_return_val_if_fail(galago_is_initted(),   FALSE);
	galago_return_val_if_fail(galago_is_connected(), FALSE);

	return _core->feed;
}

#define ADD_SIGNAL_MATCH(iface) \
	dbus_bus_add_match(dbus_conn, \
					   "type='signal'," \
					   "interface='" iface "'," \
					   "sender='" GALAGO_DBUS_SERVICE "'", \
					   &error); \
	if (dbus_error_is_set(&error)) \
	{ \
		galago_log_error("Error subscribing to %s signal: %s\n", \
						 (iface), error.message); \
		return; \
	}
#define REMOVE_SIGNAL_MATCH(iface) \
	dbus_bus_remove_match(dbus_conn, \
					   "type='signal'," \
					   "interface='" iface "'," \
					   "sender='" GALAGO_DBUS_SERVICE "'", \
					   &error); \
	if (dbus_error_is_set(&error)) \
	{ \
		galago_log_error("Error unsubscribing from %s signal: %s\n", \
						 (iface), error.message); \
		return; \
	}

void
galago_core_set_watch_all(galago_bool watch_all)
{
	DBusError error;
	DBusConnection *dbus_conn;

	galago_return_if_fail(galago_is_initted());

	if (_core->watch_all == watch_all)
		return;

	if (!galago_is_connected())
		return;

	dbus_conn = galago_core_get_dbus_conn();

	_core->watch_all = watch_all;

	dbus_error_init(&error);

	if (watch_all)
	{
		ADD_SIGNAL_MATCH(GALAGO_DBUS_CORE_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_SERVICE_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_ACCOUNT_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_AVATAR_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_PRESENCE_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_PHOTO_INTERFACE);
	}
	else
	{
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_CORE_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_SERVICE_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_ACCOUNT_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_AVATAR_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_PRESENCE_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_PHOTO_INTERFACE);
	}
}

#undef ADD_SIGNAL_MATCH
#undef REMOVE_SIGNAL_MATCH

galago_bool
galago_core_get_watch_all(void)
{
	galago_return_val_if_fail(galago_is_initted(),   FALSE);
	galago_return_val_if_fail(galago_is_connected(), FALSE);

	return _core->watch_all;
}

void
galago_core_set_exit_with_daemon(galago_bool exit_with_daemon)
{
	galago_return_if_fail(galago_is_initted());

	_core->exit_with_daemon = exit_with_daemon;
}

galago_bool
galago_core_get_exit_with_daemon(void)
{
	galago_return_val_if_fail(galago_is_initted(), FALSE);

	return _core->exit_with_daemon;
}

void
galago_core_add_service(GalagoService *service)
{
	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(service != NULL);

	galago_context_push(galago_object_get_context(_core));
	galago_context_add_service(service);
	galago_context_pop();
}

void
galago_core_remove_service(GalagoService *service)
{
	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(service != NULL);

	galago_context_push(galago_object_get_context(_core));
	galago_context_remove_service(service);
	galago_context_pop();
}

GalagoService *
galago_core_get_service(const char *id, galago_bool native, galago_bool query)
{
	GalagoService *service;

	galago_return_val_if_fail(galago_is_initted(), NULL);
	galago_return_val_if_fail(id != NULL,          NULL);

	galago_context_push(galago_object_get_context(_core));

	service = galago_context_get_service(id, native);

	if (query && service == NULL && !native && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(_core);

		galago_signal_context_freeze(signal_context);
		service = galago_dbus_send_message_with_reply(_core, "GetService",
			galago_value_new(GALAGO_TYPE_OBJECT, NULL, GALAGO_CLASS_SERVICE),
			galago_value_new(GALAGO_TYPE_STRING, &id, NULL),
			NULL);
		galago_signal_context_thaw(signal_context);
	}

	galago_context_pop();

	return service;
}

const GalagoList *
galago_core_get_services(galago_bool native, galago_bool query)
{
	const GalagoList *services;

	galago_context_push(galago_object_get_context(_core));
	services = galago_context_get_services(native);

	if (query && !native && !galago_is_daemon() && galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(_core);
		GalagoList *temp;

		galago_signal_context_freeze(signal_context);

		temp = galago_dbus_send_message_with_reply(_core, "GetServices",
			galago_value_new_list(GALAGO_TYPE_OBJECT, NULL,
								  GALAGO_CLASS_SERVICE),
			NULL);
		galago_list_destroy(temp);

		services = galago_context_get_services(native);

		galago_signal_context_thaw(signal_context);
	}

	galago_context_pop();

	return services;
}

void
galago_core_add_person(GalagoPerson *person)
{
	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(person != NULL);

	galago_context_push(galago_object_get_context(_core));
	galago_context_add_person(person);
	galago_context_pop();
}

void
galago_core_remove_person(GalagoPerson *person)
{
	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(person != NULL);

	galago_context_push(galago_object_get_context(_core));
	galago_context_remove_person(person);
	galago_context_pop();
}

GalagoPerson *
galago_core_get_person(const char *id, galago_bool native, galago_bool query)
{
	GalagoPerson *person;

	galago_return_val_if_fail(galago_is_initted(), NULL);
	galago_return_val_if_fail(id != NULL,          NULL);

	galago_context_push(galago_object_get_context(_core));

	person = galago_context_get_person(id, native);

	if (query && person == NULL && !native && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(_core);

		galago_signal_context_freeze(signal_context);
		person = galago_dbus_send_message_with_reply(_core, "GetPerson",
			galago_value_new(GALAGO_TYPE_OBJECT, NULL, GALAGO_CLASS_PERSON),
			galago_value_new(GALAGO_TYPE_STRING, &id, NULL),
			NULL);
		galago_signal_context_thaw(signal_context);
	}

	galago_context_pop();

	return person;
}

const GalagoList *
galago_core_get_people(galago_bool native, galago_bool query)
{
	const GalagoList *people;

	galago_context_push(galago_object_get_context(_core));
	people = galago_context_get_people(native);

	if (query && !native && !galago_is_daemon() && galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(_core);
		GalagoList *temp;

		galago_signal_context_freeze(signal_context);

		temp = galago_dbus_send_message_with_reply(_core, "GetPeople",
			galago_value_new_list(GALAGO_TYPE_OBJECT, NULL,
								  GALAGO_CLASS_PERSON),
			NULL);
		galago_list_destroy(temp);

		people = galago_context_get_people(native);

		galago_signal_context_thaw(signal_context);
	}

	galago_context_pop();

	return people;
}

GalagoPerson *
galago_core_get_me(galago_bool native, galago_bool query)
{
	GalagoPerson *me;

	galago_return_val_if_fail(galago_is_initted(), NULL);

	galago_context_push(galago_object_get_context(_core));

	me = galago_core_get_person(GALAGO_ME_ID, native, FALSE);

	if (query && me == NULL && !native && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(_core);

		galago_signal_context_freeze(signal_context);
		me = galago_dbus_send_message_with_reply(_core, "GetMe",
			galago_value_new(GALAGO_TYPE_OBJECT, NULL, GALAGO_CLASS_PERSON),
			NULL);
		galago_signal_context_thaw(signal_context);
	}

	galago_context_pop();

	return me;
}

GalagoCore *
galago_get_core(void)
{
	return _core;
}

/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static galago_bool
_galago_dbus_register_connection(void)
{
	GalagoList *list, *return_list = NULL;

	return_list = galago_list_append(return_list,
		galago_value_new(GALAGO_TYPE_STRING, NULL, NULL));
	return_list = galago_list_append(return_list,
		galago_value_new(GALAGO_TYPE_STRING, NULL, NULL));

	_core->registering_connection = TRUE;

	list = galago_dbus_send_message_with_reply_list(
		_core, "Register", return_list,
		galago_value_new(GALAGO_TYPE_STRING,  &_core->app_name, NULL),
		galago_value_new(GALAGO_TYPE_BOOLEAN, &_core->feed,     NULL),
		NULL);

	_core->registering_connection = FALSE;

	if (list == NULL)
	{
		galago_log_error("Error registering local Galago connection.\n");

		_core->daemon_active = FALSE;

		return FALSE;
	}

	/* TODO: Do safety checks. */
	_core->uid           = strdup((char *)list->data);
	_core->conn_obj_path = strdup((char *)list->next->data);
	galago_context_set_obj_path_prefix(_core->conn_obj_path);

	galago_list_destroy(list);

	_core->daemon_active = TRUE;
	galago_core_set_watch_all(TRUE);
	_core->registered = TRUE;

	galago_signal_emit(_core, "registered");

	return TRUE;
}

static void
_galago_dbus_unregister_connection(void)
{
	const char *obj_path = galago_core_get_obj_path();

	if (!galago_is_connected() || obj_path == NULL)
		return;

	galago_dbus_send_message(_core, "Unregister",
		galago_value_new(GALAGO_TYPE_STRING, &obj_path, NULL),
		NULL);

	dbus_connection_flush(_core->dbus_conn); /* Keep this here! */

	_core->registered = FALSE;

	galago_signal_emit(_core, "unregistered");
}

static void
_galago_dbus_core_add_service(GalagoService *service)
{
	DBusConnection *dbus_conn;
	DBusMessage *message;
	DBusMessageIter iter;

	galago_signal_emit(_core, "service-added", service);
	galago_signal_emit(_core, "updated");

	if (!galago_is_connected() || !galago_core_is_feed() ||
		!galago_service_is_native(service))
	{
		return;
	}

	dbus_conn = galago_core_get_dbus_conn();

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_core_get_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "AddService");

	galago_return_if_fail(message != NULL);

	dbus_message_set_no_reply(message, TRUE);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, service);

	dbus_connection_send(dbus_conn, message, NULL);
	dbus_message_unref(message);
}

static void
_galago_dbus_core_remove_service(GalagoService *service)
{
	DBusConnection *dbus_conn;
	DBusMessage *message;
	DBusMessageIter iter;

	galago_signal_emit(_core, "service-removed", service);
	galago_signal_emit(_core, "updated");

	if (!galago_is_connected() || !galago_core_is_feed() ||
		!galago_service_is_native(service))
	{
		return;
	}

	dbus_conn = galago_core_get_dbus_conn();

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_core_get_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "RemoveService");

	galago_return_if_fail(message != NULL);

	dbus_message_set_no_reply(message, TRUE);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, service);

	dbus_connection_send(dbus_conn, message, NULL);
	dbus_message_unref(message);
}

static void
_galago_dbus_core_add_person(GalagoPerson *person)
{
	DBusConnection *dbus_conn;
	DBusMessage *message;
	DBusMessageIter iter;

	galago_signal_emit(_core, "person-added", person);
	galago_signal_emit(_core, "updated");

	if (!galago_is_connected() || !galago_core_is_feed() ||
		!galago_person_is_native(person))
	{
		return;
	}

	dbus_conn = galago_core_get_dbus_conn();

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_core_get_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "AddPerson");

	galago_return_if_fail(message != NULL);

	dbus_message_set_no_reply(message, TRUE);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, person);

	dbus_connection_send(dbus_conn, message, NULL);
	dbus_message_unref(message);
}

static void
_galago_dbus_core_remove_person(GalagoPerson *person)
{
	DBusConnection *dbus_conn;
	DBusMessage *message;
	DBusMessageIter iter;

	galago_signal_emit(_core, "person-removed", person);
	galago_signal_emit(_core, "updated");

	if (!galago_is_connected() || !galago_core_is_feed() ||
		!galago_person_is_native(person))
	{
		return;
	}

	dbus_conn = galago_core_get_dbus_conn();

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_core_get_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "RemovePerson");

	galago_return_if_fail(message != NULL);

	dbus_message_set_no_reply(message, TRUE);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, person);

	dbus_connection_send(dbus_conn, message, NULL);
	dbus_message_unref(message);
}
