/**
 * @file libgalago/galago-hashtable.h Hash Table API
 *
 * @Copyright (C) 2004-2005 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_HASHTABLE_H_
#define _GALAGO_HASHTABLE_H_

typedef struct _GalagoHashTable GalagoHashTable;

#include <libgalago/galago-list.h>
#include <libgalago/galago-types.h>

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Hash Table API                                                  */
/**************************************************************************/
/*@{*/

/**
 * Creates a new hash table.
 *
 * @param hash_func      The hashing function.
 * @param key_equal_func Determines if the keys are equal.
 *
 * @return The hash table.
 */
GalagoHashTable *galago_hash_table_new(GalagoHashFunc hash_func,
									   GalagoEqualFunc key_equal_func);
/**
 * Creates a new hash table.
 *
 * @param hash_func          The hashing function.
 * @param key_equal_func     Determines if the keys are equal.
 * @param destroy_key_func   Destroys a key.
 * @param destroy_value_func Destroys a value.
 *
 * @return The hash table.
 */
GalagoHashTable *galago_hash_table_new_full(GalagoHashFunc hash_func,
											GalagoEqualFunc key_equal_func,
											GalagoFreeFunc destroy_key_func,
											GalagoFreeFunc destroy_value_func);

/**
 * Destroys a hash table.
 *
 * @param hash_table The hash table to destroy.
 */
void galago_hash_table_destroy(GalagoHashTable *hash_table);

/**
 * Clears a hashtable of all data.
 *
 * @param hash_table The hash table to clear.
 */
void galago_hash_table_clear(GalagoHashTable *hash_table);

/**
 * Inserts an item into a hash table.
 *
 * @param hash_table The hash table.
 * @param key        The key.
 * @param value      The value to insert.
 */
void galago_hash_table_insert(GalagoHashTable *hash_table, void *key,
							  void *value);

/**
 * Replaces an item in a hash table.
 *
 * @param hash_table The hash table.
 * @param key        The key.
 * @param value      The value to insert.
 */
void galago_hash_table_replace(GalagoHashTable *hash_table, void *key,
							   void *value);

/**
 * Removes an item from a hash table.
 *
 * @param hash_table The hash table.
 * @param key        The key.
 */
void galago_hash_table_remove(GalagoHashTable *hash_table, const void *key);

/**
 * Looks up and returns data from a hash table.
 *
 * @param hash_table The hash table.
 * @param key        The key.
 *
 * @return The value associated with that key, if any.
 */
void *galago_hash_table_lookup(const GalagoHashTable *hash_table,
							   const void *key);

/**
 * Returns whether or not data with the specified key exists in a hash table.
 *
 * @param hash_table The hash table.
 * @param key        The key.
 *
 * @return TRUE if any value exists, or FALSE.
 */
galago_bool galago_hash_table_exists(const GalagoHashTable *hash_table,
									 const void *key);

/**
 * Returns a list of all keys in a hash table.
 *
 * The returned list must be freed.
 *
 * @param hash_table The hash table.
 *
 * @return The list of keys in the hash table. This must be freed.
 */
GalagoList *galago_hash_table_get_keys(const GalagoHashTable *hash_table);

/**
 * Returns a list of all values in a hash table.
 *
 * The returned list must be freed.
 *
 * @param hash_table The hash table.
 *
 * @return The list of values in the hash table. This must be freed.
 */
GalagoList *galago_hash_table_get_values(const GalagoHashTable *hash_table);

/**
 * Calls a function on every key and value in a hash table.
 *
 * @param hash_table The hash table.
 * @param func       The function.
 * @param user_data  User-specified data.
 */
void galago_hash_table_foreach(const GalagoHashTable *hash_table,
							   GalagoHForEachFunc func, void *user_data);

/**
 * Calls a function on every key in a hash table.
 *
 * @param hash_table The hash table.
 * @param func       The function.
 * @param user_data  User-specified data.
 */
void galago_hash_table_foreach_key(const GalagoHashTable *hash_table,
								   GalagoForEachFunc func, void *user_data);

/**
 * Calls a function on every value in a hash table.
 *
 * @param hash_table The hash table.
 * @param func       The function.
 * @param user_data  User-specified data.
 */
void galago_hash_table_foreach_value(const GalagoHashTable *hash_table,
									 GalagoForEachFunc func, void *user_data);

/**
 * Returns the size of the hash table.
 *
 * @param hash_table The hash table.
 *
 * @return The number of items in the hash table.
 */
size_t galago_hash_table_get_size(const GalagoHashTable *hash_table);

/*@}*/

/**************************************************************************/
/** @name Hash Functions                                                  */
/**************************************************************************/
/*@{*/

/**
 * Checks if two strings are equal.
 *
 * @param a The first string.
 * @param b The second string.
 *
 * @return TRUE if the strings are equal, or FALSE.
 */
galago_bool galago_str_equal(const void *a, const void *b);

/**
 * Returns the hash of a string.
 *
 * @param str The string.
 *
 * @return The hash of the string.
 */
unsigned int galago_str_hash(const void *str);

/**
 * Checks if two integers are equal.
 *
 * @param a The first integer.
 * @param b The second integer.
 *
 * @return TRUE if the integers are equal, or FALSE.
 */
galago_bool galago_int_equal(const void *a, const void *b);

/**
 * Returns the hash of an integer.
 *
 * @param i The integer.
 *
 * @return The hash of the integer.
 */
unsigned int galago_int_hash(const void *i);

/**
 * Checks if two pointers are equal.
 *
 * @param a The first pointer.
 * @param b The second pointer.
 *
 * @return TRUE if the pointers are equal.
 */
galago_bool galago_direct_equal(const void *a, const void *b);

/**
 * Returns the hash of a pointer.
 *
 * @param ptr The pointer.
 *
 * @return The hash of the pointer.
 */
unsigned int galago_direct_hash(const void *ptr);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_HASHTABLE_H_ */
