/**
 * @file libgalago/galago-object.c Galago Object API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-object.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-marshal.h>
#include <libgalago/galago-signals.h>
#include <libgalago/galago-utils.h>
#include <stdio.h>
#include <string.h>

struct _GalagoObjectClassPrivate
{
	GalagoObjectClass *parent_class;
	unsigned long serial;

	char *name;
	char *dbus_iface;

	const GalagoObjectClassInfo *info;
};

struct _GalagoObjectPrivate
{
	GalagoObjectClass *klass;
	unsigned int ref_count;
	GalagoContext *context;

	unsigned long flags;

	char *dbus_path;

	galago_bool watched;

	GalagoHashTable *user_data;

	galago_bool finalizing;
};

static unsigned long last_serial = 0;

/**************************************************************************
 * Class API
 **************************************************************************/
GalagoObjectClass *
galago_class_register(GalagoObjectClass *parent_class, const char *name,
					  const char *dbus_iface, const GalagoObjectClassInfo *info)
{
	GalagoObjectClass *klass;
	GalagoObjectClass *tmp_klass;
	GalagoList *class_inits = NULL, *l;

	galago_return_val_if_fail(parent_class == NULL ||
							  GALAGO_IS_CLASS(parent_class), NULL);
	galago_return_val_if_fail(name  != NULL, NULL);
	galago_return_val_if_fail(*name != '\0', NULL);
	galago_return_val_if_fail(info  != NULL, NULL);
	galago_return_val_if_fail(info->class_size  >= sizeof(GalagoObjectClass),
							  NULL);
	galago_return_val_if_fail(info->object_size >= sizeof(GalagoObject), NULL);

	klass = galago_malloc0(info->class_size);

	klass->magic = GALAGO_CLASS_MAGIC;

	klass->priv = galago_new0(GalagoObjectClassPrivate, 1);

	klass->priv->parent_class = parent_class;
	klass->priv->serial       = ++last_serial;
	klass->priv->name         = strdup(name);
	klass->priv->info         = info;

	if (dbus_iface != NULL)
		klass->priv->dbus_iface = strdup(dbus_iface);

	klass->signal_context = galago_signal_context_new();

	galago_signal_register(klass->signal_context, "destroy",
						   galago_marshal_VOID, 0);

	for (tmp_klass = klass;
		 tmp_klass != NULL;
		 tmp_klass = tmp_klass->priv->parent_class)
	{
		if (tmp_klass->priv->info->class_init != NULL)
		{
			class_inits =
				galago_list_prepend(class_inits,
									tmp_klass->priv->info->class_init);
		}
	}

	for (l = class_inits; l != NULL; l = l->next)
		((GalagoObjectClassInitFunc)l->data)(klass);

	galago_list_destroy(class_inits);

	return klass;
}

GalagoObjectClass *
galago_class_get_parent(const GalagoObjectClass *klass)
{
	galago_return_val_if_fail(klass != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_CLASS(klass), NULL);

	return klass->priv->parent_class;
}

const char *
galago_class_get_name(const GalagoObjectClass *klass)
{
	galago_return_val_if_fail(klass != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_CLASS(klass), NULL);

	return klass->priv->name;
}

const char *
galago_class_get_dbus_iface(const GalagoObjectClass *klass)
{
	galago_return_val_if_fail(klass != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_CLASS(klass), NULL);

	return klass->priv->dbus_iface;
}

GalagoSignalContext *
galago_class_get_signal_context(const GalagoObjectClass *klass)
{
	galago_return_val_if_fail(klass != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_CLASS(klass), NULL);

	return klass->signal_context;
}


/**************************************************************************
 * Object API
 **************************************************************************/
void
_galago_object_finalize(GalagoObject *object)
{
	GalagoObjectClass *klass;

	if (GALAGO_OBJECT_HAS_FLAG(object, GALAGO_FLAG_FINALIZING))
		return;

	GALAGO_OBJECT_SET_FLAG(object, GALAGO_FLAG_FINALIZING);

	galago_signal_emit(object, "destroy");

	galago_context_push(object->priv->context);

	for (klass = object->priv->klass;
		 klass != NULL;
		 klass = klass->priv->parent_class)
	{
		if (klass->finalize != NULL)
			klass->finalize(object);
	}

	if (galago_object_is_watched(object))
		galago_object_set_watch(object, FALSE);

	if (object->priv->user_data != NULL)
		galago_hash_table_destroy(object->priv->user_data);

	if (object->priv->dbus_path != NULL)
	{
		galago_context_remove_object(object);

		free(object->priv->dbus_path);
	}

	free(object->priv);
	free(object);

	galago_context_pop();
}

void *
galago_object_new(GalagoObjectClass *klass)
{
	GalagoObject *object;
	GalagoObjectClass *tmp_klass;
	GalagoList *object_inits = NULL, *l;

	galago_return_val_if_fail(klass != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_CLASS(klass), NULL);

	if (klass->priv->info->flags & GALAGO_FLAG_ABSTRACT)
	{
		galago_log_error("Unable to create instance of abstract class '%s'\n",
						 klass->priv->name);

		return NULL;
	}

	object = galago_malloc0(klass->priv->info->object_size);

	object->magic = GALAGO_OBJECT_MAGIC;

	object->priv = galago_new0(GalagoObjectPrivate, 1);

	object->priv->context   = galago_context_get();
	object->priv->klass     = klass;
	object->priv->ref_count = 1;

	for (tmp_klass = klass;
		 tmp_klass != NULL;
		 tmp_klass = tmp_klass->priv->parent_class)
	{
		if (tmp_klass->priv->info->object_init != NULL)
		{
			object_inits =
				galago_list_prepend(object_inits,
									tmp_klass->priv->info->object_init);
		}
	}

	for (l = object_inits; l != NULL; l = l->next)
		((GalagoObjectInitFunc)l->data)(object);

	galago_list_destroy(object_inits);

	return object;
}

void *
galago_object_ref(void *object)
{
	galago_return_val_if_fail(object != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	((GalagoObject *)object)->priv->ref_count++;

	return object;
}

void *
galago_object_unref(void *_object)
{
	GalagoObject *object = (GalagoObject *)_object;

	galago_return_val_if_fail(object != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	if (GALAGO_OBJECT_HAS_FLAG(object, GALAGO_FLAG_FINALIZING))
		return NULL;

	galago_return_val_if_fail(object->priv->ref_count > 0, NULL);

	object->priv->ref_count--;

	if (object->priv->ref_count == 0)
	{
		_galago_object_finalize(object);

		return NULL;
	}

	return object;
}

GalagoObjectClass *
galago_object_get_class(const void *_object)
{
	galago_return_val_if_fail(_object != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(_object), NULL);

	return GALAGO_OBJECT(_object)->priv->klass;
}

void
galago_object_set_flags(void *_object, unsigned long flags)
{
	galago_return_if_fail(_object != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(_object));

	GALAGO_OBJECT(_object)->priv->flags = flags;
}

unsigned long
galago_object_get_flags(const void *_object)
{
	galago_return_val_if_fail(_object != NULL,           0);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(_object), 0);

	return GALAGO_OBJECT(_object)->priv->flags;
}

void
galago_object_set_dbus_path(void *_object, const char *obj_path)
{
	GalagoObject *object;

	galago_return_if_fail(_object != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(_object));

	object = GALAGO_OBJECT(_object);

	galago_context_push(object->priv->context);

	if (object->priv->dbus_path != NULL)
	{
		galago_context_remove_object(object);

		free(object->priv->dbus_path);
	}

	object->priv->dbus_path = (obj_path != NULL ? strdup(obj_path) : NULL);

	if (object->priv->dbus_path != NULL)
		galago_context_add_object(object);

	galago_context_pop();
}

const char *
galago_object_get_dbus_path(const void *_object)
{
	galago_return_val_if_fail(_object != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(_object), NULL);

	return GALAGO_OBJECT(_object)->priv->dbus_path;
}

void
galago_object_set_watch(void *_object, galago_bool watch)
{
	GalagoObject *object;

	galago_return_if_fail(_object != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(_object));

	object = GALAGO_OBJECT(_object);

	if (object->priv->watched == watch)
		return;

	object->priv->watched = watch;
}

galago_bool
galago_object_is_watched(const void *_object)
{
	galago_return_val_if_fail(_object != NULL,           FALSE);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(_object), FALSE);

	return GALAGO_OBJECT(_object)->priv->watched;
}

void
galago_object_set_data(void *_object, const char *name, void *value)
{
	GalagoObject *object;

	galago_return_if_fail(_object  != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(_object));
	galago_return_if_fail(name  != NULL);
	galago_return_if_fail(*name != '\0');

	object = GALAGO_OBJECT(_object);

	if (object->priv->user_data == NULL)
	{
		object->priv->user_data =
			galago_hash_table_new_full(galago_str_hash, galago_str_equal,
									   free, NULL);
	}

	if (value == NULL)
		galago_hash_table_remove(object->priv->user_data, name);
	else
		galago_hash_table_replace(object->priv->user_data, strdup(name), value);
}

void *
galago_object_get_data(const void *_object, const char *name)
{
	GalagoObject *object;

	galago_return_val_if_fail(_object  != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(_object), NULL);
	galago_return_val_if_fail(name  != NULL,             NULL);
	galago_return_val_if_fail(*name != '\0',             NULL);

	object = GALAGO_OBJECT(_object);

	if (object->priv->user_data == NULL)
		return NULL;

	return galago_hash_table_lookup(object->priv->user_data, name);
}

GalagoContext *
galago_object_get_context(const void *_object)
{
	galago_return_val_if_fail(_object  != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(_object), NULL);

	return GALAGO_OBJECT(_object)->priv->context;
}

GalagoObject *
galago_object_check_cast(const void *ptr, const GalagoObjectClass *klass)
{
	GalagoObjectClass *tmp_class;
	GalagoObject *obj;
	galago_bool found = FALSE;

	galago_return_val_if_fail(ptr != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(ptr),  NULL);
	galago_return_val_if_fail(klass != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_CLASS(klass), NULL);

	obj = GALAGO_OBJECT(ptr);

	for (tmp_class = obj->priv->klass;
		 tmp_class != NULL && !found;
		 tmp_class = tmp_class->priv->parent_class)
	{
		if (tmp_class->priv->serial == klass->priv->serial)
			found = TRUE;
	}

	if (!found)
	{
		galago_log_error("Invalid cast from `%s' to `%s'\n",
						 galago_class_get_name(obj->priv->klass),
						 galago_class_get_name(klass));

		return NULL;
	}

	return obj;
}
