/**
 * @file libgalago/galago-person.h Galago Person API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_PERSON_H_
#define _GALAGO_PERSON_H_

typedef struct _GalagoPerson        GalagoPerson;
typedef struct _GalagoPersonClass   GalagoPersonClass;
typedef struct _GalagoPersonPrivate GalagoPersonPrivate;

#include <libgalago/galago-account.h>
#include <libgalago/galago-list.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-photo.h>
#include <libgalago/galago-service.h>
#include <libgalago/galago-signals.h>
#include <libgalago/galago-value.h>

/**
 * A person, which may have multiple accounts.
 */
struct _GalagoPerson
{
	GalagoObject parent_object;

	GalagoPersonPrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * GalagoPerson's class.
 */
struct _GalagoPersonClass
{
	GalagoObjectClass parent_class;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

typedef enum
{
	GALAGO_PROPERTY_TYPE_UNKNOWN = -1,  /**< Unknown property type.       */
	GALAGO_PROPERTY_TYPE_STRING  = 0,   /**< String type.                 */
	GALAGO_PROPERTY_TYPE_BOOL,          /**< Boolean type.                */
	GALAGO_PROPERTY_TYPE_UINT32         /**< Unsigned 32-bit integer type */

} GalagoPropertyType;

#define GALAGO_CLASS_PERSON (galago_person_get_class())
#define GALAGO_DBUS_PERSON_INTERFACE "org.freedesktop.Galago.Person"
#define GALAGO_IS_PERSON(obj) (GALAGO_IS_OBJECT(obj) && galago_object_check_cast((obj), GALAGO_CLASS_PERSON))

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Common Property Definitions                                     */
/**************************************************************************/
/*{*/

#define GALAGO_PERSON_PROP_FIRST_NAME   "first-name"
#define GALAGO_PERSON_PROP_MIDDLE_NAME  "middle-name"
#define GALAGO_PERSON_PROP_LAST_NAME    "last-name"
#define GALAGO_PERSON_PROP_ADDRESS      "address"
#define GALAGO_PERSON_PROP_CITY         "city"
#define GALAGO_PERSON_PROP_STATE        "state"
#define GALAGO_PERSON_PROP_COUNTRY      "country"
#define GALAGO_PERSON_PROP_ZIPCODE      "zipcode"

/*@}*/

/**************************************************************************/
/** @name Person API                                                      */
/**************************************************************************/
/*@{*/

/**
 * Returns the class for a GalagoPerson.
 *
 * @return The GalagoPerson class.
 */
GalagoObjectClass *galago_person_get_class(void);

/**
 * Creates a person.
 *
 * If I could do this in real life, I'd never have be without a date again.
 *
 * @param id     The application-specific unqiue ID of the person
 * @param native TRUE if this person is native, or FALSE otherwise.
 *
 * @return The person.
 */
GalagoPerson *galago_person_new(const char *id, galago_bool native);

/**
 * Creates a new "Me" person.
 *
 * @param native TRUE if this person is native, or FALSE otherwise.
 *
 * @return The "Me" person.
 */
GalagoPerson *galago_person_me_new(galago_bool native);

/**
 * Sets a person to be the "Me" person.
 *
 * This should only be used internally. Please use galago_person_me_new()
 * from the beginning.
 *
 * This is a one-way thing. You can't make a "Me" person a normal
 * person.
 *
 * @param person The person.
 */
void galago_person_set_me(GalagoPerson *person);

/**
 * Returns whether or not a person is the "Me" person.
 *
 * @param person The person.
 *
 * @return TRUE if the person is the "Me" person, or FALSE.
 */
galago_bool galago_person_is_me(const GalagoPerson *person);

/**
 * Returns a person's application-specific unique ID.
 *
 * @param person The person.
 *
 * @return The person's ID.
 */
const char *galago_person_get_id(const GalagoPerson *person);

/**
 * Returns whether or not a person is native to the local application.
 *
 * @param person The person.
 *
 * @return TRUE if the person is native to the local application, or
 *         FALSE if it came from another feed.
 */
galago_bool galago_person_is_native(const GalagoPerson *person);

/**
 * Returns the display name of a person, if any.
 *
 * The returned display name is from a static buffer. If you wish to store
 * the name, you must strdup it.
 *
 * @param person The person.
 *
 * @return The person's display name, or NULL.
 */
const char *galago_person_get_display_name(const GalagoPerson *person);

/**
 * Sets the person's photo.
 *
 * This is used internally. You don't need to call this.
 *
 * @param person The person.
 * @param photo  The photo to set.
 */
void galago_person_set_photo(GalagoPerson *person, GalagoPhoto *photo);

/**
 * Returns the person's photo.
 *
 * @param person The person.
 * @param query  TRUE if a remote query should be done if there is no
 *               local photo found, or FALSE.
 *
 * @return The photo, if found, or NULL.
 */
GalagoPhoto *galago_person_get_photo(const GalagoPerson *person,
									 galago_bool query);

/**
 * Returns whether the person has any accounts at all.
 *
 * @param person The person.
 * @param query  TRUE if a query should be done to check the server-side
 *               state of the accounts, or FALSE.
 *
 * @return TRUE if the person has at least one account, or FALSE otherwise.
 */
galago_bool galago_person_has_accounts(const GalagoPerson *person,
									   galago_bool query);

/**
 * Returns a list of accounts in the person.
 *
 * @param person The person.
 * @param query  TRUE if a remote query should be done if there is no
 *               local account found, or FALSE.
 *
 * @return The list of accounts.
 */
const GalagoList *galago_person_get_accounts(const GalagoPerson *person,
											 galago_bool query);

/**
 * Returns the most available "priority" account.
 *
 * @param person The person.
 *
 * @return The priority account.
 */
GalagoAccount *galago_person_get_priority_account(const GalagoPerson *person);

/**
 * Adds an account to a person.
 *
 * This is used internally. You don't need to call this.
 *
 * @param person  The person.
 * @param account The account to add.
 */
void galago_person_add_account(GalagoPerson *person, GalagoAccount *account);

/**
 * Removes an account from a person.
 *
 * This is used internally. You don't need to call this.
 *
 * @param person  The person.
 * @param account The account to remove.
 */
void galago_person_remove_account(GalagoPerson *person,
								  GalagoAccount *account);

/**
 * Returns the account with the specified username and service.
 *
 * @param person   The person.
 * @param service  The service.
 * @param username The username.
 * @param query    TRUE if a remote query should be done if there is no
 *                 local account found, or FALSE.
 *
 * @return The account, if found, or NULL.
 */
GalagoAccount *galago_person_get_account(const GalagoPerson *person,
										 const GalagoService *service,
										 const char *username,
										 galago_bool query);

/**
 * Sets a string property on a person.
 *
 * @param person The person.
 * @param name   The name of the property to set.
 * @param value  The value of the property.
 */
void galago_person_set_property_string(GalagoPerson *person, const char *name,
									   const char *value);

/**
 * Sets a boolean property on a person.
 *
 * @param person The person.
 * @param name   The name of the property to set.
 * @param value  The value of the property.
 */
void galago_person_set_property_bool(GalagoPerson *person, const char *name,
									 galago_bool value);

/**
 * Sets a uint32 property on a person.
 *
 * @param person The person.
 * @param name   The name of the property to set.
 * @param value  The value of the property.
 */
void galago_person_set_property_uint32(GalagoPerson *person, const char *name,
									   dbus_uint32_t value);

/**
 * Sets a property on a person.
 *
 * This is limited to string, boolean, and uint32 value types.
 *
 * @param person The person.
 * @param name   The name of the property to set.
 * @param value  The value of the property.
 */
void galago_person_set_property(GalagoPerson *person, const char *name,
								GalagoValue *value);

/**
 * Removes a property on a person.
 *
 * @param person The person
 * @param name   The name of the property to remove
 *
 * @return TRUE if the property was removed, or FALSE.
 */
galago_bool galago_person_remove_property(GalagoPerson *person,
										  const char *name);

/**
 * Returns the value of a string property on a person.
 *
 * @param person The person.
 * @param name   The name of the property.
 *
 * @return The property value, or NULL.
 */
const char *galago_person_get_property_string(const GalagoPerson *person,
											  const char *name);

/**
 * Returns the value of a boolean property on a person.
 *
 * @param person The person.
 * @param name   The name of the property.
 *
 * @return The property value, or NULL.
 */
galago_bool galago_person_get_property_bool(const GalagoPerson *person,
											const char *name);

/**
 * Returns the value of a uint32 property on a person.
 *
 * @param person The person.
 * @param name   The name of the property.
 *
 * @return The property value, or NULL.
 */
dbus_uint32_t galago_person_get_property_uint32(const GalagoPerson *person,
												const char *name);

/**
 * Returns the value of a property on a person.
 *
 * @param person The person.
 * @param name   The name of the property.
 *
 * @return The property value, or NULL.
 */
const GalagoValue *galago_person_get_property(const GalagoPerson *person,
											  const char *name);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_PERSON_H_ */
