/**
 * @file libgalago/galago-signals.h Galago Signals API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_SIGNALS_H_
#define _GALAGO_SIGNALS_H_

typedef struct _GalagoSignalHandler GalagoSignalHandler;
typedef struct _GalagoSignalContext GalagoSignalContext;

#include <libgalago/galago-object.h>
#include <libgalago/galago-value.h>

typedef void (*GalagoMarshalFunc)(GalagoCallback cb, GalagoObject *obj,
								  const char *signal, va_list args,
								  void *data);

#define GALAGO_SIGNAL_CONTEXT(obj) \
	(galago_class_get_signal_context(GALAGO_OBJECT_CLASS(obj)))

#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-types.h>

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Signal Context API                                              */
/**************************************************************************/
/*@{*/

/**
 * Creates a signal context.
 *
 * @return The new signal context.
 */
GalagoSignalContext *galago_signal_context_new(void);

/**
 * Destroys a signal context.
 *
 * @param signal_context The signal context.
 */
void galago_signal_context_destroy(GalagoSignalContext *signal_context);

/**
 * Freezes a signal context.
 *
 * @param signal_context The signal context to freeze.
 */
void galago_signal_context_freeze(GalagoSignalContext *signal_context);

/**
 * Thaws a signal context.
 *
 * @param signal_context The signal context to thaw.
 */
void galago_signal_context_thaw(GalagoSignalContext *signal_context);

/**
 * Returns whether or not a signal context is frozen.
 *
 * @param signal_context The signal context.
 *
 * @return TRUE if the signal context is frozen, or FALSE.
 */
galago_bool galago_signal_context_is_frozen(
	const GalagoSignalContext *signal_context);

/**
 * Finds a signal handler with the specified object, signal name, and callback.
 *
 * @param signal_context The signal context.
 * @param object         The object.
 * @param signal         The signal.
 * @param cb             The callback.
 *
 * @return The signal handler, if found, or NULL.
 */
GalagoSignalHandler *galago_signal_context_find_handler(
	const GalagoSignalContext *signal_context, const void *object,
	const char *signal, GalagoCallback cb);

/**
 * Finds a signal handler with the specified ID.
 *
 * @param id The signal handler ID.
 *
 * @return The signal handler, if fond, or NULL.
 */
GalagoSignalHandler *galago_signal_context_find_handler_with_id(
	unsigned long id);

/*@}*/

/**************************************************************************/
/** @name Signals Subsystem API                                           */
/**************************************************************************/
/*@{*/

/**
 * Registers a signal in the signal context.
 *
 * @param signal_context The signal context.
 * @param signal         The signal name.
 * @param marshal        The marshaller function.
 * @param num_values     The number of values passed to the callback.
 * @param ...            The optional list of data types passed.
 *
 * @return TRUE if this signal was registered, or FALSE if there was an
 *         error or if @a signal is already registered.
 */
galago_bool galago_signal_register(GalagoSignalContext *signal_context,
								   const char *signal,
								   GalagoMarshalFunc marshal,
								   size_t num_values, ...);

/**
 * Unregisters a signal in the signal context.
 *
 * @param signal_context The signal context.
 * @param signal         The signal name.
 */
void galago_signal_unregister(GalagoSignalContext *signal_context,
							  const char *signal);

/**
 * Connects a signal handler to a signal for all objects in a class.
 *
 * @param klass     The class.
 * @param signal    The signal to connect to.
 * @param cb        The callback function.
 * @param user_data User-specified data.
 *
 * @return The ID of the signal handler, or 0 if there was an error.
 */
unsigned long galago_signal_connect_class(GalagoObjectClass *klass,
										  const char *signal,
										  GalagoCallback cb, void *user_data);

/**
 * Connects a signal handler to a signal for an object.
 *
 * @param obj       The object to connect to, or NULL for all on the context.
 * @param signal    The signal to connect to.
 * @param cb        The callback function.
 * @param user_data User-specified data.
 *
 * @return The ID of the signal handler, or 0 if there was an error.
 */
unsigned long galago_signal_connect(void *obj, const char *signal,
									GalagoCallback cb, void *user_data);

/**
 * Disconnects a signal handler with the specified ID.
 *
 * @param id The signal handler ID.
 */
void galago_signals_disconnect_by_id(unsigned long id);

/**
 * Emits a signal.
 *
 * @param object The object.
 * @param signal The signal name.
 * @param ...    The arguments list.
 */
void galago_signal_emit(void *object, const char *signal, ...);

/**
 * Emits a signal, using a va_list of arguments.
 *
 * @param object The object.
 * @param signal The signal name.
 * @param args   The arguments list.
 */
void galago_signal_emit_vargs(void *object, const char *signal, va_list args);

/**
 * Initializes the signals subsystem.
 */
void galago_signals_init(void);

/**
 * Uninitializes the signals subsystem.
 */
void galago_signals_uninit(void);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_SIGNALS_H_ */
