/**
 * @file libgalago/galago-status.c Galago Status API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-private.h>
#include <libgalago/galago-status.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-signals.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <stdio.h>
#include <string.h>

struct _GalagoStatusPrivate
{
	GalagoStatusType type;

	GalagoPresence *presence;

	char *id;
	char *name;

	galago_bool exclusive;

	GalagoHashTable *attrs_table;
	GalagoList *attrs;
};

struct _GalagoStatusAttr
{
	GalagoStatusAttrType type;

	char *id;

	union
	{
		galago_bool bool_val;
		char *string_val;
		dbus_int32_t int_val;
		double double_val;

	} value;
};

static const char *attr_type_strings[] =
{
	NULL,
	"boolean",
	"string",
	"int",
	"double"
};

static GalagoHashTable *status_id_map_table = NULL;

static GalagoStatusAttr *status_attr_new(const char *id,
										 GalagoStatusAttrType type);
static void status_attr_destroy(GalagoStatusAttr *attr);

/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_status, GalagoStatus, NULL, NULL);
DEPRECATE_OLD_GET_CLASS(galago_statuses, galago_status);

static void
galago_status_object_init(GalagoStatus *status)
{
	status->priv = galago_new0(GalagoStatusPrivate, 1);

	status->priv->attrs_table =
		galago_hash_table_new_full(galago_str_hash, galago_str_equal,
								   free, (GalagoFreeFunc)status_attr_destroy);
}

static void
galago_status_object_finalize(GalagoObject *object)
{
	GalagoStatus *status = (GalagoStatus *)object;

	galago_hash_table_destroy(status->priv->attrs_table);
	galago_list_destroy(status->priv->attrs);

	if (status->priv->id != NULL)
		free(status->priv->id);

	if (status->priv->name != NULL)
		free(status->priv->name);

	free(status->priv);
}

static void
galago_status_dbus_message_append(DBusMessageIter *iter,
                                  const GalagoObject *object)
{
	GalagoStatus *status = (GalagoStatus *)object;
	const GalagoList *attrs, *l;
	GalagoStatusType type;
	const char *id, *name;
	galago_bool exclusive;
	size_t attr_count;

	type      = galago_status_get_type(status);
	id        = galago_status_get_id(status);
	name      = galago_status_get_name(status);
	exclusive = galago_status_is_exclusive(status);

	galago_dbus_message_iter_append_byte(iter,    type);
	galago_dbus_message_iter_append_string(iter,  id);
	galago_dbus_message_iter_append_string(iter,  name);
	galago_dbus_message_iter_append_boolean(iter, exclusive);

	attrs = galago_status_get_attributes(status);

	attr_count = galago_list_get_count(attrs);
	galago_dbus_message_iter_append_uint32(iter, attr_count);

	for (l = attrs; l != NULL; l = l->next)
	{
		GalagoStatusAttr *attr = (GalagoStatusAttr *)l->data;
		GalagoStatusAttrType attr_type = galago_status_attr_get_type(attr);
		const char *attr_id = galago_status_attr_get_id(attr);

		galago_dbus_message_iter_append_byte(iter,   attr_type);
		galago_dbus_message_iter_append_string(iter, attr_id);

		switch (attr_type)
		{
			case GALAGO_STATUS_ATTR_BOOL:
			{
				galago_bool value = galago_status_attr_get_bool(attr);
				galago_dbus_message_iter_append_boolean(iter, value);
				break;
			}

			case GALAGO_STATUS_ATTR_STRING:
			{
				const char *value = galago_status_attr_get_string(attr);
				galago_dbus_message_iter_append_string(iter, value);
				break;
			}

			case GALAGO_STATUS_ATTR_INT:
			{
				int value = galago_status_attr_get_int(attr);
				galago_dbus_message_iter_append_int32(iter, value);
				break;
			}

			case GALAGO_STATUS_ATTR_DOUBLE:
			{
				double value = galago_status_attr_get_double(attr);
				galago_dbus_message_iter_append_double(iter, value);
				break;
			}

			default: /* This should never be reached. */
				break;
		}
	}
}

static void *
galago_status_dbus_message_get(DBusMessageIter *iter)
{
	GalagoStatus *status;
	char type;
	char *id, *name;
	dbus_uint32_t num_attrs, i;
	galago_bool exclusive;

	galago_dbus_message_iter_get_byte(iter, type);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_string(iter, id);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_string(iter, name);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_boolean(iter, exclusive);
	dbus_message_iter_next(iter);

	status = galago_status_new((GalagoStatusType)type, id, name, exclusive);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_free(id);
	dbus_free(name);
#endif

	galago_dbus_message_iter_get_uint32(iter, num_attrs);

	for (i = 0; i < num_attrs; i++)
	{
		char attr_type;
		char *attr_id;
		char *str;

		dbus_message_iter_next(iter);

		galago_dbus_message_iter_get_byte(iter, attr_type);
		dbus_message_iter_next(iter);

		galago_dbus_message_iter_get_string(iter, attr_id);

		switch ((GalagoStatusAttrType)attr_type)
		{
			case GALAGO_STATUS_ATTR_BOOL:
			{
				galago_bool value;
				dbus_message_iter_next(iter);
				galago_dbus_message_iter_get_boolean(iter, value);
				galago_status_set_attr_bool(status, attr_id, value);
				break;
			}

			case GALAGO_STATUS_ATTR_STRING:
				dbus_message_iter_next(iter);
				str = galago_dbus_message_iter_get_string_or_nil(iter);
				galago_status_set_attr_string(status, attr_id, str);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
				if (str != NULL)
					dbus_free(str);
#endif
				break;

			case GALAGO_STATUS_ATTR_INT:
			{
				dbus_uint32_t value;
				dbus_message_iter_next(iter);
				galago_dbus_message_iter_get_int32(iter, value);
				galago_status_set_attr_int(status, attr_id, value);
				break;
			}

			case GALAGO_STATUS_ATTR_DOUBLE:
			{
				double value;
				dbus_message_iter_next(iter);
				galago_dbus_message_iter_get_double(iter, value);
				galago_status_set_attr_double(status, attr_id, value);
				break;
			}

			default: /* This should never be reached. */
				break;
		}

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
		dbus_free(attr_id);
#endif
	}

	return status;
}

static void
galago_status_class_init(GalagoObjectClass *klass)
{
	klass->finalize            = galago_status_object_finalize;
	klass->dbus_message_append = galago_status_dbus_message_append;
	klass->dbus_message_get    = galago_status_dbus_message_get;
}


/**************************************************************************
 * Status ID to Name mapping
 **************************************************************************/
#define ADD_STATUS_MAP(id, name) \
	galago_hash_table_insert(status_id_map_table, strdup(id), strdup(name));

static void
_init_status_id_map_table(void)
{
	status_id_map_table =
		galago_hash_table_new_full(galago_str_hash, galago_str_equal,
								   free, free);

	ADD_STATUS_MAP(GALAGO_STATUS_ID_AVAILABLE,     _("Available"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_AWAY,          _("Away"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_BRB,           _("Be Right Back"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_BUSY,          _("Busy"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_DND,           _("Do Not Disturb"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_EXTENDED_AWAY, _("Extended Away"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_HIDDEN,        _("Hidden"));
	ADD_STATUS_MAP(GALAGO_STATUS_ID_OFFLINE,       _("Offline"));
}

static const char *
_galago_statuses_map_id_to_name(const char *id)
{
	galago_return_val_if_fail(id  != NULL, NULL);
	galago_return_val_if_fail(*id != '\0', NULL);

	if (status_id_map_table == NULL)
		_init_status_id_map_table();

	return (const char *)galago_hash_table_lookup(status_id_map_table, id);
}

/**************************************************************************
 * GalagoStatus API
 **************************************************************************/
GalagoStatus *
galago_status_new(GalagoStatusType type, const char *id, const char *name,
				  galago_bool exclusive)
{
	GalagoStatus *status;
	const char *name2;

	galago_return_val_if_fail(galago_is_initted(),           NULL);
	galago_return_val_if_fail(type != GALAGO_STATUS_UNSET,   NULL);
	galago_return_val_if_fail(id   != NULL,                  NULL);
	galago_return_val_if_fail(*id  != '\0',                  NULL);

	if ((name2 = _galago_statuses_map_id_to_name(id)) != NULL)
		name = name2;

	galago_return_val_if_fail(name  != NULL, NULL);
	galago_return_val_if_fail(*name != '\0', NULL);

	status = galago_object_new(GALAGO_CLASS_STATUS);

	status->priv->type      = type;
	status->priv->id        = strdup(id);
	status->priv->name      = strdup(name);
	status->priv->exclusive = exclusive;

	return status;
}

GalagoStatus *
galago_status_duplicate(const GalagoStatus *status)
{
	GalagoStatus *new_status;
	const GalagoList *l;

	galago_return_val_if_fail(status != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	new_status = galago_status_new(galago_status_get_type(status),
								   galago_status_get_id(status),
								   galago_status_get_name(status),
								   galago_status_is_exclusive(status));

	for (l = galago_status_get_attributes(status); l != NULL; l = l->next)
	{
		GalagoStatusAttr *status_attr = (GalagoStatusAttr *)l->data;
		const char *attr_id = galago_status_attr_get_id(status_attr);

		switch (galago_status_attr_get_type(status_attr))
		{
			case GALAGO_STATUS_ATTR_BOOL:
				galago_status_set_attr_bool(new_status, attr_id,
					galago_status_attr_get_bool(status_attr));
				break;

			case GALAGO_STATUS_ATTR_STRING:
				galago_status_set_attr_string(new_status, attr_id,
					galago_status_attr_get_string(status_attr));
				break;

			case GALAGO_STATUS_ATTR_INT:
				galago_status_set_attr_int(new_status, attr_id,
					galago_status_attr_get_int(status_attr));
				break;

			case GALAGO_STATUS_ATTR_DOUBLE:
				galago_status_set_attr_double(new_status, attr_id,
					galago_status_attr_get_double(status_attr));
				break;

			default:
				break;
		}
	}

	return new_status;
}

void
galago_status_set_presence(GalagoStatus *status, GalagoPresence *presence)
{
	galago_return_if_fail(status != NULL);
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	status->priv->presence = presence;
}

GalagoPresence *
galago_status_get_presence(const GalagoStatus *status)
{
	galago_return_val_if_fail(status != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	return status->priv->presence;
}

GalagoStatusType
galago_status_get_type(const GalagoStatus *status)
{
	galago_return_val_if_fail(status != NULL,           GALAGO_STATUS_UNSET);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), GALAGO_STATUS_UNSET);

	return status->priv->type;
}

const char *
galago_status_get_id(const GalagoStatus *status)
{
	galago_return_val_if_fail(status != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	return status->priv->id;
}

const char *
galago_status_get_name(const GalagoStatus *status)
{
	galago_return_val_if_fail(status != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	return status->priv->name;
}

galago_bool
galago_status_is_exclusive(const GalagoStatus *status)
{
	galago_return_val_if_fail(status != NULL,           FALSE);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), FALSE);

	return status->priv->exclusive;
}

galago_bool
galago_status_is_available(const GalagoStatus *status)
{
	GalagoStatusType type;

	galago_return_val_if_fail(status != NULL,           FALSE);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), FALSE);

	type = galago_status_get_type(status);

	return (type == GALAGO_STATUS_AVAILABLE ||
			type == GALAGO_STATUS_HIDDEN);
}

static GalagoStatusAttr *
common_make_attr(GalagoStatus *status, const char *attr_id,
				 GalagoStatusAttrType type)
{
	GalagoStatusAttr *attr;
	GalagoPresence *presence;

	attr = galago_hash_table_lookup(status->priv->attrs_table, attr_id);

	if (attr == NULL)
	{
		attr = status_attr_new(attr_id, type);

		galago_hash_table_insert(status->priv->attrs_table,
								 strdup(attr_id), attr);
		status->priv->attrs = galago_list_append(status->priv->attrs, attr);
	}
	else if (attr->type != type)
	{
		galago_log_error("Attempted to set existing attribute ID %s with "
						 "invalid type: %s\n",
						 attr_id, attr_type_strings[type]);
		return NULL;
	}

	presence = galago_status_get_presence(status);

	if (presence != NULL)
	{
		galago_signal_emit(presence, "status-updated", status, attr_id);
		galago_signal_emit(presence, "updated");
	}

	return attr;
}

void
galago_status_set_attr_bool(GalagoStatus *status, const char *attr_id,
							galago_bool value)
{
	GalagoStatusAttr *attr;

	galago_return_if_fail(status  != NULL);
	galago_return_if_fail(attr_id != NULL);
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	attr = common_make_attr(status, attr_id, GALAGO_STATUS_ATTR_BOOL);

	if (attr == NULL)
		return;

	attr->value.bool_val = value;
}

void
galago_status_set_attr_string(GalagoStatus *status, const char *attr_id,
							  const char *value)
{
	GalagoStatusAttr *attr;

	galago_return_if_fail(status  != NULL);
	galago_return_if_fail(attr_id != NULL);
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	attr = common_make_attr(status, attr_id, GALAGO_STATUS_ATTR_STRING);

	if (attr == NULL)
		return;

	if (attr->value.string_val == NULL)
		free(attr->value.string_val);

	attr->value.string_val = (value == NULL ? NULL : strdup(value));
}


void
galago_status_set_attr_int(GalagoStatus *status, const char *attr_id,
						   dbus_int32_t value)
{
	GalagoStatusAttr *attr;

	galago_return_if_fail(status  != NULL);
	galago_return_if_fail(attr_id != NULL);
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	attr = common_make_attr(status, attr_id, GALAGO_STATUS_ATTR_INT);

	if (attr == NULL)
		return;

	attr->value.int_val = value;
}

void
galago_status_set_attr_double(GalagoStatus *status, const char *attr_id,
							  double value)
{
	GalagoStatusAttr *attr;

	galago_return_if_fail(status  != NULL);
	galago_return_if_fail(attr_id != NULL);
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	attr = common_make_attr(status, attr_id, GALAGO_STATUS_ATTR_DOUBLE);

	if (attr == NULL)
		return;

	attr->value.double_val = value;
}

void
galago_status_remove_attr(GalagoStatus *status, const char *attr_id)
{
	GalagoStatusAttr *attr;

	galago_return_if_fail(status  != NULL);
	galago_return_if_fail(attr_id != NULL);
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	attr = galago_hash_table_lookup(status->priv->attrs_table, attr_id);

	if (attr != NULL)
	{
		galago_log_error("Attempting to remove invalid attribute %s\n",
						 attr_id);
		return;
	}

	galago_hash_table_remove(status->priv->attrs_table, attr_id);
	status->priv->attrs = galago_list_remove(status->priv->attrs, attr);
}

static GalagoStatusAttr *
common_get_attr(const GalagoStatus *status, const char *attr_id,
				GalagoStatusAttrType type)
{
	GalagoStatusAttr *attr;

	attr = galago_hash_table_lookup(status->priv->attrs_table, attr_id);

	if (attr == NULL)
		return NULL;
	else if (attr->type != type)
	{
		galago_log_error("Attempted to retrieve existing attribute ID %s with "
						 "invalid type: %s\n",
						 attr_id, attr_type_strings[type]);
		return NULL;
	}

	return attr;
}

galago_bool
galago_status_get_attr_bool(const GalagoStatus *status, const char *attr_id)
{
	GalagoStatusAttr *attr;

	galago_return_val_if_fail(status  != NULL,          FALSE);
	galago_return_val_if_fail(attr_id != NULL,          FALSE);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), FALSE);

	attr = common_get_attr(status, attr_id, GALAGO_STATUS_ATTR_BOOL);

	if (attr == NULL)
		return FALSE;

	return attr->value.bool_val;
}

const char *
galago_status_get_attr_string(const GalagoStatus *status, const char *attr_id)
{
	GalagoStatusAttr *attr;

	galago_return_val_if_fail(status  != NULL,          NULL);
	galago_return_val_if_fail(attr_id != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	attr = common_get_attr(status, attr_id, GALAGO_STATUS_ATTR_STRING);

	if (attr == NULL)
		return NULL;

	return attr->value.string_val;
}

dbus_int32_t
galago_status_get_attr_int(const GalagoStatus *status, const char *attr_id)
{
	GalagoStatusAttr *attr;

	galago_return_val_if_fail(status  != NULL,          -1);
	galago_return_val_if_fail(attr_id != NULL,          -1);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), -1);

	attr = common_get_attr(status, attr_id, GALAGO_STATUS_ATTR_INT);

	if (attr == NULL)
		return -1;

	return attr->value.int_val;
}

double
galago_status_get_attr_double(const GalagoStatus *status, const char *attr_id)
{
	GalagoStatusAttr *attr;

	galago_return_val_if_fail(status  != NULL,          0.0);
	galago_return_val_if_fail(attr_id != NULL,          0.0);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), 0.0);

	attr = common_get_attr(status, attr_id, GALAGO_STATUS_ATTR_DOUBLE);

	if (attr == NULL)
		return 0.0;

	return attr->value.double_val;
}

galago_bool
galago_status_attr_exists(const GalagoStatus *status, const char *attr_id)
{
	galago_return_val_if_fail(status  != NULL,          FALSE);
	galago_return_val_if_fail(attr_id != NULL,          FALSE);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), FALSE);

	return (galago_hash_table_lookup(status->priv->attrs_table,
									 attr_id) != NULL);
}

const GalagoList *
galago_status_get_attributes(const GalagoStatus *status)
{
	galago_return_val_if_fail(status != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_STATUS(status), NULL);

	return status->priv->attrs;
}

/**************************************************************************
 * GalagoStatusAttr API
 **************************************************************************/
static GalagoStatusAttr *
status_attr_new(const char *id, GalagoStatusAttrType type)
{
	GalagoStatusAttr *attr;

	galago_return_val_if_fail(id != NULL, NULL);
	galago_return_val_if_fail(type != GALAGO_STATUS_ATTR_UNSET, NULL);

	attr = galago_new0(GalagoStatusAttr, 1);
	attr->id   = strdup(id);
	attr->type = type;

	return attr;
}

static void
status_attr_destroy(GalagoStatusAttr *attr)
{
	if (attr->id != NULL)
		free(attr->id);

	if (attr->type == GALAGO_STATUS_ATTR_STRING &&
		attr->value.string_val != NULL)
	{
		free(attr->value.string_val);
	}

	free(attr);
}

GalagoStatusAttrType
galago_status_attr_get_type(const GalagoStatusAttr *attr)
{
	galago_return_val_if_fail(attr != NULL, GALAGO_STATUS_ATTR_UNSET);

	return attr->type;
}

const char *
galago_status_attr_get_id(const GalagoStatusAttr *attr)
{
	galago_return_val_if_fail(attr != NULL, NULL);

	return attr->id;
}

galago_bool
galago_status_attr_get_bool(const GalagoStatusAttr *attr)
{
	galago_return_val_if_fail(attr != NULL, FALSE);
	galago_return_val_if_fail(attr->type == GALAGO_STATUS_ATTR_BOOL, FALSE);

	return attr->value.bool_val;
}

const char *
galago_status_attr_get_string(const GalagoStatusAttr *attr)
{
	galago_return_val_if_fail(attr != NULL, NULL);
	galago_return_val_if_fail(attr->type == GALAGO_STATUS_ATTR_STRING, NULL);

	return attr->value.string_val;
}

dbus_int32_t
galago_status_attr_get_int(const GalagoStatusAttr *attr)
{
	galago_return_val_if_fail(attr != NULL, -1);
	galago_return_val_if_fail(attr->type == GALAGO_STATUS_ATTR_INT, -1);

	return attr->value.int_val;
}

double
galago_status_attr_get_double(const GalagoStatusAttr *attr)
{
	galago_return_val_if_fail(attr != NULL, 0.0);
	galago_return_val_if_fail(attr->type == GALAGO_STATUS_ATTR_DOUBLE, 0.0);

	return attr->value.double_val;
}
