/**
 * @file presence-feed.c Sample presence feed.
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include <glib.h>
#include <libgalago/galago.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>

static gboolean
read_file(const char *filename, unsigned char **ret_data, size_t *ret_size)
{
	FILE *fp;
	struct stat sb;

	g_return_val_if_fail(ret_data != NULL, FALSE);
	g_return_val_if_fail(ret_size != NULL, FALSE);

	if (stat(filename, &sb) != 0)
	{
		fprintf(stderr, "Unable to stat %s\n", filename);
		return FALSE;
	}

	*ret_size = sb.st_size;

	fp = fopen(filename, "rb");

	if (fp == NULL)
	{
		fprintf(stderr, "Unable to open %s for reading\n", filename);
		return FALSE;
	}

	*ret_data = g_malloc(*ret_size);
	fread(*ret_data, 1, *ret_size, fp);
	fclose(fp);

	return TRUE;
}

static void
setup_info(void)
{
	GalagoService *aim_service, *jabber_service;
	GalagoAccount *account;
	GalagoAccount *contact;
	GalagoPerson *me, *person;
	GalagoPresence *presence;
	GalagoStatus *status;
	unsigned char *image_data;
	size_t image_len;

	/* Setup our services */
	aim_service    = galago_service_new("aim", "AOL Instant Messenger",
										TRUE, 0);
	jabber_service = galago_service_new("jabber", "Jabber", TRUE,
										GALAGO_STRIP_SLASH);

	/* Create our local person */
	me = galago_person_me_new(TRUE);

	/* Setup our person's AIM account */
	account  = galago_account_new(aim_service, me, "GalagoAIMUser");
	presence = galago_presence_new(account);

	galago_presence_set_idle(presence, TRUE, 60 * 5);

	status = galago_status_new(GALAGO_STATUS_AWAY, "away", "Away", TRUE);
	galago_status_set_attr_string(status, "message", "I'm eating dinner.");

	galago_presence_add_status(presence, status);

	/* Add contacts to our AIM account */
	person   = galago_person_new("gonkulator-1", TRUE);
	contact  = galago_account_new(aim_service, person, "lottabs2");
	presence = galago_presence_new(contact);

	galago_presence_set_idle(presence, TRUE, 60 * 20);

	galago_account_add_contact(account, contact);

	if (read_file("avatar.png", &image_data, &image_len))
	{
		GalagoAvatar *avatar;
		avatar = galago_avatar_new(contact, image_data, image_len);
		g_free(image_data);
	}

	/* Setup our person's Jabber account */
	account  = galago_account_new(jabber_service, me,
								  "jabberuser@galago.sf.net");
	presence = galago_presence_new(account);

	status = galago_status_new(GALAGO_STATUS_AVAILABLE, "available",
							   "Available", TRUE);

	galago_presence_add_status(presence, status);

	/* Add contacts to our Jabber account */
	person   = galago_person_new("cae-1", TRUE);
	contact  = galago_account_new(jabber_service, person, "caedan@jabber.org");
	presence = galago_presence_new(contact);

	status = galago_status_new(GALAGO_STATUS_AVAILABLE, "available",
							   "Available", TRUE);
	galago_presence_add_status(presence, status);

	galago_account_add_contact(account, contact);

	contact = galago_account_new(jabber_service, person, "caelum@chipx86.com");
	presence = galago_presence_new(contact);

	status = galago_status_new(GALAGO_STATUS_OFFLINE, "offline", "Offline",
							   TRUE);
	galago_presence_add_status(presence, status);

	galago_account_add_contact(account, contact);
}

int
main(int argc, char **argv)
{
	GMainLoop *loop;

	g_type_init();

	loop = g_main_loop_new(NULL, FALSE);

	if (!galago_glib_init("test-feed", TRUE, NULL))
	{
		fprintf(stderr, "Unable to initialize Galago and connect "
				"to the server\n");
		exit(1);
	}

	setup_info();

	g_main_loop_run(loop);

	return 0;
}
