/* map.vala
 *
 * Copyright (C) 2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_MAP (gee_map_get_type ())
#define GEE_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_MAP, GeeMap))
#define GEE_IS_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_MAP))
#define GEE_MAP_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_MAP, GeeMapIface))

typedef struct _GeeMap GeeMap;
typedef struct _GeeMapIface GeeMapIface;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_SET (gee_set_get_type ())
#define GEE_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_SET, GeeSet))
#define GEE_IS_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_SET))
#define GEE_SET_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_SET, GeeSetIface))

typedef struct _GeeSet GeeSet;
typedef struct _GeeSetIface GeeSetIface;

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GeeIterator* (*iterator) (GeeIterable* self);
	GType (*get_element_type) (GeeIterable* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gboolean (*add_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*contains_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*remove_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*retain_all) (GeeCollection* self, GeeCollection* collection);
	gpointer* (*to_array) (GeeCollection* self, int* result_length1);
	gint (*get_size) (GeeCollection* self);
	gboolean (*get_is_empty) (GeeCollection* self);
};

/**
 * A set is a collection without duplicates.
 */
struct _GeeSetIface {
	GTypeInterface parent_iface;
};

/**
 * A map is a generic collection of key/value pairs.
 */
struct _GeeMapIface {
	GTypeInterface parent_iface;
	GeeSet* (*get_keys) (GeeMap* self);
	GeeCollection* (*get_values) (GeeMap* self);
	gboolean (*contains) (GeeMap* self, gconstpointer key);
	gpointer (*get) (GeeMap* self, gconstpointer key);
	void (*set) (GeeMap* self, gconstpointer key, gconstpointer value);
	gboolean (*remove) (GeeMap* self, gconstpointer key, gpointer* value);
	void (*clear) (GeeMap* self);
	void (*set_all) (GeeMap* self, GeeMap* map);
	gboolean (*remove_all) (GeeMap* self, GeeMap* map);
	gboolean (*contains_all) (GeeMap* self, GeeMap* map);
	gint (*get_size) (GeeMap* self);
	gboolean (*get_is_empty) (GeeMap* self);
};



GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_set_get_type (void);
GType gee_map_get_type (void);
GeeSet* gee_map_get_keys (GeeMap* self);
GeeCollection* gee_map_get_values (GeeMap* self);
gboolean gee_map_contains (GeeMap* self, gconstpointer key);
gpointer gee_map_get (GeeMap* self, gconstpointer key);
void gee_map_set (GeeMap* self, gconstpointer key, gconstpointer value);
gboolean gee_map_remove (GeeMap* self, gconstpointer key, gpointer* value);
void gee_map_clear (GeeMap* self);
void gee_map_set_all (GeeMap* self, GeeMap* map);
gboolean gee_map_remove_all (GeeMap* self, GeeMap* map);
gboolean gee_map_contains_all (GeeMap* self, GeeMap* map);
gint gee_map_get_size (GeeMap* self);
gboolean gee_map_get_is_empty (GeeMap* self);



/**
 * Returns the keys of this map as a read-only set.
 *
 * @return the keys of the map
 */
GeeSet* gee_map_get_keys (GeeMap* self) {
	return GEE_MAP_GET_INTERFACE (self)->get_keys (self);
}


/**
 * Returns the values of this map as a read-only collection.
 *
 * @return the values of the map
 */
GeeCollection* gee_map_get_values (GeeMap* self) {
	return GEE_MAP_GET_INTERFACE (self)->get_values (self);
}


/**
 * Determines whether this map contains the specified key.
 *
 * @param key the key to locate in the map
 *
 * @return    true if key is found, false otherwise
 */
gboolean gee_map_contains (GeeMap* self, gconstpointer key) {
	return GEE_MAP_GET_INTERFACE (self)->contains (self, key);
}


/**
 * Returns the value of the specified key in this map.
 *
 * @param key the key whose value is to be retrieved
 *
 * @return    the value associated with the key, or null if the key
 *            couldn't be found
 */
gpointer gee_map_get (GeeMap* self, gconstpointer key) {
	return GEE_MAP_GET_INTERFACE (self)->get (self, key);
}


/**
 * Inserts a new key and value into this map.
 *
 * @param key   the key to insert
 * @param value the value to associate with the key
 */
void gee_map_set (GeeMap* self, gconstpointer key, gconstpointer value) {
	GEE_MAP_GET_INTERFACE (self)->set (self, key, value);
}


/**
 * Removes the specified key from this map.
 *
 * @param key   the key to remove from the map
 * @param value the receiver variable for the removed value
 *
 * @return    true if the map has been changed, false otherwise
 */
gboolean gee_map_remove (GeeMap* self, gconstpointer key, gpointer* value) {
	return GEE_MAP_GET_INTERFACE (self)->remove (self, key, value);
}


/**
 * Removes all items from this collection. Must not be called on
 * read-only collections.
 */
void gee_map_clear (GeeMap* self) {
	GEE_MAP_GET_INTERFACE (self)->clear (self);
}


/**
 * Inserts all items that are contained in the input map to this map.
 *
 *  @param map the map which items are inserted to this map
 */
void gee_map_set_all (GeeMap* self, GeeMap* map) {
	GEE_MAP_GET_INTERFACE (self)->set_all (self, map);
}


/**
 * Removes all items from this map that are common to the input map 
 * and this map.
 *
 *  @param map the map which common items are deleted from this map
 */
gboolean gee_map_remove_all (GeeMap* self, GeeMap* map) {
	return GEE_MAP_GET_INTERFACE (self)->remove_all (self, map);
}


/**
 * Returns true it this map contains all items as the input map.
 *
 * @param map the map which items will be compared with this map.
 */
gboolean gee_map_contains_all (GeeMap* self, GeeMap* map) {
	return GEE_MAP_GET_INTERFACE (self)->contains_all (self, map);
}


gint gee_map_get_size (GeeMap* self) {
	return GEE_MAP_GET_INTERFACE (self)->get_size (self);
}


gboolean gee_map_get_is_empty (GeeMap* self) {
	return GEE_MAP_GET_INTERFACE (self)->get_is_empty (self);
}


static void gee_map_base_init (GeeMapIface * iface) {
	static gboolean initialized = FALSE;
	if (!initialized) {
		initialized = TRUE;
		g_object_interface_install_property (iface, g_param_spec_int ("size", "size", "size", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
		g_object_interface_install_property (iface, g_param_spec_boolean ("is-empty", "is-empty", "is-empty", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	}
}


GType gee_map_get_type (void) {
	static GType gee_map_type_id = 0;
	if (gee_map_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeMapIface), (GBaseInitFunc) gee_map_base_init, (GBaseFinalizeFunc) NULL, (GClassInitFunc) NULL, (GClassFinalizeFunc) NULL, NULL, 0, 0, (GInstanceInitFunc) NULL, NULL };
		gee_map_type_id = g_type_register_static (G_TYPE_INTERFACE, "GeeMap", &g_define_type_info, 0);
		g_type_interface_add_prerequisite (gee_map_type_id, G_TYPE_OBJECT);
	}
	return gee_map_type_id;
}




