/* testlinkedlist.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Authors:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Mark Lee <marklee@src.gnome.org> (port to LinkedList)
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>




void test_linkedlist_get (void);
void test_linkedlist_set (void);
void test_linkedlist_insert (void);
void test_linkedlist_remove_at (void);
void test_linkedlist_index_of (void);
void test_linkedlist_slice (void);
void test_linkedlist_add (void);
void test_linkedlist_clear (void);
void test_linkedlist_contains (void);
void test_linkedlist_remove (void);
void test_linkedlist_size (void);
void test_linkedlist_iterator (void);
static void _test_linkedlist_get_gcallback (void);
static void _test_linkedlist_set_gcallback (void);
static void _test_linkedlist_insert_gcallback (void);
static void _test_linkedlist_remove_at_gcallback (void);
static void _test_linkedlist_index_of_gcallback (void);
static void _test_linkedlist_slice_gcallback (void);
static void _test_linkedlist_add_gcallback (void);
static void _test_linkedlist_clear_gcallback (void);
static void _test_linkedlist_contains_gcallback (void);
static void _test_linkedlist_remove_gcallback (void);
static void _test_linkedlist_size_gcallback (void);
static void _test_linkedlist_iterator_gcallback (void);
void _main (char** args, int args_length1);
static int _vala_strcmp0 (const char * str1, const char * str2);



void test_linkedlist_get (void) {
	GeeLinkedList* linkedlistOfString;
	char* _tmp1_;
	char* _tmp4_;
	char* _tmp5_;
	char* _tmp6_;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check get for empty list*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp0_;
		_tmp0_ = NULL;
		_tmp0_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0);
		_tmp0_ = (g_free (_tmp0_), NULL);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check get for valid index in list with one element*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "1");
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "1") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	/* Check get for indexes out of range*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp2_;
		_tmp2_ = NULL;
		_tmp2_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1);
		_tmp2_ = (g_free (_tmp2_), NULL);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check get for invalid index number*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp3_;
		_tmp3_ = NULL;
		_tmp3_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, -1);
		_tmp3_ = (g_free (_tmp3_), NULL);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check get for valid indexes in list with multiple element*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "3");
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "1") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "2") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	_tmp6_ = NULL;
	g_assert (_vala_strcmp0 (_tmp6_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "3") == 0);
	_tmp6_ = (g_free (_tmp6_), NULL);
	/* Check get if list is cleared and empty again*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlistOfString);
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		char* _tmp7_;
		_tmp7_ = NULL;
		_tmp7_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0);
		_tmp7_ = (g_free (_tmp7_), NULL);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
}


void test_linkedlist_set (void) {
	GeeLinkedList* linkedlistOfString;
	char* _tmp0_;
	char* _tmp1_;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check set when list is empty.*/
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 0);
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_set ((GeeAbstractList*) linkedlistOfString, 0, "0");
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 0);
	/* Check set when one item is in list*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "1");
	/* Add item "1"*/
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 1);
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "1") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	gee_abstract_list_set ((GeeAbstractList*) linkedlistOfString, 0, "2");
	/* Set the item to value 2*/
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 1);
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "2") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	/* Check set when index out of range*/
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 1);
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_set ((GeeAbstractList*) linkedlistOfString, 1, "0");
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 1);
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
}


void test_linkedlist_insert (void) {
	GeeLinkedList* linkedlistOfString;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	char* _tmp6_;
	char* _tmp7_;
	char* _tmp8_;
	char* _tmp9_;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check inserting in empty list
	 Inserting at index 1*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_insert ((GeeAbstractList*) linkedlistOfString, 1, "0");
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Inserting at index 0*/
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 0);
	gee_abstract_list_insert ((GeeAbstractList*) linkedlistOfString, 0, "10");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 1);
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "10") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	/* Check insert to the beginning*/
	gee_abstract_list_insert ((GeeAbstractList*) linkedlistOfString, 0, "5");
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "5") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "10") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	/* Check insert in between*/
	gee_abstract_list_insert ((GeeAbstractList*) linkedlistOfString, 1, "7");
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "5") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "7") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "10") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	/* Check insert into index out of current range*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_insert ((GeeAbstractList*) linkedlistOfString, 4, "20");
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check insert to the end*/
	gee_abstract_list_insert ((GeeAbstractList*) linkedlistOfString, 3, "20");
	_tmp6_ = NULL;
	g_assert (_vala_strcmp0 (_tmp6_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "5") == 0);
	_tmp6_ = (g_free (_tmp6_), NULL);
	_tmp7_ = NULL;
	g_assert (_vala_strcmp0 (_tmp7_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "7") == 0);
	_tmp7_ = (g_free (_tmp7_), NULL);
	_tmp8_ = NULL;
	g_assert (_vala_strcmp0 (_tmp8_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "10") == 0);
	_tmp8_ = (g_free (_tmp8_), NULL);
	_tmp9_ = NULL;
	g_assert (_vala_strcmp0 (_tmp9_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 3), "20") == 0);
	_tmp9_ = (g_free (_tmp9_), NULL);
	/* Check insert into invalid index*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_insert ((GeeAbstractList*) linkedlistOfString, -1, "0");
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
}


void test_linkedlist_remove_at (void) {
	GeeLinkedList* linkedlistOfString;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	char* _tmp6_;
	char* _tmp7_;
	char* _tmp8_;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check removing in empty list*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_remove_at ((GeeAbstractList*) linkedlistOfString, 0);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_remove_at ((GeeAbstractList*) linkedlistOfString, 1);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* add 5 items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "3");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "4");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "5");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 5);
	/* Check remove_at first*/
	gee_abstract_list_remove_at ((GeeAbstractList*) linkedlistOfString, 0);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 4);
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "2") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "3") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "4") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 3), "5") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	/* Check remove_at last*/
	gee_abstract_list_remove_at ((GeeAbstractList*) linkedlistOfString, 3);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 3);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "2") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "3") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	_tmp6_ = NULL;
	g_assert (_vala_strcmp0 (_tmp6_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "4") == 0);
	_tmp6_ = (g_free (_tmp6_), NULL);
	/* Check remove_at in between*/
	gee_abstract_list_remove_at ((GeeAbstractList*) linkedlistOfString, 1);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 2);
	_tmp7_ = NULL;
	g_assert (_vala_strcmp0 (_tmp7_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "2") == 0);
	_tmp7_ = (g_free (_tmp7_), NULL);
	_tmp8_ = NULL;
	g_assert (_vala_strcmp0 (_tmp8_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "4") == 0);
	_tmp8_ = (g_free (_tmp8_), NULL);
	/* Check remove_at when index out of range*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_remove_at ((GeeAbstractList*) linkedlistOfString, 2);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check remove_at when invalid index*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_abstract_list_remove_at ((GeeAbstractList*) linkedlistOfString, -1);
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
}


void test_linkedlist_index_of (void) {
	GeeLinkedList* linkedlistOfString;
	GeeLinkedList* linkedlistOfInt;
	GeeLinkedList* linkedlistOfObjects;
	GObject* object1;
	GObject* object2;
	GObject* object3;
	GObject* object4;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check empty list*/
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "one") == (-1));
	/* Check one item*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "one");
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "one") == 0);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "two") == (-1));
	/* Check more items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "two");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "three");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "four");
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "one") == 0);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "two") == 1);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "three") == 2);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "four") == 3);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfString, "five") == (-1));
	/* Check list of ints*/
	linkedlistOfInt = gee_linked_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	/* Check more items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (1));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (2));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (3));
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfInt, GINT_TO_POINTER (1)) == 0);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfInt, GINT_TO_POINTER (2)) == 1);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfInt, GINT_TO_POINTER (3)) == 2);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfInt, GINT_TO_POINTER (4)) == (-1));
	/* Check list of objects*/
	linkedlistOfObjects = gee_linked_list_new (G_TYPE_OBJECT, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	object1 = g_object_new (G_TYPE_OBJECT, NULL);
	object2 = g_object_new (G_TYPE_OBJECT, NULL);
	object3 = g_object_new (G_TYPE_OBJECT, NULL);
	object4 = g_object_new (G_TYPE_OBJECT, NULL);
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfObjects, object1);
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfObjects, object2);
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfObjects, object3);
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfObjects, object4);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfObjects, object1) == 0);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfObjects, object2) == 1);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfObjects, object3) == 2);
	g_assert (gee_abstract_list_index_of ((GeeAbstractList*) linkedlistOfObjects, object4) == 3);
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
	(linkedlistOfInt == NULL) ? NULL : (linkedlistOfInt = (g_object_unref (linkedlistOfInt), NULL));
	(linkedlistOfObjects == NULL) ? NULL : (linkedlistOfObjects = (g_object_unref (linkedlistOfObjects), NULL));
	(object1 == NULL) ? NULL : (object1 = (g_object_unref (object1), NULL));
	(object2 == NULL) ? NULL : (object2 = (g_object_unref (object2), NULL));
	(object3 == NULL) ? NULL : (object3 = (g_object_unref (object3), NULL));
	(object4 == NULL) ? NULL : (object4 = (g_object_unref (object4), NULL));
}


void test_linkedlist_slice (void) {
	GeeLinkedList* linkedlistOfString;
	GeeList* slicedLinkedListOfString;
	GeeList* _tmp0_;
	GeeList* _tmp1_;
	char* _tmp2_;
	GeeList* _tmp3_;
	char* _tmp4_;
	GeeList* _tmp5_;
	char* _tmp6_;
	char* _tmp7_;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	slicedLinkedListOfString = NULL;
	/* Check empty list*/
	_tmp0_ = NULL;
	g_assert (gee_collection_get_size ((GeeCollection*) (_tmp0_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, 0, 0))) == 0);
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL));
	/* Check one item*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "one");
	_tmp1_ = NULL;
	slicedLinkedListOfString = (_tmp1_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, 0, 1), (slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL)), _tmp1_);
	g_assert (gee_collection_get_size ((GeeCollection*) slicedLinkedListOfString) == 1);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_list_get (slicedLinkedListOfString, 0), "one") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	/* Check more items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "two");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "three");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "four");
	_tmp3_ = NULL;
	slicedLinkedListOfString = (_tmp3_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, 1, 2), (slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL)), _tmp3_);
	g_assert (gee_collection_get_size ((GeeCollection*) slicedLinkedListOfString) == 1);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_list_get (slicedLinkedListOfString, 0), "two") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	slicedLinkedListOfString = (_tmp5_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, 2, 4), (slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL)), _tmp5_);
	g_assert (gee_collection_get_size ((GeeCollection*) slicedLinkedListOfString) == 2);
	_tmp6_ = NULL;
	g_assert (_vala_strcmp0 (_tmp6_ = (char*) gee_list_get (slicedLinkedListOfString, 0), "three") == 0);
	_tmp6_ = (g_free (_tmp6_), NULL);
	_tmp7_ = NULL;
	g_assert (_vala_strcmp0 (_tmp7_ = (char*) gee_list_get (slicedLinkedListOfString, 1), "four") == 0);
	_tmp7_ = (g_free (_tmp7_), NULL);
	/* Check slice when start/stop out of range*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		GeeList* _tmp8_;
		_tmp8_ = NULL;
		_tmp8_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, -1, 0);
		(_tmp8_ == NULL) ? NULL : (_tmp8_ = (g_object_unref (_tmp8_), NULL));
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		(slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		GeeList* _tmp9_;
		_tmp9_ = NULL;
		_tmp9_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, 0, -1);
		(_tmp9_ == NULL) ? NULL : (_tmp9_ = (g_object_unref (_tmp9_), NULL));
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		(slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check slice when start > stop*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		GeeList* _tmp10_;
		_tmp10_ = NULL;
		_tmp10_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, 1, 0);
		(_tmp10_ == NULL) ? NULL : (_tmp10_ = (g_object_unref (_tmp10_), NULL));
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		(slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	/* Check slice when stop > size*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		GeeList* _tmp11_;
		_tmp11_ = NULL;
		_tmp11_ = gee_abstract_list_slice ((GeeAbstractList*) linkedlistOfString, 1, 5);
		(_tmp11_ == NULL) ? NULL : (_tmp11_ = (g_object_unref (_tmp11_), NULL));
		(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
		(slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
	(slicedLinkedListOfString == NULL) ? NULL : (slicedLinkedListOfString = (g_object_unref (slicedLinkedListOfString), NULL));
}


void test_linkedlist_add (void) {
	GeeLinkedList* linkedlistOfString;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	GeeLinkedList* linkedlistOfInt;
	GeeLinkedList* linkedlistOfGLibObject;
	GObject* fooObject;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "42"));
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 1);
	/* check for correct order of elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "43");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "44");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "45");
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "42") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "43") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "44") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 3), "45") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 4);
	/* check adding of ints*/
	linkedlistOfInt = gee_linked_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (42));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (42)));
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfInt) == 1);
	/* check adding of objects*/
	linkedlistOfGLibObject = gee_linked_list_new (G_TYPE_OBJECT, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	fooObject = g_object_new (G_TYPE_OBJECT, NULL);
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfGLibObject, fooObject);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfGLibObject, fooObject));
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfGLibObject) == 1);
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
	(linkedlistOfInt == NULL) ? NULL : (linkedlistOfInt = (g_object_unref (linkedlistOfInt), NULL));
	(linkedlistOfGLibObject == NULL) ? NULL : (linkedlistOfGLibObject = (g_object_unref (linkedlistOfGLibObject), NULL));
	(fooObject == NULL) ? NULL : (fooObject = (g_object_unref (fooObject), NULL));
}


void test_linkedlist_clear (void) {
	GeeLinkedList* linkedlistOfString;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 0);
	/* Check clear on empty list*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlistOfString);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 0);
	/* Check clear one item*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 1);
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlistOfString);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 0);
	/* Check clear multiple items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "1");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "2");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "3");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 3);
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlistOfString);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 0);
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
}


void test_linkedlist_contains (void) {
	GeeLinkedList* linkedlistOfString;
	GeeLinkedList* linkedlistOfInt;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check on empty list*/
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "1"));
	/* Check items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "10");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "10"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "20"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "30"));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "20");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "10"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "20"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "30"));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "30");
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "10"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "20"));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "30"));
	/* Clear and recheck*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlistOfString);
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "10"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "20"));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "30"));
	linkedlistOfInt = gee_linked_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	/* Check items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (10));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (10)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (30)));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (20));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (30)));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (30));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (20)));
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (30)));
	/* Clear and recheck*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlistOfInt);
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (10)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (30)));
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
	(linkedlistOfInt == NULL) ? NULL : (linkedlistOfInt = (g_object_unref (linkedlistOfInt), NULL));
}


void test_linkedlist_remove (void) {
	GeeLinkedList* linkedlistOfString;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	char* _tmp6_;
	char* _tmp7_;
	char* _tmp8_;
	GeeLinkedList* linkedlistOfInt;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check remove if list is empty*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfString, "42");
	/* Add 5 same elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	/* Check remove one element*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 4);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "42"));
	/* Check remove another one element*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 3);
	g_assert (gee_abstract_collection_contains ((GeeAbstractCollection*) linkedlistOfString, "42"));
	/* Clear the list to start from scratch*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlistOfString);
	/* Add 5 different elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "43");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "44");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "45");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 5);
	/* Check remove first*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 4);
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "43") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "44") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "45") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 3), "46") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	/* Check remove last*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 3);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "43") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "44") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	_tmp6_ = NULL;
	g_assert (_vala_strcmp0 (_tmp6_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 2), "45") == 0);
	_tmp6_ = (g_free (_tmp6_), NULL);
	/* Check remove in between*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfString, "44");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfString) == 2);
	_tmp7_ = NULL;
	g_assert (_vala_strcmp0 (_tmp7_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 0), "43") == 0);
	_tmp7_ = (g_free (_tmp7_), NULL);
	_tmp8_ = NULL;
	g_assert (_vala_strcmp0 (_tmp8_ = (char*) gee_abstract_list_get ((GeeAbstractList*) linkedlistOfString, 1), "45") == 0);
	_tmp8_ = (g_free (_tmp8_), NULL);
	/* Check removing of int element*/
	linkedlistOfInt = gee_linked_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	/* Add 5 different elements*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (42));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (43));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (44));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (45));
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfInt) == 5);
	/* Remove first*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (42));
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfInt) == 4);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 0)) == 43);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 1)) == 44);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 2)) == 45);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 3)) == 46);
	/* Remove last*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfInt) == 3);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 0)) == 43);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 1)) == 44);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 2)) == 45);
	/* Remove in between*/
	gee_abstract_collection_remove ((GeeAbstractCollection*) linkedlistOfInt, GINT_TO_POINTER (44));
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlistOfInt) == 2);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 0)) == 43);
	g_assert (GPOINTER_TO_INT (gee_abstract_list_get ((GeeAbstractList*) linkedlistOfInt, 1)) == 45);
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
	(linkedlistOfInt == NULL) ? NULL : (linkedlistOfInt = (g_object_unref (linkedlistOfInt), NULL));
}


void test_linkedlist_size (void) {
	GeeLinkedList* linkedlist;
	linkedlist = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check empty list*/
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlist) == 0);
	/* Check when one item*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlist, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlist) == 1);
	/* Check when more items*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlist, "2");
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlist) == 2);
	/* Check when items cleared*/
	gee_abstract_collection_clear ((GeeAbstractCollection*) linkedlist);
	g_assert (gee_collection_get_size ((GeeCollection*) linkedlist) == 0);
	(linkedlist == NULL) ? NULL : (linkedlist = (g_object_unref (linkedlist), NULL));
}


void test_linkedlist_iterator (void) {
	GeeLinkedList* linkedlistOfString;
	GeeIterator* iterator;
	GeeIterator* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	linkedlistOfString = gee_linked_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	/* Check iterate empty list*/
	iterator = gee_abstract_collection_iterator ((GeeAbstractCollection*) linkedlistOfString);
	g_assert (!gee_iterator_next (iterator));
	/* Check iterate list*/
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "42");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "43");
	gee_abstract_collection_add ((GeeAbstractCollection*) linkedlistOfString, "44");
	_tmp0_ = NULL;
	iterator = (_tmp0_ = gee_abstract_collection_iterator ((GeeAbstractCollection*) linkedlistOfString), (iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL)), _tmp0_);
	g_assert (gee_iterator_next (iterator));
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_iterator_get (iterator), "42") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	g_assert (gee_iterator_next (iterator));
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_iterator_get (iterator), "43") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	g_assert (gee_iterator_next (iterator));
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_iterator_get (iterator), "44") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	g_assert (!gee_iterator_next (iterator));
	(linkedlistOfString == NULL) ? NULL : (linkedlistOfString = (g_object_unref (linkedlistOfString), NULL));
	(iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL));
}


static void _test_linkedlist_get_gcallback (void) {
	test_linkedlist_get ();
}


static void _test_linkedlist_set_gcallback (void) {
	test_linkedlist_set ();
}


static void _test_linkedlist_insert_gcallback (void) {
	test_linkedlist_insert ();
}


static void _test_linkedlist_remove_at_gcallback (void) {
	test_linkedlist_remove_at ();
}


static void _test_linkedlist_index_of_gcallback (void) {
	test_linkedlist_index_of ();
}


static void _test_linkedlist_slice_gcallback (void) {
	test_linkedlist_slice ();
}


static void _test_linkedlist_add_gcallback (void) {
	test_linkedlist_add ();
}


static void _test_linkedlist_clear_gcallback (void) {
	test_linkedlist_clear ();
}


static void _test_linkedlist_contains_gcallback (void) {
	test_linkedlist_contains ();
}


static void _test_linkedlist_remove_gcallback (void) {
	test_linkedlist_remove ();
}


static void _test_linkedlist_size_gcallback (void) {
	test_linkedlist_size ();
}


static void _test_linkedlist_iterator_gcallback (void) {
	test_linkedlist_iterator ();
}


void _main (char** args, int args_length1) {
	g_test_init (&args_length1, &args, NULL);
	/* Methods of List interface*/
	g_test_add_func ("/LinkedList/List/get", (GCallback) _test_linkedlist_get_gcallback);
	g_test_add_func ("/LinkedList/List/set", (GCallback) _test_linkedlist_set_gcallback);
	g_test_add_func ("/LinkedList/List/insert", (GCallback) _test_linkedlist_insert_gcallback);
	g_test_add_func ("/LinkedList/List/remove_at", (GCallback) _test_linkedlist_remove_at_gcallback);
	g_test_add_func ("/LinkedList/List/index_of", (GCallback) _test_linkedlist_index_of_gcallback);
	g_test_add_func ("/LinkedList/List/slice", (GCallback) _test_linkedlist_slice_gcallback);
	/* Methods of Collection interface*/
	g_test_add_func ("/LinkedList/Collection/add", (GCallback) _test_linkedlist_add_gcallback);
	g_test_add_func ("/LinkedList/Collection/clear", (GCallback) _test_linkedlist_clear_gcallback);
	g_test_add_func ("/LinkedList/Collection/contains", (GCallback) _test_linkedlist_contains_gcallback);
	g_test_add_func ("/LinkedList/Collection/remove", (GCallback) _test_linkedlist_remove_gcallback);
	g_test_add_func ("/LinkedList/Collection/size", (GCallback) _test_linkedlist_size_gcallback);
	/* Methods of Iterable interface*/
	g_test_add_func ("/LinkedList/Iterable/iterator", (GCallback) _test_linkedlist_iterator_gcallback);
	g_test_run ();
}


int main (int argc, char ** argv) {
	g_type_init ();
	_main (argv, argc);
	return 0;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




