/*
******************************************************************************

   Generic color default library implementation

   Copyright (C) 2000-2001 Christoph Egger	[Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/


#include <stdlib.h>

#include "color.h"



/* Gamma map manipulation
 */

/* RGBA */
int GGI_color_GetRGBAGamma(ggi_visual_t vis,ggi_float *r,ggi_float *g,ggi_float *b)
{
	return ggiGetGamma(vis, r,g,b);
}	/* GGI_color_GetRGBAGamma */


int GGI_color_SetRGBAGamma(ggi_visual_t vis,ggi_float r,ggi_float g,ggi_float b)
{
	return ggiSetGamma(vis, r,g,b);
}	/* GGI_color_SetRGBAGamma */


int GGI_color_GetRGBAGammaMap(ggi_visual_t vis,int s,int len,gcp_RGBAcolor *gammamap)
{
	return ggiGetGammaMap(vis, s,len, gammamap);
}	/* GGI_color_GetRGBAGammaMap */


int GGI_color_SetRGBAGammaMap(ggi_visual_t vis,int s,int len,gcp_RGBAcolor *gammamap)
{
	return ggiSetGammaMap(vis, s, len, gammamap);
}	/* GGI_color_SetRGBAGammaMap */





/* YUV */
int GGI_color_GetYUVGamma(ggi_visual_t vis,ggi_float *y,ggi_float *u,ggi_float *v)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_YUVcolor yuv;

	rc = gcpGetRGBAGamma(vis, &r,&g,&b);
	if (rc != 0) goto err0;

	rgba.r = r;
	rgba.g = g;
	rgba.b = b;
	rgba.a = 0xFFFF;

	rc = gcpRGBA2YUV(vis, &rgba, &yuv);
	LIB_ASSERT(rc == 0);

	*y = yuv.y;
	*u = yuv.u;
	*v = yuv.v;

err0:
	return rc;
}	/* GGI_color_GetYUVGamma */


int GGI_color_SetYUVGamma(ggi_visual_t vis,ggi_float y,ggi_float u,ggi_float v)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_YUVcolor yuv;

	yuv.y = y;
	yuv.u = u;
	yuv.v = v;

	rc = gcpYUV2RGBA(vis, &yuv, &rgba);
	if (rc != 0) goto err0;

	r = rgba.r;
	g = rgba.g;
	b = rgba.b;

	rc = gcpSetRGBAGamma(vis, r,g,b);

err0:
	return rc;
}	/* GGI_color_SetYUVGamma */


int GGI_color_GetYUVGammaMap(ggi_visual_t vis,int s,int len,gcp_YUVcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	rc = gcpGetRGBAGammaMap(vis, s, len, gmap);
	if (rc != 0) goto err1;

	for (i = 0; i < len; i++) {
			rc = gcpRGBA2YUV(vis, &gmap[i], &gammamap[i]);
			LIB_ASSERT(rc == 0);
	}	/* for */

err1:
	free(gmap);
err0:
	return rc;
}	/* GGI_color_GetYUVGammaMap */


int GGI_color_SetYUVGammaMap(ggi_visual_t vis,int s,int len,gcp_YUVcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	for (i = 0; i < len; i++) {
		rc = gcpYUV2RGBA(vis, &gammamap[i], &gmap[i]);
		LIB_ASSERT(rc == 0);
	}	/* for */

	rc = gcpSetRGBAGammaMap(vis, s, len, gmap);

	free(gmap);
err0:
	return rc;
}	/* GGI_color_SetYUVGammaMap */





/* CMYK */
int GGI_color_GetCMYKGamma(ggi_visual_t vis,ggi_float *c,ggi_float *m,ggi_float *y,ggi_float *k)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_CMYKcolor cmyk;

	rc = gcpGetRGBAGamma(vis, &r,&g,&b);
	if (rc != 0) goto err0;

	rgba.r = r;
	rgba.g = g;
	rgba.b = b;
	rgba.a = 0xFFFF;

	rc = gcpRGBA2CMYK(vis, &rgba, &cmyk);
	LIB_ASSERT(rc == 0);

	*c = cmyk.c;
	*m = cmyk.m;
	*y = cmyk.y;
	*k = cmyk.k;

err0:
	return rc;
}	/* GGI_color_GetCMYKGamma */


int GGI_color_SetCMYKGamma(ggi_visual_t vis,ggi_float c,ggi_float m,ggi_float y,ggi_float k)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_CMYKcolor cmyk;

	cmyk.c = c;
	cmyk.m = m;
	cmyk.y = y;
	cmyk.k = k;

	rc = gcpCMYK2RGBA(vis, &cmyk, &rgba);
	if (rc != 0) goto err0;

	r = rgba.r;
	g = rgba.g;
	b = rgba.b;

	rc = gcpSetRGBAGamma(vis, r,g,b);

err0:
	return rc;
}	/* GGI_color_SetCMYKGamma */


int GGI_color_GetCMYKGammaMap(ggi_visual_t vis,int s,int len,gcp_CMYKcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	rc = gcpGetRGBAGammaMap(vis, s, len, gmap);
	if (rc != 0) goto err1;

	for (i = 0; i < len; i++) {
			rc = gcpRGBA2CMYK(vis, &gmap[i], &gammamap[i]);
			LIB_ASSERT(rc == 0);
	}	/* for */

err1:
	free(gmap);
err0:
	return rc;
}	/* GGI_color_GetCMYKGammaMap */


int GGI_color_SetCMYKGammaMap(ggi_visual_t vis,int s,int len,gcp_CMYKcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	for (i = 0; i < len; i++) {
		rc = gcpCMYK2RGBA(vis, &gammamap[i], &gmap[i]);
		LIB_ASSERT(rc == 0);
	}	/* for */

	rc = gcpSetRGBAGammaMap(vis, s, len, gmap);

	free(gmap);
err0:
	return rc;
}	/* GGI_color_SetCMYKGammaMap */





/* HSV */
int GGI_color_GetHSVGamma(ggi_visual_t vis,ggi_float *h,ggi_float *s,ggi_float *v)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_HSVcolor hsv;

	rc = gcpGetRGBAGamma(vis, &r,&g,&b);
	if (rc != 0) goto err0;

	rgba.r = r;
	rgba.g = g;
	rgba.b = b;
	rgba.a = 0xFFFF;

	rc = gcpRGBA2HSV(vis, &rgba, &hsv);
	LIB_ASSERT(rc == 0);

	*h = hsv.h;
	*s = hsv.s;
	*v = hsv.v;

err0:
	return rc;
}	/* GGI_color_GetHSVGamma */


int GGI_color_SetHSVGamma(ggi_visual_t vis,ggi_float h,ggi_float s,ggi_float v)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_HSVcolor hsv;

	hsv.h = h;
	hsv.s = s;
	hsv.v = v;

	rc = gcpHSV2RGBA(vis, &hsv, &rgba);
	if (rc != 0) goto err0;

	r = rgba.r;
	g = rgba.g;
	b = rgba.b;

	rc = gcpSetRGBAGamma(vis, r,g,b);

err0:
	return rc;
}	/* GGI_color_SetHSVGamma */


int GGI_color_GetHSVGammaMap(ggi_visual_t vis,int s,int len,gcp_HSVcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	rc = gcpGetRGBAGammaMap(vis, s, len, gmap);
	if (rc != 0) goto err1;

	for (i = 0; i < len; i++) {
			rc = gcpRGBA2HSV(vis, &gmap[i], &gammamap[i]);
			LIB_ASSERT(rc == 0);
	}	/* for */

err1:
	free(gmap);
err0:
	return rc;
}	/* GGI_color_GetHSVGammaMap */


int GGI_color_SetHSVGammaMap(ggi_visual_t vis,int s,int len,gcp_HSVcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	for (i = 0; i < len; i++) {
		rc = gcpHSV2RGBA(vis, &gammamap[i], &gmap[i]);
		LIB_ASSERT(rc == 0);
	}	/* for */

	rc = gcpSetRGBAGammaMap(vis, s, len, gmap);

	free(gmap);
err0:
	return rc;
}	/* GGI_color_SetHSVGammaMap */



/* Kodak PhotoCD */
int GGI_color_GetYCCGamma(ggi_visual_t vis,ggi_float *y,ggi_float *c1,ggi_float *c2)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_YCCcolor ycc;

	rc = gcpGetRGBAGamma(vis, &r,&g,&b);
	if (rc != 0) goto err0;

	rgba.r = r;
	rgba.g = g;
	rgba.b = b;
	rgba.a = 0xFFFF;

	rc = gcpRGBA2YCC(vis, &rgba, &ycc);
	LIB_ASSERT(rc == 0);

	*y = ycc.y;
	*c1 = ycc.c1;
	*c2 = ycc.c2;

err0:
	return rc;
}	/* GGI_color_GetYCCGamma */


int GGI_color_SetYCCGamma(ggi_visual_t vis,ggi_float y,ggi_float c1,ggi_float c2)
{
	int rc;
	ggi_float r,g,b;
	gcp_RGBAcolor rgba;
	gcp_YCCcolor ycc;

	ycc.y = y;
	ycc.c1 = c1;
	ycc.c2 = c2;

	rc = gcpYCC2RGBA(vis, &ycc, &rgba);
	if (rc != 0) goto err0;

	r = rgba.r;
	g = rgba.g;
	b = rgba.b;

	rc = gcpSetRGBAGamma(vis, r,g,b);

err0:
	return rc;
}	/* GGI_color_SetYCCGamma */


int GGI_color_GetYCCGammaMap(ggi_visual_t vis,int s,int len,gcp_YCCcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	rc = gcpGetRGBAGammaMap(vis, s, len, gmap);
	if (rc != 0) goto err1;

	for (i = 0; i < len; i++) {
			rc = gcpRGBA2YCC(vis, &gmap[i], &gammamap[i]);
			LIB_ASSERT(rc == 0);
	}	/* for */

err1:
	free(gmap);
err0:
	return rc;
}	/* GGI_color_GetYCCGammaMap */


int GGI_color_SetYCCGammaMap(ggi_visual_t vis,int s,int len,gcp_YCCcolor *gammamap)
{
	int rc;
	int i;
	gcp_RGBAcolor *gmap;

	gmap = malloc(sizeof(gcp_RGBAcolor) * len);
	if (!gmap) {
		rc = -1;
		goto err0;
	}	/* if */

	for (i = 0; i < len; i++) {
		rc = gcpYCC2RGBA(vis, &gammamap[i], &gmap[i]);
		LIB_ASSERT(rc == 0);
	}	/* for */

	rc = gcpSetRGBAGammaMap(vis, s, len, gmap);

	free(gmap);
err0:
	return rc;
}	/* GGI_color_SetYCCGammaMap */

