// ENGINE.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This package is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This package is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this package; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

/*################################################################################################*/

#include "libghemicalconfig2.h"
#include "engine.h"

#include "v3d.h"
#include "model.h"

#include "local_i18n.h"
#include "notice.h"

#include <vector>
#include <sstream>
using namespace std;

/*################################################################################################*/

setup::setup(model * p1)
{
	mdl = p1;
	
	// initialize the tables...
	
	has_setup_tables = false;
	
	atmtab = NULL; natm = NOT_DEFINED;
	
	qm_atmtab = NULL; qm_natm = NOT_DEFINED;
	qm_bndtab = NULL; qm_nbnd = NOT_DEFINED;
	
	mm_atmtab = NULL; mm_natm = NOT_DEFINED;
	mm_bndtab = NULL; mm_nbnd = NOT_DEFINED;
	
	boundary_bndtab = NULL; boundary_nbnd = NOT_DEFINED;
	
	sf_atmtab = NULL; sf_natm = NOT_DEFINED;
	
	current_eng = NULL;
	current_eng_index = 0;
}

setup::~setup(void)
{
	DiscardCurrentEngine();
	DiscardSetupInfo();
}

void setup::DiscardSetupInfo(void)
{
	if (atmtab != NULL) { delete[] atmtab; atmtab = NULL;}
	
	if (qm_atmtab != NULL) { delete[] qm_atmtab; qm_atmtab = NULL; }
	if (qm_bndtab != NULL) { delete[] qm_bndtab; qm_bndtab = NULL; }
	
	if (mm_atmtab != NULL) { delete[] mm_atmtab; mm_atmtab = NULL; }
	if (mm_bndtab != NULL) { delete[] mm_bndtab; mm_bndtab = NULL; }
	
	if (boundary_bndtab != NULL) { delete[] boundary_bndtab; boundary_bndtab = NULL; }
		
	if (sf_atmtab != NULL) { delete[] sf_atmtab; sf_atmtab = NULL; }
	
	has_setup_tables = false;
}

void setup::UpdateSetupInfo(void)
{
	// discard the previous tables, if needed...
	
	DiscardSetupInfo();
	
	// clear all atom flags, and then bring them up-to-date.
	
	for (iter_al it1 = GetModel()->GetAtomsBegin();it1 != GetModel()->GetAtomsEnd();it1++)
	{
		(* it1).flags &= (~ ATOMFLAG_IS_QM_ATOM);
		(* it1).flags &= (~ ATOMFLAG_IS_MM_ATOM);
		(* it1).flags &= (~ ATOMFLAG_IS_SF_ATOM);
		(* it1).flags &= (~ ATOMFLAG_IS_HIDDEN);
	}
	
	UpdateAtomFlags();
	
	// start initializing the tables...
	
	natm = 0; i32s nbnd = 0;
	
	qm_natm = 0; qm_nbnd = 0;
	mm_natm = 0; mm_nbnd = 0;
	boundary_nbnd = 0;
	
	sf_natm = 0;
	
	// count the atoms and bonds that are taken into computations.
	// count the atoms and bonds that are taken into computations.
	// count the atoms and bonds that are taken into computations.
	
	iter_al ita = mdl->GetAtomsBegin();
	while (ita != mdl->GetAtomsEnd())
	{
		bool test1 = ((* ita).flags & (ATOMFLAG_IS_QM_ATOM | ATOMFLAG_IS_MM_ATOM | ATOMFLAG_IS_SF_ATOM));
		bool test2 = ((* ita).flags & ATOMFLAG_IS_HIDDEN);
		bool skip = (!test1 || test2);
		
		if (!skip)
		{
			natm++;
			
			if ((* ita).flags & ATOMFLAG_IS_QM_ATOM)
			{
				qm_natm++;
			}
			
			if ((* ita).flags & ATOMFLAG_IS_MM_ATOM)
			{
				mm_natm++;
			}
			
			if ((* ita).flags & ATOMFLAG_IS_SF_ATOM)
			{
				sf_natm++;
			}
		}
		
		ita++;
	}
	
	iter_bl itb = mdl->GetBondsBegin();
	while (itb != mdl->GetBondsEnd())
	{
		bool testA1 = ((* itb).atmr[0]->flags & (ATOMFLAG_IS_QM_ATOM | ATOMFLAG_IS_MM_ATOM));
		bool testA2 = ((* itb).atmr[0]->flags & ATOMFLAG_IS_HIDDEN);
		bool skipA = (!testA1 || testA2);
		
		bool testB1 = ((* itb).atmr[1]->flags & (ATOMFLAG_IS_QM_ATOM | ATOMFLAG_IS_MM_ATOM));
		bool testB2 = ((* itb).atmr[1]->flags & ATOMFLAG_IS_HIDDEN);
		bool skipB = (!testB1 || testB2);
		
		if (!skipA && !skipB)
		{
			nbnd++;
			
			if (((* itb).atmr[0]->flags & ATOMFLAG_IS_QM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_QM_ATOM))
			{
				qm_nbnd++;
			}
			
			if (((* itb).atmr[0]->flags & ATOMFLAG_IS_MM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_MM_ATOM))
			{
				mm_nbnd++;
			}
			
			bool tqmmm1 = (((* itb).atmr[0]->flags & ATOMFLAG_IS_QM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_MM_ATOM));
			bool tqmmm2 = (((* itb).atmr[0]->flags & ATOMFLAG_IS_MM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_QM_ATOM));
			if (tqmmm1 || tqmmm2)
			{
				boundary_nbnd++;
			}
		}
		
		itb++;
	}
	
	// bonds (in contrast to atoms) should only be counted in a single category; check!!!
	
	if (qm_nbnd + mm_nbnd + boundary_nbnd != nbnd)
	{
		ostringstream msg;
		msg << "bond counting mismatch : " << qm_nbnd << " + " << mm_nbnd << " + " << boundary_nbnd << " != " << nbnd << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
	// allocate the tables.
	
	atmtab = new ref_atom[natm];
	
	qm_atmtab = new ref_atom[qm_natm];
	qm_bndtab = new ref_bond[qm_nbnd];
	
	mm_atmtab = new ref_atom[mm_natm];
	mm_bndtab = new ref_bond[mm_nbnd];

	boundary_bndtab = new ref_bond[boundary_nbnd];
	
	sf_atmtab = new ref_atom[sf_natm];
	
	// fill the tables; BE SURE TO USE THE SAME RULES AS IN THE COUNTING STAGE!!!
	// fill the tables; BE SURE TO USE THE SAME RULES AS IN THE COUNTING STAGE!!!
	// fill the tables; BE SURE TO USE THE SAME RULES AS IN THE COUNTING STAGE!!!
	
	i32u ac = 0;
	i32u qm_ac = 0;
	i32u mm_ac = 0;
	i32u sf_ac = 0;
	
	ita = mdl->GetAtomsBegin();
	while (ita != mdl->GetAtomsEnd())
	{
		bool test1 = ((* ita).flags & (ATOMFLAG_IS_QM_ATOM | ATOMFLAG_IS_MM_ATOM | ATOMFLAG_IS_SF_ATOM));
		bool test2 = ((* ita).flags & ATOMFLAG_IS_HIDDEN);
		bool skip = (!test1 || test2);
		
		if (!skip)
		{
			(* ita).varind = ac;		// SET THE VARIABLE INDEX!!!
			atmtab[ac++] = & (* ita);
			
			if ((* ita).flags & ATOMFLAG_IS_QM_ATOM)
			{
				qm_atmtab[qm_ac++] = & (* ita);
			}
			
			if ((* ita).flags & ATOMFLAG_IS_MM_ATOM)
			{
				mm_atmtab[mm_ac++] = & (* ita);
			}
			
			if ((* ita).flags & ATOMFLAG_IS_SF_ATOM)
			{
				sf_atmtab[sf_ac++] = & (* ita);
			}
		}
		else (* ita).varind = NOT_DEFINED;
		
		ita++;
	}
	
	i32u qm_bc = 0;
	i32u mm_bc = 0;
	i32u boundary_bc = 0;
	
	itb = mdl->GetBondsBegin();
	while (itb != mdl->GetBondsEnd())
	{
		bool testA1 = ((* itb).atmr[0]->flags & (ATOMFLAG_IS_QM_ATOM | ATOMFLAG_IS_MM_ATOM));
		bool testA2 = ((* itb).atmr[0]->flags & ATOMFLAG_IS_HIDDEN);
		bool skipA = (!testA1 || testA2);
		
		bool testB1 = ((* itb).atmr[1]->flags & (ATOMFLAG_IS_QM_ATOM | ATOMFLAG_IS_MM_ATOM));
		bool testB2 = ((* itb).atmr[1]->flags & ATOMFLAG_IS_HIDDEN);
		bool skipB = (!testB1 || testB2);
		
		if (!skipA && !skipB)
		{
			if (((* itb).atmr[0]->flags & ATOMFLAG_IS_QM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_QM_ATOM))
			{
				qm_bndtab[qm_bc++] = & (* itb);
			}
			
			if (((* itb).atmr[0]->flags & ATOMFLAG_IS_MM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_MM_ATOM))
			{
				mm_bndtab[mm_bc++] = & (* itb);
			}
			
			bool tqmmm1 = (((* itb).atmr[0]->flags & ATOMFLAG_IS_QM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_MM_ATOM));
			bool tqmmm2 = (((* itb).atmr[0]->flags & ATOMFLAG_IS_MM_ATOM) && ((* itb).atmr[1]->flags & ATOMFLAG_IS_QM_ATOM));
			if (tqmmm1 || tqmmm2)
			{
				boundary_bndtab[boundary_bc++] = & (* itb);
			}
		}
		
		itb++;
	}
	
	// finally check that the counts match!!!
	// finally check that the counts match!!!
	// finally check that the counts match!!!
	
	if ((i32s) ac != natm)
	{
		ostringstream msg;
		msg << "atom mismatch : " << ac << " != " << natm << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
	if ((i32s) qm_ac != qm_natm)
	{
		ostringstream msg;
		msg << "qm-atom mismatch : " << qm_ac << " != " << qm_natm << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
	if ((i32s) qm_bc != qm_nbnd)
	{
		ostringstream msg;
		msg << "qm-bond mismatch : " << qm_bc << " != " << qm_nbnd << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
	if ((i32s) mm_ac != mm_natm)
	{
		ostringstream msg;
		msg << "mm-atom mismatch : " << mm_ac << " != " << mm_natm << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
	if ((i32s) mm_bc != mm_nbnd)
	{
		ostringstream msg;
		msg << "mm-bond mismatch : " << mm_bc << " != " << mm_nbnd << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
	if ((i32s) boundary_bc != boundary_nbnd)
	{
		ostringstream msg;
		msg << "boundary-bond mismatch : " << boundary_bc << " != " << boundary_nbnd << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
	if ((i32s) sf_ac != sf_natm)
	{
		ostringstream msg;
		msg << "sf-atom mismatch : " << sf_ac << " != " << sf_natm << ends;
		assertion_failed(__FILE__, __LINE__, msg.str().c_str());
	}
	
//////////////////////////////////////////////////
//////////////////////////////////////////////////
//cout << "atoms : " << qm_ac << " " << mm_ac << " " << sf_ac << "     bonds : " << qm_bc << " " << mm_bc << endl;	//CGI!
//////////////////////////////////////////////////
//////////////////////////////////////////////////
	
	has_setup_tables = true;
}

void setup::CreateCurrentEngine(void)
{
	DiscardCurrentEngine();
	current_eng = CreateEngineByIndex(current_eng_index);
}

void setup::DiscardCurrentEngine(void)
{
	if (current_eng != NULL)
	{
		delete current_eng;
		current_eng = NULL;
	}
}

engine * setup::CreateEngineByIDNumber(i32u id)
{
	i32u index = 0;
	while (index < GetEngineCount())
	{
		if (GetEngineIDNumber(index) == id) break;
		else index++;
	}
	
	if (index < GetEngineCount())
	{
		return CreateEngineByIndex(index);
	}
	else
	{
		cout << "WARNING : setup::CreateEngineByIDNumber() failed!" << endl;
		return NULL;
	}
}

/*################################################################################################*/

engine::engine(setup * p1, i32u p2)
{
	stp = p1;
	if (!stp->HasSetupTables()) assertion_failed(__FILE__, __LINE__, "no setup tables");
	
	// here it is very important that we take the atom count from setup::GetAtomCount(), not model::GetAtomCount()!!!
	// it is possible that these are not at all the same!!! some atoms might be hidden from the calculations.
	
	natm = stp->GetAtomCount();
	
	crd = new f64[natm * 3];
	
	if (p2 > 0) d1 = new f64[natm * 3];
	else d1 = NULL;
	
	if (p2 > 1) d2 = new f64[natm * natm * 9];
	else d2 = NULL;
	
	virial[0] = 0.0;
	virial[1] = 0.0;
	virial[2] = 0.0;
	
	update_neighbor_list = false;
	
	ECOMPcycles = 0;
	ECOMPstore_size = 0;
	ECOMPstore = NULL;
	
	if (stp->GetModel()->ecomp_enabled)
	{
		const int n = stp->GetModel()->ecomp_grp_names.size();
		ECOMPstore_size = n * (n + 1) / 2;
		
		ECOMPstore = new ecomp_data[ECOMPstore_size];
		
		ecomp_Reset();
	}
}

engine::~engine(void)
{
	delete[] crd;
	crd = NULL;
	
	if (d1 != NULL)
	{
		delete[] d1;
		d1 = NULL;
	}
	
	if (d2 != NULL)
	{
		delete[] d2;
		d2 = NULL;
	}
	
	if (ECOMPstore != NULL)
	{
		delete[] ECOMPstore;
		ECOMPstore = NULL;
	}
}

void engine::Check(i32u)
{
	const f64 delta = 0.000001;	// the finite difference step...
	
	Compute(1);
	f64 tmp1 = energy;
	
	f64 tmp2; f64 old;
	for (i32s n1 = 0;n1 < natm;n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			old = crd[n1 * 3 + n2];
			crd[n1 * 3 + n2] = old + delta;
			Compute(0); tmp2 = (energy - tmp1) / delta;
			crd[n1 * 3 + n2] = old;
			
			cout << n1 << ((char) ('x' + n2)) << " ";
			cout << "a = " << d1[n1 * 3 + n2] << " ";
			cout << "n = " << tmp2 << endl;
			
			if ((n1 % 5) == 4) cin >> old;
		}
	}
}

f64 engine::GetGradientVectorLength(void)
{
	f64 sum = 0.0;
	
	for (i32s n1 = 0;n1 < natm;n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			f64 tmp1 = d1[n1 * 3 + n2];
			sum += tmp1 * tmp1;
		}
	}
	
	return sqrt(sum);
}

void engine::ScaleCRD(f64 kx, f64 ky, f64 kz)
{
	atom ** glob_atmtab = GetSetup()->GetAtoms();
	for (i32s n1 = 0;n1 < natm;n1++)
	{
		if (glob_atmtab[n1]->flags & ATOMFLAG_USER_LOCKED) continue;
		
		crd[n1 * 3 + 0] *= kx;
		crd[n1 * 3 + 1] *= ky;
		crd[n1 * 3 + 2] *= kz;
	}
}

void engine::DoSHAKE(bool)
{
}

f64 engine::GetEnergy(void)
{
	return energy;
}

f64 engine::ReadCRD(int i)
{
	if (i < 0) return 0.0;
	if (i >= natm * 3) return 0.0;
	
	return crd[i];
}

void engine::ecomp_AddCycle(void)
{
	ECOMPcycles++;
}

void engine::ecomp_AddStore2(int gA, int gB, int sc, double value)
{
	const int gLO = (gA < gB ? gA : gB);
	const int gHI = (gA > gB ? gA : gB);
	
	const int index = (gHI * (gHI + 1) / 2) + gLO;
	
	if (index >= ECOMPstore_size) assertion_failed(__FILE__, __LINE__, "index overflow");
	
	ECOMPstore[index][sc] += value;
}

void engine::ecomp_AddStoreX(vector<int> & gv, int sc, double value)
{
	if (gv.size() < 1) return;
	
	if (gv.size() == 1)
	{
		ecomp_AddStore2(gv[0], gv[0], sc, value);
		return;
	}
	
	if (gv.size() == 2)
	{
		ecomp_AddStore2(gv[0], gv[1], sc, value);
		return;
	}
	
	vector<int> uniq;
	uniq.push_back(gv[0]);
	
	for (int n1 = 1;n1 < (int) gv.size();n1++)
	{
		bool is_uniq = true;
		for (int n2 = 0;n2 < (int) uniq.size();n2++)
		{
			if (uniq[n2] == gv[n1])
			{
				is_uniq = false;
				break;
			}
		}
		
		if (is_uniq) uniq.push_back(gv[n1]);
	}
	
	if (uniq.size() == 1)
	{
		ecomp_AddStore2(uniq[0], uniq[0], sc, value);
	}
	else
	{
		const int s = uniq.size();
		const int n = s * (s - 1) / 2;
		const double valX = value / (double) n;
		
		for (int x = 0;x < s - 1;x++)
		{
			for (int y = x + 1;y < s;y++)
			{
				ecomp_AddStore2(uniq[x], uniq[y], sc, valX);
			}
		}
	}
}

void engine::ecomp_Reset(void)
{
	ECOMPcycles = 0;
	
	for (int i = 0;i < ECOMPstore_size;i++)
	{
		ECOMPstore[i][ECOMP_DATA_IND_B_bs] = 0.0;
		ECOMPstore[i][ECOMP_DATA_IND_B_ab] = 0.0;
		ECOMPstore[i][ECOMP_DATA_IND_B_ti] = 0.0;
		ECOMPstore[i][ECOMP_DATA_IND_NB_lj] = 0.0;
		ECOMPstore[i][ECOMP_DATA_IND_NB_es] = 0.0;
	}
}

double engine::ecomp_ReadStore(int gA, int gB, int sc)
{
	const int gLO = (gA < gB ? gA : gB);
	const int gHI = (gA > gB ? gA : gB);
	
	const int index = (gHI * (gHI + 1) / 2) + gLO;
	
	if (index >= ECOMPstore_size) assertion_failed(__FILE__, __LINE__, "index overflow");
	
	return (ECOMPstore[index][sc] / (double) ECOMPcycles);
}

/*################################################################################################*/

fGL value_VDWSurf(engine * eng, fGL * crd, fGL * grad)
{
	return eng->GetVDWSurf(crd, grad);
}

fGL value_ESP(engine * eng, fGL * crd, fGL * grad)
{
	return eng->GetESP(crd, grad);
}

fGL value_ElDens(engine * eng, fGL * crd, fGL * grad)
{
	return eng->GetElDens(crd, grad);
}

fGL value_Orbital(engine * eng, fGL * crd, fGL * grad)
{
	return eng->GetOrbital(crd, grad);
}

fGL value_OrbDens(engine * eng, fGL * crd, fGL * grad)
{
	return eng->GetOrbDens(crd, grad);
}

// this #include macro is located here, because the eng1_sf class derived from the
// engine class and therefore it only can be introduced after the engine class itself...

#include "eng1_sf.h"

void CopyCRD(model * p1, engine * p2, i32u p3)
{
	if (p3 >= p1->cs_vector.size()) assertion_failed(__FILE__, __LINE__, "cs overflow");
	
	atom ** glob_atmtab = p2->GetSetup()->GetAtoms();
	for (i32s n1 = 0;n1 < p2->GetSetup()->GetAtomCount();n1++)
	{
		const fGL * cdata = glob_atmtab[n1]->GetCRD(p3);
		
		p2->crd[n1 * 3 + 0] = cdata[0];
		p2->crd[n1 * 3 + 1] = cdata[1];
		p2->crd[n1 * 3 + 2] = cdata[2];
	}
	
	// the rest is SF-related...
	// the rest is SF-related...
	// the rest is SF-related...
	
	eng1_sf * esf = dynamic_cast<eng1_sf *>(p2);
	setup1_sf * ssf = dynamic_cast<setup1_sf *>(p2->GetSetup());
	if (esf != NULL && ssf != NULL)
	{
		i32s bt3_counter = 0;
		
		for (i32u cc = 0;cc < ssf->chn_vector.size();cc++)
		{
			for (i32s rc = 1;rc < ((i32s) ssf->chn_vector[cc].res_vector.size()) - 2;rc++)
			{
				const fGL * prev = ssf->chn_vector[cc].res_vector[rc - 1].atmr[0]->GetCRD(p3);
				const fGL * curr = ssf->chn_vector[cc].res_vector[rc + 0].atmr[0]->GetCRD(p3);
				const fGL * next = ssf->chn_vector[cc].res_vector[rc + 1].atmr[0]->GetCRD(p3);
				
				atom * ref_to_C = ssf->chn_vector[cc].res_vector[rc + 0].peptide[2];
				atom * ref_to_O = ssf->chn_vector[cc].res_vector[rc + 0].peptide[3];
				
				v3d<fGL> v1(curr, prev); v3d<fGL> v2(curr, next);
				v3d<fGL> v3(ref_to_O->GetCRD(p3), ref_to_C->GetCRD(p3));
				fGL pbdd = v1.tor(v2, v3);
				
				if (bt3_counter >= (i32s) esf->bt3_vector.size()) assertion_failed(__FILE__, __LINE__, "bt3_counter overflow");
				
				esf->bt3_vector[bt3_counter++].pbdd = pbdd;
			}
		}
	}
	
	// ready!!!
	// ready!!!
	// ready!!!
}

void CopyCRD(engine * p1, model * p2, i32u p3)
{
	if (p3 >= p2->cs_vector.size()) assertion_failed(__FILE__, __LINE__, "cs overflow");
	
	atom ** glob_atmtab = p1->GetSetup()->GetAtoms();
	for (i32s n1 = 0;n1 < p1->GetSetup()->GetAtomCount();n1++)
	{
		fGL x = p1->crd[n1 * 3 + 0];
		fGL y = p1->crd[n1 * 3 + 1];
		fGL z = p1->crd[n1 * 3 + 2];
		
		glob_atmtab[n1]->SetCRD(p3, x, y, z);
	}
	
	// the rest is SF-related...
	// the rest is SF-related...
	// the rest is SF-related...
	
	eng1_sf * esf = dynamic_cast<eng1_sf *>(p1);
	setup1_sf * ssf = dynamic_cast<setup1_sf *>(p1->GetSetup());
	if (esf != NULL && ssf != NULL)
	{
		i32s bt3_counter = 0;
		
		for (i32u cc = 0;cc < ssf->chn_vector.size();cc++)
		{
			for (i32s rc = 1;rc < ((i32s) ssf->chn_vector[cc].res_vector.size()) - 2;rc++)
			{
				const fGL * prev = ssf->chn_vector[cc].res_vector[rc - 1].atmr[0]->GetCRD(p3);
				const fGL * curr = ssf->chn_vector[cc].res_vector[rc + 0].atmr[0]->GetCRD(p3);
				const fGL * next = ssf->chn_vector[cc].res_vector[rc + 1].atmr[0]->GetCRD(p3);

				v3d<fGL> v1(curr, prev); v3d<fGL> v2(curr, next);
				v3d<fGL> v3 = v1.vpr(v2); v3 = v3 / v3.len();		// this is vector c in the JCC 2001 paper.
				v3d<fGL> v4 = v2.vpr(v3); v4 = v4 / v4.len();		// this is vector n in the JCC 2001 paper.
				
				if (bt3_counter >= (i32s) esf->bt3_vector.size()) assertion_failed(__FILE__, __LINE__, "bt3_counter overflow");
				
				fGL pbdd = esf->bt3_vector[bt3_counter++].pbdd;
				v3d<fGL> v5 = (v4 * cos(pbdd)) - (v3 * sin(pbdd));
				
				v2 = v2 / v2.len(); const fGL scale = 0.3785;
				v3d<fGL> vC = (v2 * (+0.384 * scale)) + (v5 * (-0.116 * scale));
				v3d<fGL> vO = (v2 * (+0.442 * scale)) + (v5 * (-0.449 * scale));
				v3d<fGL> vN = (v2 * (+0.638 * scale)) + (v5 * (+0.109 * scale));
				
				atom * ref_to_C = ssf->chn_vector[cc].res_vector[rc + 0].peptide[2];
				atom * ref_to_O = ssf->chn_vector[cc].res_vector[rc + 0].peptide[3];
				atom * ref_to_N = ssf->chn_vector[cc].res_vector[rc + 1].peptide[0];
				
				fGL x; fGL y; fGL z;
				
				x = curr[0] + vC[0]; y = curr[1] + vC[1]; z = curr[2] + vC[2]; ref_to_C->SetCRD(p3, x, y, z);
				x = curr[0] + vO[0]; y = curr[1] + vO[1]; z = curr[2] + vO[2]; ref_to_O->SetCRD(p3, x, y, z);
				x = curr[0] + vN[0]; y = curr[1] + vN[1]; z = curr[2] + vN[2]; ref_to_N->SetCRD(p3, x, y, z);
			}
		}
	}
	
	// ready!!!
	// ready!!!
	// ready!!!
}

/*################################################################################################*/

engine_bp::engine_bp(setup * p1, i32u p2) : engine(p1, p2)
{
	use_bp = GetSetup()->GetModel()->use_boundary_potential;
	bp_rad_solute = GetSetup()->GetModel()->saved_boundary_potential_rad_solute;
	bp_rad_solvent = GetSetup()->GetModel()->saved_boundary_potential_rad_solvent;
	
	nd_eval = NULL;
	rdf_eval = NULL;
}

engine_bp::~engine_bp(void)
{
	if (nd_eval != NULL) delete nd_eval;
	if (rdf_eval != NULL) delete rdf_eval;
}

/*################################################################################################*/

engine_pbc::engine_pbc(setup * p1, i32u p2) : engine(p1, p2)
{
	f64 * tmp = GetSetup()->GetModel()->saved_periodic_box_HALFdim;
	
	box_HALFdim[0] = tmp[0];
	box_HALFdim[1] = tmp[1];
	box_HALFdim[2] = tmp[2];
	
	tmp = NULL;
	
	num_mol = 0;
	
// count the molecules present in the full atom set ; since the "mol"
// level is the highest criteria in sorting, atoms in a molecule should
// be adjacent -> a continuous range of pointers.
	
	if (!GetSetup()->GetModel()->IsGroupsSorted()) assertion_failed(__FILE__, __LINE__, "not_sorted");
	
// here we calculate the molecule locations precisely, but a simple trigger atom could be used as well...
	
	i32s previous = -123;	// what is the safest setting here??? NOT_DEFINED might be used there???
	
	atom ** atmtab = GetSetup()->GetAtoms();
	for (i32s index = 0;index < GetSetup()->GetAtomCount();index++)
	{
		if (atmtab[index]->id[0] != previous)
		{
			num_mol++;
			previous = atmtab[index]->id[0];
		}
	}
	
	mrange = new i32s[num_mol + 1];
	
	mrange[0] = 0; i32s a_index = 0;	// a_index counts LOCAL atom indices.
	for (i32s n1 = 0;n1 < num_mol;n1++)
	{
		i32s m_index = atmtab[a_index]->id[0];
		
		// m_index counts atom::id[0] molecule numbers.
		// m_index MAY APPEAR DISCONTINUOUS IF eng1_sf!
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		while (a_index < GetSetup()->GetAtomCount() && atmtab[a_index]->id[0] == m_index) a_index++;
		mrange[n1 + 1] = a_index;
	}
}

engine_pbc::~engine_pbc(void)
{
	delete[] mrange;
}

void engine_pbc::CheckLocations(void)
{
	atom ** atmtab = GetSetup()->GetAtoms();
	for (i32s n1 = 0;n1 < num_mol;n1++)
	{
		f64 sum[3] = { 0.0, 0.0, 0.0 };
		f64 ac = (f64) (mrange[n1 + 1] - mrange[n1]);
		for (i32s n2 = mrange[n1];n2 < mrange[n1 + 1];n2++)
		{
			i32u index = atmtab[n2]->varind;
			for (i32s n3 = 0;n3 < 3;n3++)
			{
				sum[n3] += crd[index * 3 + n3];
			}
		}
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 test = sum[n2] / ac;
			
			if (test < -box_HALFdim[n2])
			{
				for (i32s n3 = mrange[n1];n3 < mrange[n1 + 1];n3++)
				{
					i32u index = atmtab[n3]->varind;
					crd[index * 3 + n2] += 2.0 * box_HALFdim[n2];
				}
			}
			else if (test > +box_HALFdim[n2])
			{
				for (i32s n3 = mrange[n1];n3 < mrange[n1 + 1];n3++)
				{
					i32u index = atmtab[n3]->varind;
					crd[index * 3 + n2] -= 2.0 * box_HALFdim[n2];
				}
			}
		}
	}
}

// TODO :
// GetVDWSurf() for engine_pbc???
// GetESP() for engine_pbc???

/*################################################################################################*/

number_density_evaluator::number_density_evaluator(engine_bp * p1, bool p2, i32s p3)
{
	eng = p1;
	linear = p2;
	classes = p3;
	
	if (!eng->use_bp) assertion_failed(__FILE__, __LINE__, "use_bp is false");
	
	upper_limits = new f64[classes];
	class_volumes = new f64[classes];
	UpdateClassLimits();
	
	counts = new i32u[classes + 1];
	ResetCounters();
}

number_density_evaluator::~number_density_evaluator(void)
{
	delete[] upper_limits;
	delete[] class_volumes;
	
	delete[] counts;
}

void number_density_evaluator::UpdateClassLimits(void)
{
	if (linear)
	{
		f64 prev_radius = 0.0;
		for (i32s n1 = 0;n1 < classes;n1++)
		{
			f64 next_radius = eng->bp_rad_solvent * ((f64) (n1 + 1)) / (f64) classes;
			upper_limits[n1] = next_radius;
			
			f64 volume1 = 4.0 * M_PI * prev_radius * prev_radius * prev_radius / 3.0;
			f64 volume2 = 4.0 * M_PI * next_radius * next_radius * next_radius / 3.0;
			
			class_volumes[n1] = volume2 - volume1;
			
			prev_radius = next_radius;	// this is the last operation...
		}
	}
	else
	{
		f64 volume1 = 4.0 * M_PI * eng->bp_rad_solvent * eng->bp_rad_solvent * eng->bp_rad_solvent / 3.0;
		f64 volume2 = volume1 / (f64) classes;
		
		f64 prev_radius = 0.0;
		for (i32s n1 = 0;n1 < classes;n1++)
		{
			f64 tmp1 = volume2 + 4.0 * M_PI * prev_radius * prev_radius * prev_radius / 3.0;
			f64 tmp2 = tmp1 / (4.0 * M_PI / 3.0);
			
			f64 next_radius = pow(tmp2, 1.0 / 3.0);
			
			upper_limits[n1] = next_radius;
			class_volumes[n1] = volume2;
			
			prev_radius = next_radius;	// this is the last operation...
		}
	}
}

void number_density_evaluator::ResetCounters(void)
{
	cycles = 0;
	for (i32s n1 = 0;n1 < classes + 1;n1++)
	{
		counts[n1] = 0;
	}
}

void number_density_evaluator::PrintResults(ostream & str)
{
	str << "ND : ";
	for (i32s n1 = 0;n1 < classes;n1++)
	{
		f64 tmp1 = ((f64) counts[n1]) / ((f64) cycles);
		f64 tmp2 = tmp1 / class_volumes[n1];
		
		str << tmp2 << " ";
	}
	
	f64 tmp1 = ((f64) counts[classes]) / ((f64) cycles);
	str << _("(outside bp_radius = ") << tmp1 << ")." << endl;
	
	ResetCounters();
}

/*################################################################################################*/

radial_density_function_evaluator::radial_density_function_evaluator(engine_bp * p1, i32s p2, f64 gb, f64 ge, f64 cb, f64 ce)
{
	eng = p1;
	classes = p2;
	
	graph_begin = gb;
	graph_end = ge;
	
	count_begin = gb;
	count_end = ge;
	
	if (count_begin < 0.0)
	{
		if (!eng->use_bp) assertion_failed(__FILE__, __LINE__, "use_bp is false");
		
		if (!eng->nd_eval) assertion_failed(__FILE__, __LINE__, "nd_eval is NULL");
		
		f64 graph_width = graph_end - graph_begin;
		f64 count_width = count_end - count_begin;
		if (count_width < graph_width) assertion_failed(__FILE__, __LINE__, "bad width");
	}
	
	upper_limits = new f64[classes];
	class_volumes = new f64[classes];
	
	f64 prev_radius = graph_begin;
	for (i32s n1 = 0;n1 < classes;n1++)
	{
		f64 next_radius = graph_begin + (graph_end - graph_begin) * ((f64) (n1 + 1)) / (f64) classes;
		upper_limits[n1] = next_radius;
		
		f64 volume1 = 4.0 * M_PI * prev_radius * prev_radius * prev_radius / 3.0;
		f64 volume2 = 4.0 * M_PI * next_radius * next_radius * next_radius / 3.0;
		
		class_volumes[n1] = volume2 - volume1;
		
		prev_radius = next_radius;	// this is the last operation...
	}
	
	counts = new i32u[classes];
	ResetCounters();
}

radial_density_function_evaluator::~radial_density_function_evaluator(void)
{
	delete[] upper_limits;
	delete[] class_volumes;
	
	delete[] counts;
}

void radial_density_function_evaluator::ResetCounters(void)
{
	cycles = 0;
	for (i32s n1 = 0;n1 < classes;n1++)
	{
		counts[n1] = 0;
	}
}

void radial_density_function_evaluator::PrintResults(ostream & str)
{
	str << "RDF : ";
	for (i32s n1 = 0;n1 < classes;n1++)
	{
		f64 tmp1 = ((f64) counts[n1]) / ((f64) cycles);
		f64 tmp2 = tmp1 / class_volumes[n1];
		
		str << tmp2 << " ";
	}	str << endl;
	
	ResetCounters();
}

/*################################################################################################*/

// eof
