/* $Id: conf.c,v 1.11 2004/09/10 10:05:18 pekberg Exp $
******************************************************************************
  
   LibGG - Configuration handling
  
   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]
   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]
   
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include "plat.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <ctype.h>

#include <ggi/gg.h>

/* Private structures */
typedef struct gg_string_list {
	const char *str;
	struct gg_string_list *next;
} gg_string_list;

typedef struct gg_dlconfig {
	const char *name;
	const char *filename;
	gg_string_list *version;
	struct gg_dlconfig *next;
} gg_dlconfig;

static int
_ggDoLoad(const char *file, gg_dlconfig **list, int incl, char *oldroot)
{
	FILE *inf;
	char root[2048], line[2048], buff[2048], *cp, *bp;
	char *name, *fname;
	const char *fp;
#if CHAR_DIRDELIM != '/'
	const char *fp2;
#endif
	char tmp;
	size_t fname_len;
	gg_dlconfig *dltmp, *dlconf;
	root[0] = '\0';

	inf = fopen(file, "r");
	if (inf == NULL) return GGI_ENOFILE;

	dlconf = *list;

	if (oldroot) {
		ggstrlcpy(root, oldroot, sizeof(root));
	}

	while (fgets(line, 2048, inf)) {
		cp = line;

		while (isspace((int)*cp)) cp++;
		if (*cp == '\0' || *cp == '#' || *cp == '\n')
			continue;

		if (strncmp(cp, ".root:", 6) == 0) {
			cp += 6;
			while (isspace((int)*cp)) cp++;
			name = cp;
			while (*cp && !isspace((int)*cp)) cp++;
			/* The .root '*' is special and denotes an alias
			 * expansion section.
			 */
			if (*name == '/' || *name == '*')
				ggstrlcpy(root, name, (size_t)(cp - name + 1));
			else {
				fp = strrchr(file, '/');
				if (fp == NULL)  fp  = file;
#if CHAR_DIRDELIM != '/'
				fp2 = strrchr(file, CHAR_DIRDELIM);
				if (fp2 == NULL) fp2 = file;
				if (fp < fp2)
					fp = fp2;
#endif
				if (fp != file)
					++fp;
				ggstrlcpy(root, file, (size_t)(fp - file + 1));
				tmp = *cp; *cp = '\0';
				ggstrlcat(root, name, sizeof(root));
				*cp = tmp;
			}
			continue;
		}
		if (strncmp(cp, ".include", 8) == 0) {
			cp += 8;
			while (isspace((int)*cp)) cp++;
			name = cp;
			while (*cp && !isspace((int)*cp)) cp++;
			if (*name == '/')
				ggstrlcpy(buff, name, (size_t)(cp - name + 1));
			else {
				fp = strrchr(file, '/');
				if (fp == NULL)  fp  = file;
#if CHAR_DIRDELIM != '/'
				fp2 = strrchr(file, CHAR_DIRDELIM);
				if (fp2 == NULL) fp2 = file;
				if (fp < fp2)
					fp = fp2;
#endif
				if (fp != file)
					++fp;
				ggstrlcpy(buff, file, (size_t)(fp - file + 1));
				tmp = *cp; *cp = '\0';
				ggstrlcat(buff, name, sizeof(buff));
				*cp = tmp;
			}
			if (incl+1 < 20) {
				*list = dlconf;
				_ggDoLoad(buff, list, incl+1,
					  *root != '\0' ? root : NULL);
				dlconf = *list;
			}
			else {
				fprintf(stderr,"libgg: .include recursion "
						"too deep\n");
			}
			continue;
		}	
		bp = buff;
		while (*cp && !isspace((int)*cp)) *(bp++) = *(cp++);
		*(bp) = '\0';
		if ((name = strdup(buff)) == NULL) {
			return GGI_ENOMEM;
		}
		while (isspace((int)*cp)) cp++;
		if (*cp == '\0' || *cp == '#' || *cp == '\n') {
			free(name);
			continue;
		}
		bp=buff;
		while (*cp && !isspace((int)*cp)) *(bp++) =* (cp++);
		*(bp) = '\0';
		if (buff[0] != CHAR_DIRDELIM) {
			fname_len = strlen(buff) + strlen(root)
				+ strlen(STRING_DIRDELIM) + 1;
			fname = (char *) malloc(fname_len);
			if (fname == NULL) {
				return GGI_ENOMEM;
			}
			ggstrlcpy(fname, root, fname_len);
			ggstrlcat(fname,STRING_DIRDELIM, fname_len);
			ggstrlcat(fname,buff, fname_len);
		} else {
			if ((fname = strdup(buff)) == NULL) {
				return GGI_ENOMEM;
			}
		}
		/* Add the config entry....
		 */	
		if ((dltmp = (gg_dlconfig *)malloc(sizeof(gg_dlconfig)))
		    == NULL) {
			return GGI_ENOMEM;
		}
		dltmp->name = name;
		dltmp->filename = fname;
		dltmp->version = NULL;
		dltmp->next = dlconf;
		dlconf = dltmp;

		do {	
			gg_string_list *vtmp;

			while (isspace((int)*cp)) cp++;
			if (*cp == '\0' || *cp == '#') 
				break;
			bp = buff;
			while (*cp && !isspace((int)*cp)) *(bp++) = *(cp++);
			*(bp) = '\0';
			if ((vtmp = (gg_string_list *)malloc(sizeof(gg_string_list)))
			    == NULL 
			    || (vtmp->str = strdup(buff)) == NULL) {
				return GGI_ENOMEM;
			}
			vtmp->next = dltmp->version;
			dltmp->version = vtmp;
		} while (*cp != '\0');
			
	}
	fclose(inf);

	*list = dlconf;

	return 0;
}

int ggLoadConfig(const char *file, void **ret)
{
	*ret = NULL;
	return _ggDoLoad(file, (gg_dlconfig **)ret, 0, NULL);
}


/*
 * ggFreeConfig()
 *
 * Free the entire config list
 */

void ggFreeConfig(void *conf)
{
	gg_dlconfig *conftmp, *confptr = (gg_dlconfig*) conf;
	
	while (confptr != NULL) {
		gg_string_list *strlist = confptr->version, *strtmp;
		free((char*)confptr->name);
		free((char*)confptr->filename);
		while (strlist != NULL) {
			strtmp = strlist->next;
			free((char*)strlist->str);
			free(strlist);
			strlist = strtmp;
		}
		conftmp = confptr->next;
		free(confptr);
		confptr = conftmp;
	}
}


/* Takes a gg_dlconfig, lib name, and a version. Returns the
 * closest match (Ha!)
 */
const char *ggMatchConfig(const void *conf, const char *name,const char *ver)
{
	gg_dlconfig *dlc,*dtmp,*best = NULL;
	gg_string_list *vtmp;

	dlc = (gg_dlconfig*) conf;

	for (dtmp = dlc; dtmp != NULL; dtmp = dtmp->next) {
#ifdef HAVE_STRCASECMP
		if (strcasecmp(name, dtmp->name) != 0) {
#else
		if (strcmp(name, dtmp->name) != 0) {
#endif
			continue;
		}

		if (dtmp->version == NULL) {
			best = dtmp;
			continue;
		}

		for (vtmp = dtmp->version; vtmp != NULL; vtmp = vtmp->next) 
			if (strcmp(ver, vtmp->str) == 0) {
				return dtmp->filename;
			}
	}

	return (best == NULL) ? NULL : best->filename;
}

int ggConfigExpandAlias(void *confhandle, const char *list_in, 
	char *list_out, size_t outmax) {
	
	char *current,*newcur;
	const char *match;
	char buffer[1024];
	int loopcnt=0;
	int matchlen,outremain;

	if (outmax>0) *list_out='\0';
	if (strlen(list_in)+1>outmax) return GGI_ENOMEM;
	strcpy(list_out,list_in);
	outremain=outmax-strlen(list_out)-1;

	current=list_out;
	while(*current) {
		if (*current==':') {current++;continue;}
		newcur=ggParseTarget(current,buffer,sizeof(buffer));
		if (NULL==newcur) return GGI_EARGINVAL;
		match=ggMatchConfig(confhandle, buffer, NULL);
		if (match && match[0]=='*') {
			if (loopcnt++>100) return GGI_EARGINVAL; /* FIXME ELOOP */
			matchlen = strlen(match + 2);
			if ((outremain + newcur - current - matchlen) > 0) {
				 memmove(current + matchlen,
					newcur,strlen(newcur)+1);
				 memcpy(current, match+2,
					(unsigned)matchlen);
				 outremain += newcur - current - matchlen - 1;
			} else {
				return GGI_ENOMEM;
			}
		} else {
			current = newcur;
		}
	}
	return 0;
}
