#include <stdlib.h>
#include <string.h>
#include <signal.h>

#include "gripgesturemanager.h"

static gint    rotate = 0;
static gdouble scale = 1.0;
static gdouble translate[2] = { 200, 200 };
static gint    in_touch = 0;

static gboolean
expose_event (GtkWidget      *widget,
              GdkEventExpose *event,
              gpointer        data)
{
  cairo_t *cr;
  gdouble radians;
  gint width = (in_touch > 0) ? 10 : 1;

  cr = gdk_cairo_create (widget->window);

  cairo_set_source_rgb (cr, 0, 0, 0);
  cairo_set_line_width (cr, width);

  radians = rotate * (G_PI / 180);
  cairo_translate (cr, translate[0], translate[1]);
  cairo_scale (cr, scale, scale);
  cairo_rotate (cr, radians);

  cairo_rectangle (cr, -50, -50, 100, 100);
  cairo_stroke_preserve (cr);
  cairo_set_source_rgb (cr, 1, 0, 1);
  cairo_fill (cr);

  cairo_destroy (cr);

  return FALSE;
}

static void
gesture_callback (GtkWidget        *widget,
                  GripTimeType      time_type,
                  GripGestureEvent  *event,
                  gpointer           user_data)
{
  if (time_type == GRIP_TIME_START)
    {
      in_touch++;
    }
  else if (time_type == GRIP_TIME_END)
    {
      in_touch--;
    }
  else
    {
      switch (event->type)
        {
        case GRIP_GESTURE_DRAG:
          {
            GripEventGestureDrag *e = (GripEventGestureDrag *)event;

            translate[0] += e->delta_x;
            translate[1] += e->delta_y;
          }
          break;

        case GRIP_GESTURE_PINCH:
          {
            GripEventGesturePinch *e = (GripEventGesturePinch *)event;

            scale += e->radius_delta / 100;
          }
          break;

        case GRIP_GESTURE_ROTATE:
          {
            GripEventGestureRotate *e = (GripEventGestureRotate *)event;

            rotate += e->angle_delta * 100;
          }
          break;

        case GRIP_GESTURE_TAP:
          break;
        }
    }

  gtk_widget_queue_draw (GTK_WIDGET (widget));
}


static GtkTreeModel *
create_model ()
{
  return (GtkTreeModel *)gtk_list_store_new (1, G_TYPE_STRING);
}

static GtkWidget *
create_window (GripGestureManager *manager)
{
  GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  GtkWidget *da;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkWidget *tv;
  GtkTreeModel *model;
  GtkWidget *sw;
  const GdkColor white = { 0, 0xffff, 0xffff, 0xffff };

  da = gtk_drawing_area_new ();

  gtk_window_set_title (GTK_WINDOW (window), "Touch Demo");
  gtk_window_set_default_size (GTK_WINDOW (window), 600, 600);
  g_signal_connect (window, "destroy", G_CALLBACK (gtk_widget_destroyed), &window);
  g_signal_connect (window, "destroy", G_CALLBACK (gtk_main_quit), NULL);

  hbox = gtk_hbox_new (FALSE, 0);
  vbox = gtk_vbox_new (FALSE, 0);

  gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), da, TRUE, TRUE, 0);

  model = create_model ();
  sw = gtk_scrolled_window_new (NULL, NULL);
  tv = gtk_tree_view_new_with_model (model);
  g_object_unref (model);

  gtk_container_add (GTK_CONTAINER (sw), tv);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);

  button = gtk_button_new_with_label ("Quit");
  g_signal_connect (button,
                    "clicked",
                    G_CALLBACK (gtk_main_quit),
                    NULL);

  gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);

  gtk_container_add (GTK_CONTAINER (window), hbox);

  gtk_widget_modify_bg (da, GTK_STATE_NORMAL, &white);

  g_signal_connect (da, "expose-event",
                    G_CALLBACK (expose_event), NULL);

  grip_gesture_manager_register_window (manager,
                                        da,
                                        GRIP_GESTURE_PINCH,
                                        2,
                                        gesture_callback,
                                        NULL, NULL);

  grip_gesture_manager_register_window (manager,
                                        da,
                                        GRIP_GESTURE_ROTATE,
                                        2,
                                        gesture_callback,
                                        NULL, NULL);

  grip_gesture_manager_register_window (manager,
                                        da,
                                        GRIP_GESTURE_DRAG,
                                        2,
                                        gesture_callback,
                                        NULL, NULL);

  grip_gesture_manager_register_window (manager,
                                        da,
                                        GRIP_GESTURE_TAP,
                                        2,
                                        gesture_callback,
                                        NULL, NULL);

  return window;
}

static void
abort_handler (int x)
{
  g_print (" **** ABORT ****\n");

  exit (1);
}

int
main (int argc, char **argv)
{
  GtkWidget *window;
  GripGestureManager *manager;

  gtk_init (&argc, &argv);

  /* Don't crash X if we're using some shitty Intel graphics like
   * my Dell XT2 has in it. */
  signal (SIGABRT, abort_handler);

  manager = grip_gesture_manager_get ();
  window = create_window (manager);

  g_signal_connect (window, "destroy", G_CALLBACK (gtk_main_quit), NULL);

  gtk_widget_show_all (GTK_WIDGET (window));

  gtk_main ();

  return 0;
}

