/*
 * Copyright 2010 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authors:
 *    Cody Russell <crussell@canonical.com>
 */

#ifndef __GRIP_GESTURE_MANAGER_H__
#define __GRIP_GESTURE_MANAGER_H__

#include <gtk/gtk.h>

G_BEGIN_DECLS

#define GRIP_TYPE_GESTURE_MANAGER         (grip_gesture_manager_get_type ())
#define GRIP_GESTURE_MANAGER(o)           (G_TYPE_CHECK_INSTANCE_CAST ((o), GRIP_TYPE_GESTURE_MANAGER, GripGestureManager))
#define GRIP_GESTURE_MANAGER_CLASS(c)     (G_TYPE_CHECK_CLASS_CAST ((c), GRIP_TYPE_GESTURE_MANAGER, GripGestureManagerClass))
#define GRIP_IS_GESTURE_MANAGER(o)        (G_TYPE_CHECK_INSTANCE_TYPE ((o), GRIP_TYPE_GESTURE_MANAGER))
#define GRIP_IS_GESTURE_MANAGER_CLASS(c)  (G_TYPE_CHECK_CLASS_TYPE ((c), GRIP_TYPE_GESTURE_MANAGER))
#define GRIP_GESTURE_MANAGER_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), GRIP_TYPE_GESTURE_MANAGER, GripGestureManagerClass))

typedef struct _GripGestureManager        GripGestureManager;
typedef struct _GripGestureManagerClass   GripGestureManagerClass;
typedef struct _GripGestureManagerPrivate GripGestureManagerPrivate;

typedef union  _GripGestureEvent       GripGestureEvent;
typedef struct _GripEventGestureAny    GripEventGestureAny;
typedef struct _GripEventGestureDrag   GripEventGestureDrag;
typedef struct _GripEventGesturePinch  GripEventGesturePinch;
typedef struct _GripEventGestureRotate GripEventGestureRotate;
typedef struct _GripEventGestureTap    GripEventGestureTap;

typedef enum {
  GRIP_GESTURE_DRAG    = 0,
  GRIP_GESTURE_PINCH   = 1,
  GRIP_GESTURE_ROTATE  = 2,
  GRIP_GESTURE_TAP     = 15
} GripGestureType;

typedef enum {
  GRIP_TIME_START,
  GRIP_TIME_UPDATE,
  GRIP_TIME_END
} GripTimeType;

struct _GripEventGestureAny
{
  GdkEventType  type;
  GdkWindow    *window;
};

struct _GripEventGestureDrag
{
  GripGestureType   type;
  guint             id;
  GdkWindow        *window;
  GdkWindow        *root;
  GdkWindow        *child;
  guint32           timestamp;
  gint              fingers;
  gdouble           focus_x;
  gdouble           focus_y;
  gint              delta_x;
  gint              delta_y;
  gdouble           velocity_x;
  gdouble           velocity_y;
  gdouble           position_x;
  gdouble           position_y;
};

struct _GripEventGesturePinch
{
  GripGestureType   type;
  guint             id;
  GdkWindow        *window;
  GdkWindow        *root;
  GdkWindow        *child;
  guint32           timestamp;
  guint             fingers;
  gdouble           focus_x;
  gdouble           focus_y;
  gdouble           radius_delta;
  gdouble           radial_velocity;
  gdouble           radius;
};

struct _GripEventGestureRotate
{
  GripGestureType  type;
  guint             id;
  GdkWindow        *window;
  GdkWindow        *root;
  GdkWindow        *child;
  guint32           timestamp;
  guint             fingers;
  gdouble           focus_x;
  gdouble           focus_y;
  gdouble           angle_delta;
  gdouble           angular_velocity;
  gdouble           angle;
};

struct _GripEventGestureTap
{
  GripGestureType  type;
  guint             id;
  GdkWindow        *window;
  GdkWindow        *root;
  GdkWindow        *child;
  guint32           timestamp;
  guint             fingers;

  guint32           tap_time;
  gfloat            focus_x;
  gfloat            focus_y;
  gfloat            position_x;
  gfloat            position_y;
};

union _GripGestureEvent
{
  GripGestureType        type;
  GripEventGestureAny    any;
  GripEventGestureDrag   drag;
  GripEventGesturePinch  pinch;
  GripEventGestureRotate rotate;
  GripEventGestureTap    tap;
};

/**
   GripGestureManager
   @parent: #GObject.

   A singleton manager into which a window may be registered to receive
   multitouch gesture events.
 */
struct _GripGestureManager
{
  GObject parent_instance;

  /*< Private >*/
  GripGestureManagerPrivate *priv;
};

/**
   GripGestureManagerClass
   @parent_class: #GObjectClass
*/
struct _GripGestureManagerClass
{
  GObjectClass parent_class;
};

/**
 * GripGestureCallback:
 * @widget: (in): A #GtkWidget pointer
 * @time: A #GripTimeType
 * @gesture: (in): A #GripGestureEvent pointer
 * @user_data: (transfer none): user data
 **/
typedef void (* GripGestureCallback) (GtkWidget         *widget,
                                      GripTimeType       time,
                                      GripGestureEvent  *gesture,
                                      gpointer            user_data);

GType                grip_gesture_manager_get_type        (void) G_GNUC_CONST;

/**
 * grip_gesture_manager_get:
 *
 * Retrieves a #GripGestureManager pointer.
 *
 * Return value: (transfer none): A #GripGestureManager
 **/
GripGestureManager *grip_gesture_manager_get              (void);

/**
 * grip_gesture_manager_register_window:
 * @manager: self
 * @widget: A #GtkWidget on which to register the gesture
 * @gesture_type: Gesture type
 * @touch_points: Number of touch points
 * @callback: Callback
 * @user_data: (transfer none): User data
 * @destroy: (transfer none) (scope async): Destroy
 */
void                 grip_gesture_manager_register_window (GripGestureManager *manager,
                                                           GtkWidget          *widget,
                                                           GripGestureType     gesture_type,
                                                           gint                touch_points,
                                                           GripGestureCallback callback,
                                                           gpointer            user_data,
                                                           GDestroyNotify      destroy);

GType             grip_gesture_event_get_type (void) G_GNUC_CONST;
GripGestureEvent *grip_gesture_event_new      (GripGestureType   gesture_type);
void              grip_gesture_event_free     (GripGestureEvent *event);
GripGestureEvent *grip_gesture_event_copy     (const GripGestureEvent *event);


G_END_DECLS

#endif /* __GRIP_GESTURE_MANAGER_H__ */
