/* -*- C++ -*- */

// The Hoard Multiprocessor Memory Allocator
// www.hoard.org
//
// Author: Emery Berger, http://www.cs.umass.edu/~emery
//
// Copyright (c) 1998-2003, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
 * @file   libhoard.cpp
 * @brief  This file replaces malloc etc. in your application.
 * @author Emery Berger <http://www.cs.umass.edu/~emery>
 */

#include "processheap.h"
#include "wrapper.cpp"

#if defined(_WIN32)
#pragma warning(disable:4273)
#endif

int tidmap[MAX_THREADS];
int inusemap[hoardHeap::MAX_HEAPS];

static void initializeMaps (void) {
  int i;
  for (i = 0; i < MAX_THREADS; i++) {
    tidmap[i] = 0;
  }
  for (i = 0; i < hoardHeap::MAX_HEAPS; i++) {
    inusemap[i] = 0;
  }
}

static void findUnusedHeap (void) {
  // Find an unused heap.
  int i = 0;
  while ((i < hoardHeap::MAX_HEAPS) && (inusemap[i]))
    i++;
  if (i >= hoardHeap::MAX_HEAPS) {
    // Every heap is in use: pick a random victim.
    i = (int) ( hoardHeap::MAX_HEAPS * ((double) rand() / (double) RAND_MAX));
  }
  inusemap[i]++;
  
  // NOTE: This works for Solaris. What about Linux?
  
  int tid = (int) hoardGetThreadID();
  tid = (int) tid % MAX_THREADS;
  tidmap[tid] = i;
}

static void releaseHeap (void) {
  // Decrement the ref-count on the current heap.
  // When we hit zero, clear the appropriate heap.
  
  int tid = hoardGetThreadID() % MAX_THREADS;
  int i = tidmap[tid];
  
  inusemap[i]--;
  if (inusemap[i] == 0) {
    // WE SHOULD DO THIS ONLY WHEN i > NCPUS.
    //      getCustomHeap()->getMallocHeap().getPerThreadHeap(i)->clear();
  }
}

#if !defined(_WIN32)

// NOTE: Relies on libpthread being a shared library.

#include <pthread.h>
#include <dlfcn.h>

static void * handle;

template <class T1, class T2>
class MyPair {
public:
  T1 first;
  T2 second;
};

class InitializeMe {
public:
  InitializeMe (void) {
    handle = dlopen ("libpthread.so", RTLD_NOW);
    initializeMaps();
  }
};

extern "C" {

  typedef void * (*threadFunctionType) (void *);

  typedef  
  int (*pthread_create_function) (pthread_t *thread,
				  const  pthread_attr_t *attr,
				  threadFunctionType start_routine,
				  void *arg);
  
  typedef
  void (*pthread_exit_function) (void *);

  static InitializeMe& startRoutine (void) {
    static InitializeMe init;
    return init;
  }

  static void exitRoutine (void) {
#if defined(linux)
    static pthread_exit_function * f =
      (pthread_exit_function *) dlsym (handle, "pthread_exit");
#else
    static pthread_exit_function * f =
      (pthread_exit_function *) dlsym (handle, "_pthread_exit");
#endif
    releaseHeap();
  }


  void * startMeUp (void * a)
  {
    //    printf ("WOWOOW\n");

    findUnusedHeap();

    //    printf ("acquired %d, tid = %d\n", i, tid);
    
    MyPair<threadFunctionType, void *> * z
      = (MyPair<threadFunctionType, void *> *) a;
    threadFunctionType f = z->first;
    void * arg = z->second;
    delete z;
    void * result = (*f)(arg);
    exitRoutine();
    return result;
  }

  int pthread_create (pthread_t *thread,
		      const  pthread_attr_t *attr,
		      threadFunctionType start_routine,
		      void * arg)
  {
    static InitializeMe init = startRoutine();
#if defined(linux)
    static pthread_create_function f
      = (pthread_create_function) dlsym (handle, "pthread_create");
#else
    static pthread_create_function f
      = (pthread_create_function) dlsym (handle, "_pthread_create");
#endif

    MyPair<threadFunctionType, void *> * newarg
      = new MyPair<threadFunctionType, void *>();

    newarg->first  = start_routine;
    newarg->second = arg;

    int result = (*f)(thread, attr, startMeUp, newarg);
    
    return result;
  }

  void pthread_exit (void * arg)
  {
#if defined(linux)
    static pthread_exit_function f
      = (pthread_exit_function) dlsym (handle, "pthread_exit");
#else
    static pthread_exit_function f
      = (pthread_exit_function) dlsym (handle, "_pthread_exit");
#endif
    exitRoutine();
    (*f)(arg);
  }
}

#elif defined(_WIN32)

#include <stdio.h>

BOOL WINAPI DllMain(HANDLE hinstDLL, DWORD fdwReason, LPVOID lpreserved)
{
  int i;
  int tid;
  int np;
  switch (fdwReason) {
  case DLL_PROCESS_ATTACH:
    np = hoardGetNumProcessors();
    initializeMaps();
    break;
  case DLL_THREAD_ATTACH:
    if (np == 1) {
      tidmap[(int) hoardGetThreadID() % MAX_THREADS] = 0;
    } else {
      findUnusedHeap();
    }
    break;
  case DLL_THREAD_DETACH:
    if (np != 1) {
      releaseHeap();
    }
    break;
  default:
    return TRUE;
  }
  return TRUE;
}

#endif
