/* iksemel (XML parser for Jabber)
** Copyright (C) 2000-2003 Gurer Ozen <madcat@e-kolay.net>
** This code is free software; you can redistribute it and/or
** modify it under the terms of GNU Lesser General Public License.
*/

#include "common.h"
#include "iksemel.h"

#include <fcntl.h>
#include <netdb.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>

#ifdef HAVE_GNUTLS
#include <gnutls/gnutls.h>
#endif

#define SF_FOREIGN 1

struct stream_data {
	iksparser *prs;
	ikstack *s;
	char *name_space;
	void *user_data;
	iksStreamHook *streamHook;
	iksLogHook *logHook;
	iks *current;
	char *buf;
	int sock;
	unsigned int flags;
};

static void
insert_attribs (iks *x, char **atts)
{
	if (atts) {
		int i = 0;
		while (atts[i]) {
			iks_insert_attrib (x, atts[i], atts[i+1]);
			i += 2;
		}
	}
}

static int
tagHook (struct stream_data *data, char *name, char **atts, int type)
{
	iks *x;
	int err;

	switch (type) {
		case IKS_OPEN:
		case IKS_SINGLE:
			if (data->current) {
				x = iks_insert (data->current, name);
				insert_attribs (x, atts);
			} else {
				x = iks_new (name);
				insert_attribs (x, atts);
				if (iks_strcmp (name, "stream:stream") == 0) {
					err = data->streamHook (data->user_data, IKS_NODE_START, x);
					if (err != IKS_OK) return err;
					break;
				}
			}
			data->current = x;
			if (IKS_OPEN == type) break;
		case IKS_CLOSE:
			x = data->current;
			if (NULL == x) {
				err = data->streamHook (data->user_data, IKS_NODE_STOP, NULL);
				if (err != IKS_OK) return err;
				break;
			}
			if (NULL == iks_parent (x)) {
				if (iks_strcmp (name, "stream:error") == 0) {
					err = data->streamHook (data->user_data, IKS_NODE_ERROR, x);
					if (err != IKS_OK) return err;
				} else {
					err = data->streamHook (data->user_data, IKS_NODE_NORMAL, x);
					if (err != IKS_OK) return err;
				}
				data->current = NULL;
				break;
			}
			data->current = iks_parent (x);
	}
	return IKS_OK;
}

static int
cdataHook (struct stream_data *data, char *cdata, size_t len)
{
	if (data->current) iks_insert_cdata (data->current, cdata, len);
	return IKS_OK;
}

static void
deleteHook (struct stream_data *data)
{
	if ((data->flags & SF_FOREIGN) == 0 && data->sock >= 0) close (data->sock);
	data->sock = -1;
	if (data->current) iks_delete (data->current);
	data->current = NULL;
	data->flags = 0;
}

iksparser *
iks_stream_new (char *name_space, void *user_data, iksStreamHook *streamHook)
{
	ikstack *s;
	struct stream_data *data;

	s = iks_stack_new (DEFAULT_STREAM_CHUNK_SIZE);
	if (NULL == s) return NULL;
	data = iks_stack_alloc (s, sizeof (struct stream_data));
	memset (data, 0, sizeof (struct stream_data));
	data->s = s;
	data->prs = iks_sax_extend (s, data, (iksTagHook *)tagHook, (iksCDataHook *)cdataHook, (iksDeleteHook *)deleteHook);
	data->name_space = name_space;
	data->user_data = user_data;
	data->streamHook = streamHook;
	data->sock = -1;
	return data->prs;
}

void *
iks_stream_user_data (iksparser *prs)
{
	struct stream_data *data = iks_user_data (prs);

	return data->user_data;
}

void
iks_set_log_hook (iksparser *prs, iksLogHook *logHook)
{
	struct stream_data *data = iks_user_data (prs);

	data->logHook = logHook;
}

int
iks_has_tls (void)
{
#ifdef HAVE_GNUTLS
	return 1;
#else
	return 0;
#endif
}

int
iks_connect_tcp (iksparser *prs, const char *server, int port)
{
	return iks_connect_via (prs, server, port, server);
}

int
iks_connect_via (iksparser *prs, const char *server, int port, const char *server_name)
{
	struct stream_data *data = iks_user_data (prs);
#ifdef HAVE_GETADDRINFO
	struct addrinfo hints;
	struct addrinfo *addr_res, *addr_ptr;
	char port_str[6];
#else
	struct hostent *host;
	struct sockaddr_in sin;
#endif
	int tmp;

	if (!data->buf) {
		data->buf = iks_stack_alloc (data->s, NET_IO_BUF_SIZE);
		if (NULL == data->buf) return IKS_NOMEM;
	}
#ifdef HAVE_GETADDRINFO
	hints.ai_flags = AI_CANONNAME;
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = 0;
	hints.ai_addrlen = 0;
	hints.ai_canonname = NULL;
	hints.ai_addr = NULL;
	hints.ai_next = NULL;
	sprintf (port_str, "%i", port);
	if (getaddrinfo (server, port_str, &hints, &addr_res) != 0) {
#else
	host = gethostbyname (server);
	if (!host) {
#endif
		iks_disconnect (prs);
		return IKS_NET_NODNS;
	}
#ifdef HAVE_GETADDRINFO
	addr_ptr = addr_res;
	while (addr_ptr) {
		data->sock = socket (addr_ptr->ai_family, addr_ptr->ai_socktype, addr_ptr->ai_protocol);
		if (data->sock != -1) break;
		addr_ptr = addr_ptr->ai_next;
	}
#else
	memcpy (&sin.sin_addr, host->h_addr, host->h_length);
	sin.sin_family = host->h_addrtype;
	sin.sin_port = htons (port);
	data->sock = socket (host->h_addrtype, SOCK_STREAM, 0);
#endif
	if (data->sock < 0) {
		iks_disconnect (prs);
		return IKS_NET_NOSOCK;
	}
#ifdef HAVE_GETADDRINFO
	tmp = connect (data->sock, addr_ptr->ai_addr, addr_ptr->ai_addrlen);
	freeaddrinfo (addr_res);
#else
	tmp = connect (data->sock, (struct sockaddr *)&sin, sizeof (struct sockaddr_in));
#endif
	if (tmp != 0) {
		iks_disconnect (prs);
		return IKS_NET_NOCONN;
	}
	return iks_send_header (prs, server_name);
}

int
iks_connect_fd (iksparser *prs, int fd)
{
	struct stream_data *data = iks_user_data (prs);

	if (!data->buf) {
		data->buf = iks_stack_alloc (data->s, NET_IO_BUF_SIZE);
		if (NULL == data->buf) return IKS_NOMEM;
	}
	data->flags |= SF_FOREIGN;
	data->sock = fd;
	return IKS_OK;
}

int
iks_fd (iksparser *prs)
{
	struct stream_data *data = iks_user_data (prs);

	return data->sock;
}

int
iks_recv (iksparser *prs, int timeout)
{
	struct stream_data *data = iks_user_data (prs);
	fd_set fds;
	struct timeval tv, *tvptr;
	enum iksneterror ret;

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	FD_ZERO (&fds);
	FD_SET (data->sock, &fds);
	tv.tv_sec = timeout;
	if (timeout != -1) tvptr = &tv; else tvptr = NULL;
	while (select (data->sock + 1, &fds, NULL, NULL, tvptr) > 0) {
		int len;
		len = recv (data->sock, data->buf, NET_IO_BUF_SIZE - 1, 0);
		if (len > 0) {
			data->buf[len] = '\0';
			if (data->logHook) data->logHook (data->user_data, data->buf, len, 1);
			ret = iks_parse (prs, data->buf, len, 0);
			if (ret != IKS_OK) return ret;
		} else if (len <= 0) {
			return IKS_NET_RWERR;
		}
		FD_ZERO (&fds);
		FD_SET (data->sock, &fds);
		tv.tv_sec = 0;
	}
	return IKS_OK;
}

int
iks_send_header (iksparser *prs, const char *to)
{
	struct stream_data *data = iks_user_data (prs);
	ikstack *s;
	s = iks_stack_new (256);
	iks_stack_strcat (s, "<?xml version='1.0'?><stream:stream xmlns:stream='http://etherx.jabber.org/streams' xmlns='", 91);
	iks_stack_strcat (s, data->name_space, 0);
	iks_stack_strcat (s, "' to='", 6);
	iks_stack_strcat (s, to, 0);
	iks_stack_strcat (s, "'>", 2);
	if (IKS_OK != iks_send_raw (prs, iks_stack_print (s))) return IKS_NET_RWERR;
	iks_stack_delete (s);
	return IKS_OK;
}

int
iks_send (iksparser *prs, iks *x)
{
	return iks_send_raw (prs, iks_string (iks_stack (x), x));
}

int
iks_send_raw (iksparser *prs, const char *xmlstr)
{
	struct stream_data *data = iks_user_data (prs);

	if (send (data->sock, xmlstr, strlen (xmlstr), 0) == -1) return IKS_NET_RWERR;
	if (data->logHook) data->logHook (data->user_data, xmlstr, strlen (xmlstr), 0);
	return IKS_OK;
}

void
iks_disconnect (iksparser *prs)
{
	iks_parser_reset (prs);
}
