/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDSEEKBACKCALCULATOR_H
#define ISDSEEKBACKCALCULATOR_H

/**
 * This class implements the "seekback" algorithm.
 * It is used to calculate the smallest amount of rects
 * to go back in a video-stream, and to rebuild the video-image
 * from that rectangle on, onto the current image.
 *
 * The algorithm in a nutshell:
 * Given a rectangle-update at any particular position in the video-stream,
 * with a position (x,y) in the imagebuffer and a width w and height h,
 * calculate a list of rectangles that "emerge" when the previous rectangle 
 * (to this one) is substracted from this rect. So, cut the dimensions
 * of the previous one out of this one and generate a number of new rectangles
 * (maximum 8 and all a sub-rect of the current one) and put them in list FREE_RECTS.
 * Now, go back in the rectangle-chain, even further (the pre-previous one) and do
 * the same for all rects in list FREE_RECTS, deleting a rect in that list when the
 * new one completely overlaps the one in the list, until FREE_RECTS is empty.
 * At that point, you "are at" the seekback rectangle.
 *
 * Put another way, when we are at rect-update x and want to display rect-update x-1,
 * we must "erase" update x by going back in time until the area that was masked by x is 
 * visible again.
 *
 * @author Bram Biesbrouck <b@beligum.org>
*/

#include <list>

#include <libinstrudeo/isdobject.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while calculating a seekback rectangle: \n"
#include <libinstrudeo/isdloggermacros.h>

using namespace std;

class ISDRectangle;

class ISDSeekBackCalculator : public ISDObject
{

 public:
    //-----CONSTRUCTORS-----
    /**
     * Default constructor
     */
    ISDSeekBackCalculator();
    
    ~ISDSeekBackCalculator();

    //-----METHODS-----
    /**
     * Calculates the seekback for rect and stores it in that rect.
     *
     * @param rect The rect to calc the seekback for.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode calcSeekBack(ISDRectangle* rect);

 protected:
    //-----METHODS-----
    /**
     * Function that does the actual work.
     *
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode doCalcSeekBack();

    /**
     * Checks if the source rectangle overshadows the target rectangle.
     *
     * Note: this method isn't used yet, but is included for eventual future use.
     *
     * @param sourceRect The rectangle that's supposed to shadow the target.
     * @param targetRect The target rectangle.
     * @return true if the source shadows the target, false otherwise.
     */
    bool shadowsRect(ISDRectangle* sourceRect, ISDRectangle* targetRect);

    /**
     * Fills the list with the rectangles that can be generated after "cutting out"
     * the cutRect out of this rect. If there occurred cutting or there was a full overlap,
     * this returns true, otherwise, false.
     *
     * @param rect The rectangle to cut out of.
     * @param cutRect The rectangle to cut out of this one.
     * @return true if cutting (or overlap) occurred, false otherwise
     */
    bool cutOutRect(ISDRectangle* rect, ISDRectangle* cutRect);

    //-----VARIABLES-----
    std::list<ISDRectangle> freeRects;
    ISDRectangle* currentRect;
};

#endif
