/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  value.cc - GConfEntry, GConfValue C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "schema.h"
#include <inti/glib/error.h>

using namespace Inti;

/*  GConf::Value
 */

GConf::Value::Value()
: value_(0)
{
}

GConf::Value::Value(ValueType type)
: value_(gconf_value_new((GConfValueType)type))
{
}

GConf::Value::Value(ValueType type, const String& value_str, G::Error *error)
: value_(gconf_value_new_from_string((GConfValueType)type, value_str.c_str(), *error))
{
}

GConf::Value::Value(const GConfValue *src)
: value_(src ? gconf_value_copy(src) : 0)
{
}

GConf::Value::Value(const Value& src)
: value_(src.value_ ? gconf_value_copy(src.gconf_value()) : 0)
{
}

GConf::Value::~Value()
{
	if (value_)
		gconf_value_free(value_);
}

GConf::Value::Value& 
GConf::Value::operator=(const Value& src)
{
	if (src.value_ == value_)
		return *this;
		
	if (value_)
		gconf_value_free(value_);
		
	value_ = src.value_ ? gconf_value_copy(src.value_) : 0;
	return *this;
}

bool
GConf::Value::is_set() const
{
	return value_ != 0;
}

GConf::ValueType
GConf::Value::get_type() const
{
	return (ValueType)value_->type;
}

GConf::ValueType 
GConf::Value::get_list_type() const
{
	return (ValueType)gconf_value_get_list_type(gconf_value());
}

int 
GConf::Value::get_int() const
{
	return gconf_value_get_int(gconf_value());
}

bool 
GConf::Value::get_bool() const
{
	return gconf_value_get_bool(gconf_value());
}

double 
GConf::Value::get_float() const
{
	return gconf_value_get_float(gconf_value());
}

String 
GConf::Value::get_string() const
{
	return gconf_value_get_string(gconf_value());
}

GConf::Schema
GConf::Value::get_schema() const
{
	return Schema(gconf_value_get_schema(gconf_value()));
}

GConf::Value
GConf::Value::get_car() const
{
	return Value(gconf_value_get_car(gconf_value()));
}

GConf::Value
GConf::Value::get_cdr() const
{
	return Value(gconf_value_get_cdr(gconf_value()));
}

bool 
GConf::Value::get_list(std::vector<int>& list) const
{
	g_return_val_if_fail(get_list_type() == VALUE_INT, false);
	g_return_val_if_fail(list.empty(), false);

	GSList *first = gconf_value_get_list(gconf_value());
	GSList *next = first;

	while (next != 0)
	{
		list.push_back(gconf_value_get_int((GConfValue*)next->data));
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Value::get_list(std::vector<bool>& list) const
{
	g_return_val_if_fail(get_list_type() == VALUE_BOOL, false);
	g_return_val_if_fail(list.empty(), false);

	GSList *first = gconf_value_get_list(gconf_value());
	GSList *next = first;

	while (next != 0)
	{
		list.push_back(gconf_value_get_bool((GConfValue*)next->data));
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Value::get_list(std::vector<double>& list) const
{
	g_return_val_if_fail(get_list_type() == VALUE_FLOAT, false);
	g_return_val_if_fail(list.empty(), false);

	GSList *first = gconf_value_get_list(gconf_value());
	GSList *next = first;

	while (next != 0)
	{
		list.push_back(gconf_value_get_float((GConfValue*)next->data));
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Value::get_list(std::vector<String>& list) const
{
	g_return_val_if_fail(get_list_type() == VALUE_STRING, false);
	g_return_val_if_fail(list.empty(), false);

	GSList *first = gconf_value_get_list(gconf_value());
	GSList *next = first;

	while (next != 0)
	{
		list.push_back(gconf_value_get_string((GConfValue*)next->data));
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

bool 
GConf::Value::get_list(std::vector<Schema>& list) const
{
	g_return_val_if_fail(get_list_type() == VALUE_SCHEMA, false);
	g_return_val_if_fail(list.empty(), false);

	GSList *first = gconf_value_get_list(gconf_value());
	GSList *next = first;

	while (next != 0)
	{
		list.push_back(gconf_value_get_schema((GConfValue*)next->data));
		next = g_slist_next(next);
	}

	g_slist_free(first);
	return !list.empty();
}

void 
GConf::Value::set_int(int value)
{
	gconf_value_set_int(gconf_value(), value);
}

void
GConf::Value::set_bool(bool value)
{
	gconf_value_set_bool(gconf_value(), value);
}

void 
GConf::Value::set_float(double value)
{
	gconf_value_set_float(gconf_value(), value);
}

void 
GConf::Value::set_string(const String& str)
{
	gconf_value_set_string(gconf_value(), str.c_str());
}

void 
GConf::Value::set_schema(const Schema& sc)
{
	gconf_value_set_schema(gconf_value(), sc.gconf_schema());
}

void 
GConf::Value::set_car(const Value& car)
{
	gconf_value_set_car(gconf_value(), car.gconf_value());
}

void 
GConf::Value::set_cdr(const Value& cdr)
{
	gconf_value_set_cdr(gconf_value(), cdr.gconf_value());
}

namespace {

void real_set_list_and_free(GConfValue *value, GConfValueType type, GSList *list)
{
	gconf_value_set_list_type(value, type);
	gconf_value_set_list(value, list);

	GSList *next = list;
	while (next != 0)
	{
		gconf_value_free((GConfValue*)next->data);
		next = g_slist_next(next);
	}

	g_slist_free(list);
}

} // namespace

void
GConf::Value::set_list(std::vector<int>& list)
{
	int count = list.size();
	GSList *tmp_list = 0;

	int i = 0;
	while (i < count)
	{
		GConfValue *value = gconf_value_new(GCONF_VALUE_INT);
		gconf_value_set_int(value, list[i]);
		tmp_list = g_slist_append(tmp_list, value);
		i++;
	}

	real_set_list_and_free(gconf_value(), GCONF_VALUE_INT, tmp_list);
}
	
void 
GConf::Value::set_list(std::vector<bool>& list)
{
	int count = list.size();
	GSList *tmp_list = 0;

	int i = 0;
	while (i < count)
	{
		GConfValue *value = gconf_value_new(GCONF_VALUE_BOOL);
		gconf_value_set_bool(value, list[i]);
		tmp_list = g_slist_append(tmp_list, value);
		i++;
	}

	real_set_list_and_free(gconf_value(), GCONF_VALUE_BOOL, tmp_list);
}

void 
GConf::Value::set_list(std::vector<double>& list)
{
	int count = list.size();
	GSList *tmp_list = 0;

	int i = 0;
	while (i < count)
	{
		GConfValue *value = gconf_value_new(GCONF_VALUE_FLOAT);
		gconf_value_set_float(value, list[i]);
		tmp_list = g_slist_append(tmp_list, value);
		i++;
	}

	real_set_list_and_free(gconf_value(), GCONF_VALUE_FLOAT, tmp_list);
}

void 
GConf::Value::set_list(std::vector<String>& list)
{
	int count = list.size();
	GSList *tmp_list = 0;

	int i = 0;
	while (i < count)
	{
		GConfValue *value = gconf_value_new(GCONF_VALUE_STRING);
		gconf_value_set_string(value, list[i].c_str());
		tmp_list = g_slist_append(tmp_list, value);
		i++;
	}

	real_set_list_and_free(gconf_value(), GCONF_VALUE_STRING, tmp_list);
}

void 
GConf::Value::set_list(std::vector<Schema>& list)
{
	int count = list.size();
	GSList *tmp_list = 0;

	int i = 0;
	while (i < count)
	{
		GConfValue *value = gconf_value_new(GCONF_VALUE_SCHEMA);
		gconf_value_set_schema(value, list[i].gconf_schema());
		tmp_list = g_slist_append(tmp_list, value);
		i++;
	}

	real_set_list_and_free(gconf_value(), GCONF_VALUE_SCHEMA, tmp_list);
}

String 
GConf::Value::to_string()
{
	char *tmp = gconf_value_to_string(gconf_value());
	String s(tmp);
	g_free(tmp);
	return s;
}

/* GConf::Entry
 */

namespace {

GConfEntry* inti_gconf_entry_copy(const GConfEntry *src)
{
	GConfEntry *entry = gconf_entry_new(gconf_entry_get_key(src), gconf_entry_get_value(src));
	gconf_entry_set_schema_name(entry, gconf_entry_get_schema_name(src));
	gconf_entry_set_is_default(entry, gconf_entry_get_is_default(src));
	gconf_entry_set_is_writable(entry, gconf_entry_get_is_writable(src));
	return entry;
}

} // namespace
	
GConf::Entry::Entry(const String& key, const Value& value)
: entry_(gconf_entry_new(key.c_str(), value.gconf_value()))
{
}

GConf::Entry::Entry(const GConfEntry *src)
: entry_(inti_gconf_entry_copy(src))
{
}

GConf::Entry::Entry(const Entry& src)
: entry_(inti_gconf_entry_copy(src.gconf_entry()))
{
}

GConf::Entry::~Entry()
{
	if (entry_)
		gconf_entry_free(entry_);
}

GConf::Entry::Entry& 
GConf::Entry::operator=(const Entry& src)
{
	if (src.entry_ == entry_)
		return *this;
		
	if (entry_)
		gconf_entry_free(entry_);
		
	entry_ = inti_gconf_entry_copy(src.entry_);
	return *this;
}

bool
GConf::Entry::is_set() const
{
	return get_value().is_set();
}

String
GConf::Entry::get_key() const
{
	return gconf_entry_get_key(gconf_entry());
}

GConf::Value 
GConf::Entry::get_value() const
{
	return Value(gconf_entry_get_value(gconf_entry()));
}

String
GConf::Entry::get_schema_name() const
{
	return gconf_entry_get_schema_name(gconf_entry());
}

bool 
GConf::Entry::get_is_default() const
{
	return gconf_entry_get_is_default(gconf_entry());
}

bool 
GConf::Entry::get_is_writable() const
{
	return gconf_entry_get_is_writable(gconf_entry());
}

void 
GConf::Entry::set_value(const Value& value)
{
	gconf_entry_set_value(gconf_entry(), value.gconf_value());
}

void 
GConf::Entry::set_schema_name(const String& name)
{
	gconf_entry_set_schema_name(gconf_entry(), name.c_str());
}

void 
GConf::Entry::set_is_default(bool is_default)
{
	gconf_entry_set_is_default(gconf_entry(), is_default);
}

void 
GConf::Entry::set_is_writable(bool is_writable)
{
	gconf_entry_set_is_writable(gconf_entry(), is_writable);
}

