/*
 *   Greebo: An Ant task to manage dependency files.
 *
 *   Copyright (C) 2002 Ozben Evren
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

//
// Heavily modified for use with JBoss
//

package oevren.ant.greebo;

import java.io.File;
import java.net.URL;
import java.net.MalformedURLException;

import org.apache.tools.ant.BuildException;

/**
 *
 * <p>
 * <pre>
 * $Log: JarDependency.java,v $
 * Revision 1.1.2.1  2003/03/08 18:07:43  user57
 *  o import of heavily modified Greebo dependency manager task
 *  o modified buildmagic/tasks module to use the new fluff
 *
 * Revision 1.1  2002/11/22 02:41:47  ozbene
 * Initial check-in to CVS
 *
 * </pre>
 *
 * @author   Ozben Evren
 * @version  $Id: JarDependency.java 21628 2008-08-19 22:06:02Z pgier $
 *
 */
public class JarDependency 
{
    private String name;
    private String groupName;
    private String version;
    private String jar;
    private String repository;

    public JarDependency() {}

    public String getName() {
        return name;
    }

    public void setName(final String name) {
        this.name = name;
    }

    public String getGroupName() {
        if (groupName != null)
            return groupName;
        if (name != null)
            return name;
        
        return "ungrouped";
    }
    
    public void setGroupName(final String name) {
        this.groupName = name;
    }
    
    public String getVersion() {
        return version;
    }

    public void setVersion(final String version) {
        this.version = version;
    }

    public String getJar() {
        return jar;
    }

    public void setJar(final String jar) {
        this.jar = jar;
    }

    public String getRepository() {
        return repository;
    }

    public void setRepository(final String repository) {
        this.repository = repository;
    }

    public String toString() {
        if (jar != null)
            return jar;
        return name + "-" + version + ".jar";
    }

    public String getFileName()
    {
        if (jar == null) {
            if (name == null || version == null)
                throw new BuildException("Name and Version, or jar must be specified");

            return name + "-" + version + ".jar";
        } 
        
        return jar;
    }
    
    public String getFileNameFragment()
    {
        return new File(new File(new File(getGroupName()), "lib"), getFileName()).toString();
    }
    
    /**
     * Generates and returns the complete repository to fetch this dependency.
     * If only <code>name</code> and <code>version</code> is defined, then it
     * will use those and passed <code>repositoryUrl</code> to create the
     * repository. If <code>jar</code> is defined, then it will be used instead of
     * <code>name</code> and <code>version</code>. If <code>repository</code> is specified,
     * then it will override the passed <code>repositoryUrl</code>
     * @param defaultRepository The URL to the default repository.
     * @return The full repository for this dependency
     */
    public URL getURL(Repository defaultRepository) throws MalformedURLException, BuildException 
    {
        String fileName = getFileName();
        
        if (repository != null)
            return new URL(repository + "/" + fileName);
        
        if (name == null || version == null)
            throw new BuildException("Name and Version must be specified, if no repository is specified");
        
        URL base = defaultRepository.getURL();
        String suffix = "UNKNOWN";
        
        switch (defaultRepository.getTypeID())
        {
            case Repository.FLAT_ID:
                suffix = fileName;
                break;
                
            case Repository.MAVEN_ID:
                suffix = name + "/jars/" + fileName;
                break;
            
            case Repository.JBOSS_ID:
                suffix = getGroupName() + "/lib/" + fileName;
                break;
        }
        
        return new URL(base, suffix);
    }
}
