/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.tools.buildmagic.task.config;

import java.io.File;
import java.util.Map;

import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.PatternSet;

import org.jboss.tools.buildmagic.task.MissingAttributeException;
import org.jboss.tools.buildmagic.task.util.TaskLogger;

import org.jboss.util.DirectoryBuilder;

/**
 * A helper task to setup library the properties and path
 * elements for using a thirdparty library.  
 *
 * <p>This will setup the following properties:
 * <ul>
 *   <li><tt><em>vendor</em>.<em>name</em>.root</tt>
 *   <li><tt><em>vendor</em>.<em>name</em>.lib</tt>
 * </ul>
 *
 * <p>As well as the following path, based on set attributes
 *    and nested elements:
 * <ul>
 *   <li><tt><em>vendor</em>.<em>name</em>.classpath</tt>
 *
 * <p>Usage:
 * <xmp>
 *
 *   <library vendor="sun" name="jmx" root="${project.thirdparty}">
 *     <classpath>
 *       <include name="*.jar"/>
 *     </classpath>
 *   </library>
 *
 *   <library vendor="sun" name="jaxp" root="${project.thirdparty}">
 *     <include name="*.jar"/>
 *   </library>
 *
 *   <library vendor="sun" name="jaax" root="${project.thirdparty}" includes="*.jar"/>
 *
 * </xmp>
 *
 * <p>If no includes/excludes or classpath elements are specified,
 *    then a default includes of "*" is used.
 *  
 * @version <tt>$Revision: 21939 $</tt>
 * @author  <a href="mailto:jason@planet57.com">Jason Dillon</a>
 */
public class Library
   extends Task 
{
   /** Instance logger. */
   protected final TaskLogger log = new TaskLogger(this);

   protected String vendor;
   protected String name;
   protected File root;
   protected FileSet classpathSet;

   protected Path classpath;

   public void setVendor(final String vendor)
   {
      this.vendor = vendor;
   }

   public void setName(final String name)
   {
      this.name = name;
   }

   public void setRoot(final File root)
   {
      this.root = root;
   }

   public void setIncludes(final String include)
   {
      // add to classpath
      FileSet fs = createClasspath();
      PatternSet.NameEntry entry = fs.createInclude();
      entry.setName(include);
   }

   public void setExcludes(final String exclude)
   {
      // add to classpath
      FileSet fs = createClasspath();
      PatternSet.NameEntry entry = fs.createExclude();
      entry.setName(exclude);
   }

   public FileSet createClasspath() 
   {
      if (classpathSet == null) {
         classpathSet = new FileSet();
      }

      return classpathSet;
   }

   /** Helpers */
   public PatternSet.NameEntry createInclude() 
   {
      return createClasspath().createInclude();
   }

   /** Helpers */
   public PatternSet.NameEntry createExclude() 
   {
      return createClasspath().createExclude();
   }

   /**
    * Validate the attributes for this task.
    *
    * @throws BuildException    Attributes are not valid.
    */
   protected void validate() throws BuildException 
   {
      if (vendor == null)
         throw new MissingAttributeException("vendor", this);
      if (name == null)
         throw new MissingAttributeException("name", this);
      if (root == null)
         throw new MissingAttributeException("root", this);
   }

   protected File getVendorDir()
   {
      DirectoryBuilder builder = new DirectoryBuilder(root);

      // make vendor dir
      builder.cd(vendor).cd(name);
      File dir = builder.get();

      return dir;
   }

   protected File getLibraryDir()
   {
      DirectoryBuilder builder = new DirectoryBuilder(getVendorDir());

      // make vendor dir
      builder.cd("lib");
      File dir = builder.get();

      return dir;
   }

   /**
    * Execute the task.
    *
    * @throws BuildException    Failed to execute.
    */
   public void execute() throws BuildException 
   {
      // short circuit if we are done
      if (classpath != null) return;

      validate();

      log.debug("vendor: " + vendor);
      log.debug("name: " + name);
      log.debug("root: " + root);
      
      File vendorDir = getVendorDir();
      getProject().setProperty(vendor + "." + name + ".root", vendorDir.toString());
      log.debug("vendorDir: " + vendorDir);

      // make lib dir
      File libDir = getLibraryDir();
      getProject().setProperty(vendor + "." + name + ".lib", libDir.toString());
      log.debug("libDir: " + libDir);

      // assemple classpath
      if (classpathSet == null) {
         setIncludes("*");
      }

      log.debug("classpathSet: " + classpathSet);
      classpathSet.setDir(libDir);
      classpath = new Path(getProject());
      classpath.addFileset(classpathSet);
      log.debug("classpath: " + classpath);

      // set classpath ref
      Map refs = getProject().getReferences();
      refs.put(vendor + "." + name + ".classpath", classpath);

      // help gc
      vendor = null;
      name = null;
      root = null;
      classpathSet = null;
   }

   /**
    * For helper tasks.
    */
   public Path getClasspath()
   {
      return classpath;
   }
}
