/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.tools.buildmagic.task.module;

import java.io.File;
import java.io.PrintWriter;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Map;

import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildEvent;
import org.apache.tools.ant.TaskAdapter;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Target;
import org.apache.tools.ant.ProjectHelper;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.taskdefs.Property;
import org.apache.tools.ant.taskdefs.Echo;

import org.jboss.util.Strings;

import org.jboss.tools.buildmagic.task.IllegalAttributeException;
import org.jboss.tools.buildmagic.task.ResolveProperties;
import org.jboss.tools.buildmagic.task.Ant;
import org.jboss.tools.buildmagic.task.AbstractBuildListener;

import org.jboss.tools.buildmagic.task.util.TaskLogger;

/**
 * Executes a set of modules (sub-projects).
 *
 * @version <pre>$Id: ExecuteModules.java 21939 2008-08-27 21:01:35Z pgier $</pre>
 * @author  <a href="mailto:jason@planet57.com">Jason Dillon</a>
 */
public class ExecuteModules
   extends Task
{
   /*
   static {
      Runtime.getRuntime().addShutdownHook(new Thread("Buildmagic Shutdown Hook")
         {
            public void run() {
               // Halt when ask'd to shutdown
               Runtime.getRuntime().halt(-1);
            }
         });
   }
   */
   
   /** Instance logger. */
   protected final TaskLogger log = new TaskLogger(this);

   /** The target name to execute. */
   protected String target;

   /** The buildfile to use. */
   protected String antfile;
   
   /** The list of module names to exectue. */
   protected List modules;

   /** The root directory that contains modules. */
   protected String root;

   /** User properties that will be passed to each module. */
   protected List properties = new LinkedList();

   /** A internal project to hold property defs. */
   protected Project _project;

   protected String moduleProperty = "module";
   protected String targetProperty = "target";
   
   /** Skip over modules that do not exist. */
   protected boolean skipMissing = false;
   protected boolean inheritAll = false;
   
   /** Tasks to echo some output before and after each module executes. */
   protected List headers = new LinkedList();
   protected List footers = new LinkedList();

   protected List beforeHooks = new LinkedList();
   protected List afterHooks = new LinkedList();

   protected ModuleBuildListener listener = new ModuleBuildListener();

   protected List exportProperties = new LinkedList();

   protected String module;

   protected boolean threading = false;

   public void setThreading(boolean flag)
   {
      threading = flag;
   }
   
   public void setExportproperties(String list) {
      StringTokenizer stok = new StringTokenizer(list, ",");
      while (stok.hasMoreTokens()) {
         exportProperties.add(stok.nextToken().trim());
      }
   }
   
   public void setModuleproperty(String property) {
      moduleProperty = property;
   }

   public void setTargetproperty(String property) {
      targetProperty = property;
   }

   public void setInheritAll(boolean flag) {
      inheritAll = flag;
   }

   /** Setup the internal project. */
   public void init() {
      _project = new Project();
      _project.setJavaVersionProperty();
      Map tasks = getProject().getTaskDefinitions();
      _project.addTaskDefinition("property", (Class)tasks.get("property"));
      _project.addTaskDefinition("echo", (Class)tasks.get("echo"));
   }

   /** Re-initialize the internal project. */
   protected void reinit() {
      init();

      // copy the list of properties to the new internal project
      for (int i=0; i<properties.size(); i++) {
         Property a = (Property)properties.get(i);
         Property b = (Property)_project.createTask("property");
         copy(a, b);            
         properties.set(i, b);
      }
   }

   /** Copy a property values to another property. */
   protected void copy(final Property a, final Property b) {
      b.setName(a.getName());
      
      if (a.getValue() != null) {
         b.setValue(a.getValue());
      }
            
      if (a.getFile() != null) {
         b.setFile(a.getFile());
      }
        
      if (a.getResource() != null) {
         b.setResource(a.getResource());
      }
   }
    
   /** Set the target name. */
   public void setTarget(final String target) {
      this.target = target;
   }

   /** Set the buildfile to use. */
   public void setAntfile(final String antfile) {
      this.antfile = antfile;
   }
   
   /** Set the modules to execute. */
   public void setModules(String names) {
      names = ResolveProperties.subst(names, getProject(), true);
      StringTokenizer stok = new StringTokenizer(names, ",");
      modules = new LinkedList();
      while (stok.hasMoreTokens()) {
         modules.add(stok.nextToken().trim());
      }
   }

   public void setSkipmissing(boolean flag) {
      skipMissing = flag;
   }
   
   /** Set the root. */
   public void setRoot(final String dir) throws BuildException {
      this.root = dir;
   }
    
   /**
    * Execute this task.
    *
    * @throws BuildException    Failed to execute task.
    */
   public void execute() throws BuildException {
      try {
         // need at least one module name
         if (modules == null || modules.size() == 0) {
            throw new BuildException
               ("No module names were specified", getLocation());
         }

         // need the root directory
         if (root == null) {
            throw new BuildException
               ("Root directory not specified", getLocation());
         }

         Iterator iter = modules.iterator();
         while (iter.hasNext()) {
            executeModule((String)iter.next());
         }
      }
      finally {
         // help the gc
         _project = null;
      }
   }
   
   /**
    * Execute a single module.
    */
   protected void executeModule(final String module)
      throws BuildException
   {
      this.module = module;

      // create and setup the ant task
      final Ant ant = (Ant)getProject().createTask("Ant");
      ant.setInheritAll(inheritAll);
      ant.init();

      // add a property for the name of the module (our name that is)
      /*Property p = createProperty(null);

      p.setName(moduleProperty);
      p.setValue(module);
      copy(p, ant.createProperty());
      System.out.println("Setting propery - " + moduleProperty + ":" + module);
      p.execute();*/
      // project.setUserProperty(moduleProperty, module);
      _project.setProperty(moduleProperty, module);
      
      // add a property for the target of the module (our name that is)
      String tempTargetName = (target == null) ? "<default>" : target;
      evaluateExpression(tempTargetName);
      /*p = createProperty(null);
      p.setName(targetProperty);
      p.setValue(tempTargetName);
      copy(p, ant.createProperty());
      p.execute();*/
      // project.setUserProperty(targetProperty, tempTargetName);
      _project.setProperty(targetProperty, tempTargetName);
      
      ant.setLocation(getLocation());
      tempTargetName = ResolveProperties.subst(root, getProject());
      tempTargetName = Strings.subst("@MODULE@", module, tempTargetName);
      tempTargetName = Strings.subst("@TARGET@", target, tempTargetName);
      
      File moduleRoot = new File(tempTargetName);
      log.verbose("module root: " + moduleRoot);
      ant.setDir(moduleRoot);
      ant.setAntfile(antfile);
      
      if (target != null) {
         log.verbose("using target: " + target);
         ant.setTarget(target);
      }

      Map props = getProject().getProperties();
      
      // see if this is a valid ant file
      try {
         if (ant.getBuildFile() != null);
      }
      catch (BuildException e) {
         log.verbose("exception: " + e);
         if (skipMissing) {
            listener.skipped(module);
            log.warning("Missing build file; skipping module: " + module);

            props.remove(moduleProperty);
            props.remove(targetProperty);
            return;
         }

         log.error("Missing build file: " + module);
         throw e;
      }

      runHooks(beforeHooks);
      
      ant.addBuildListener(listener);
      
      // set up any properties
      Iterator iter = properties.iterator();
      while (iter.hasNext()) {
         Property a = (Property)iter.next();
         Property b = ant.createProperty();
         copy(a, b);
         a.execute();
      }

      // export any listed property names
      iter = exportProperties.iterator();
      while (iter.hasNext()) {
         String name = (String)iter.next();
         String value = (String)props.get(name);
         Property p = ant.createProperty();
         p.setName(name);
         p.setValue(value);

         log.verbose("Exported property " + name + "=" + value);
      }

      final String targetName = tempTargetName;
      
      Runnable runner = new Runnable()
         {
            public void run()
            {
               // execute the task
               printHeading(headers);
               log.verbose("Executing " + targetName + " in module '" + module + "'...");

               ant.execute();
        
               log.verbose("Finished with " + targetName + " in module '" + module + "'...");
               printHeading(footers);

               runHooks(afterHooks);
            }
         };
      
      if (threading) {
         new Thread(runner, "Module Runner (" + module + ":" + targetName + ")").start();
      }
      else {
         runner.run();
      }
            
      // shit this sucks
      // jason: this will not work in ant 1.5
      // props.remove(moduleProperty);
      // props.remove(targetProperty);
        
      // iter = properties.iterator();
      // while (iter.hasNext()) {
      // Property a = (Property)iter.next();
      // props.remove(a.getName());
      // }
   }   

   protected void printHeading(List headers) {
      Iterator iter = headers.iterator();
      while (iter.hasNext()) {
         MyEcho header = (MyEcho)iter.next();
         header.execute();
      }
   }
   
   /** Create a nested property. */
   public Property createProperty() {
      Property prop = createProperty(properties);
      return prop;
   }

   /** Create a nested property. */
   public Property createProperty(List list) {
      if (_project == null) {
         reinit();
      }

      Property prop = (Property)_project.createTask("property");
        
      if (list != null) {
         list.add(0, prop);
      }
        
      return prop;
   }
    
   /** Create a nested header. */
   public Echo createHeader() {
      if (_project == null) {
         reinit();
      }

      MyEcho header = new MyEcho(getProject());
      headers.add(header);
      return header;
   }

   /** Create a nested footer. */
   public Echo createFooter() {
      if (_project == null) {
         reinit();
      }

      
      MyEcho footer = new MyEcho(getProject());
      footers.add(footer);
      return footer;
   }

   /**
    * Takes a string which may contain properties and
    * resolves the properties to their values.
    * 
    * @param expr
    * @return
    */
   public String evaluateExpression(String expr) {
      Map props = getProject().getUserProperties();
      expr = ResolveProperties.subst(expr, props, false);
      
      props = _project.getUserProperties();
      expr = ResolveProperties.subst(expr, props, false);

      props = _project.getProperties();
      expr = ResolveProperties.subst(expr, props, false);
      
      return expr;
   }
   
   /*public void printProperties(Map props, String header)
   {
	   Iterator keys = props.keySet().iterator();
	   System.out.println(header);
	   while(keys.hasNext())
	   {
		   Object key = keys.next();
		   System.out.println(key + ":" + props.get(key));
	   }
   }*/
   
   /**
    * Nested echo class to hold header and footer data.
    */
   protected class MyEcho
      extends Echo
   {
      Project project;
      
      public MyEcho(Project project)
      {
         this.project = project;
      }
      
      public void addText(String msg) {
         message += msg;
      }
        
      public void execute() throws BuildException {
         String temp = evaluateExpression(message);
         
         temp = Strings.subst("@MODULE@", module, temp);
         temp = Strings.subst("@TARGET@", ExecuteModules.this.target, temp);

         project.log(temp, logLevel);
      }
      
      public String getMessage()
      {
    	 return this.message;
      }
   }

   public Hook createBefore() {
      Hook hook = new Hook(this);
      beforeHooks.add(hook);
      return hook;
   }
   
   public Hook createAfter() {
      Hook hook = new Hook(this);
      afterHooks.add(hook);
      return hook;
   }
   
   protected class Hook
   {
      public ExecuteModules task;
      public String target;

      
      public Hook(ExecuteModules task) {
         this.task = task;
      }
      
      public void setTarget(String target) {
         this.target = target;
      }

      public void execute() throws BuildException {
         Project project = task.getProject();
         String t = task.evaluateExpression(target);

         t = Strings.subst("@MODULE@", module, t);
         t = Strings.subst("@TARGET@", ExecuteModules.this.target, t);
         
         if (project.getTargets().containsKey(t)) {
            project.executeTarget(t);
         }
         else {
            log.verbose("skipping missing hook: " + t);
         }
      }
   }

   protected void runHooks(List list) throws BuildException {
      log.verbose("executing hooks");
      log.debug("list: " + list);
      
      Iterator iter = list.iterator();
      while (iter.hasNext()) {
         Hook hook = (Hook)iter.next();
         log.debug("executing hook: " + hook);
         hook.execute();
      }
   }
   
   /**
    * This is meant to provide the ability to generate an xml file
    * which contains what modules we have invoked, their targets and
    * output.
    *
    * <p>The goal was to allow this to be applied to a stylesheet
    *    to produce documentaion for the build, only showing the relevant
    *    bits.
    */
   protected class ModuleBuildListener
      extends AbstractBuildListener
   {
      //
      // hookup xml output here
      //
      public void skipped(String module) {
         // System.out.println("skipped module: " + module);
      }
      public void buildStarted(BuildEvent event) {
         // System.out.println("started build: " + event);
      }
      public void buildFinished(BuildEvent event) {
         // System.out.println("finished build: " + event);
      }
      public void targetStarted(BuildEvent event) {
         // System.out.println("started target: " + event.getTarget());
      }
      public void targetFinished(BuildEvent event) {
         // System.out.println("finished target: " + event.getTarget());
      }
      public void taskStarted(BuildEvent event) {
         // System.out.println("started task: " + event.getTask());
      }
      public void taskFinished(BuildEvent event) {
         // System.out.println("finished task: " + event.getTask());
      }
      public void messageLogged(BuildEvent event) {
         Throwable t = event.getException();
         int pri = event.getPriority();
         String message = event.getMessage();
         
         if (t != null) {
            // System.out.println("exception: " + t);
         }

         switch (pri) {
          case Project.MSG_ERR:
             // System.out.println("error: " + message);
             break;
          case Project.MSG_WARN:
             // System.out.println("warning: " + message);
             break;
          case Project.MSG_INFO:
          case Project.MSG_VERBOSE:
          case Project.MSG_DEBUG:
         }
      }
   }
}
