/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.tools.buildmagic.task.module;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.StringTokenizer;

import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildException;

import org.jboss.tools.buildmagic.task.Tasks;
import org.jboss.tools.buildmagic.task.MissingAttributeException;
import org.jboss.tools.buildmagic.task.MissingElementException;
import org.jboss.tools.buildmagic.task.util.TaskLogger;
import org.jboss.tools.buildmagic.task.util.ConditionalExecution;

/**
 * Module configuration task.
 *
 * @todo Merge groups and modules. they will share namespace, but
 *       will be easier to work with.
 *
 * @version <pre>$Id: ModuleConfig.java 21939 2008-08-27 21:01:35Z pgier $</pre>
 * @author  <a href="mailto:jason@planet57.com">Jason Dillon</a>
 */
public class ModuleConfig
   extends Task
{
   /** Instance logger. */
   protected TaskLogger log = new TaskLogger(this);

   /** The list of modules. */
   protected List modules = new LinkedList();

   /** The list of groups. */
   protected List groups = new LinkedList();

   /** The name of the property that is to be set. */
   protected String property = "modules";

   /** The selected group. */
   protected String selected;

   /** The conditional execution helper. */
   protected ConditionalExecution cond = new ConditionalExecution(this);
   
   /** Set the property name. */
   public void setProperty(String property) {
      this.property = property;
   }

   /** Set the selected group. */
   public void setSelected(String selected) {
      this.selected = selected;
   }

   /** Create a module. */
   public Module createModule() {
      Module m = new Module(this);
      modules.add(m);
      return m;
   }

   /** Create a group. */
   public Group createGroup() {
      Group g = new Group(this);
      groups.add(g);
      return g;
   }

   /** Create a condition. */
   public ConditionalExecution.Condition createCondition() {
      return cond.createCondition();
   }
   
   /**
    * Execute this task.
    *
    * @throws BuildException    Failed to execute task.
    */
   public void execute() throws BuildException {
      if (selected == null)
         throw new MissingAttributeException("name", this);
      if (modules.size() == 0)
         throw new MissingElementException("module", this);
      if (groups.size() == 0)
         throw new MissingElementException("group", this);

      if (!cond.canExecute()) return;

      List list = new LinkedList();
      StringTokenizer stok = new StringTokenizer(selected, ",");
      while (stok.hasMoreTokens()) {
         String name = stok.nextToken().trim();
         log.verbose("selected group: " + name);
         Group group = getGroup(name);
         if (group == null)
            throw new BuildException("Invalid module group: " + selected);

         log.debug("group: " + group);

         List allModules = group.getModules();
         if (allModules == null)
            throw new BuildException("Invalid group: " + name);
         
         list.addAll(allModules);
      }
      log.debug("full module list: " + list);
      
      String value = generatePropertyValue(list);
      getProject().setProperty(property, value);
      log.verbose("Module list: " + value);
   }

   /**
    * Generate the property value from the given module list.
    */
   protected String generatePropertyValue(List list) {
      StringBuffer buff = new StringBuffer();

      Iterator iter = list.iterator();
      while (iter.hasNext()) {
         Module mod = (Module)iter.next();
         if (mod == null)
            throw new BuildException("Null module in list: " + list);
         
         buff.append(mod.getName());
         if (iter.hasNext()) {
            buff.append(",");
         }
      }

      return buff.toString();
   }

   /**
    * Get an element by name.
    */
   protected NamedElement getNamedElement(List list, String name) {
      Iterator iter = list.iterator();

      while (iter.hasNext()) {
         NamedElement elem = (NamedElement)iter.next();
         if (elem.getName().equals(name)) {
            return elem;
         }
      }

      return null;
   }

   /**
    * Get a module by name.
    */
   protected Module getModule(String name) {
      return (Module)getNamedElement(modules, name);
   }
   
   /**
    * Get a group by name.
    */
   protected Group getGroup(String name) {
      return (Group)getNamedElement(groups, name);
   }

   
   /////////////////////////////////////////////////////////////////////////
   //                             Nested Elements                         //
   /////////////////////////////////////////////////////////////////////////
   
   /**
    * An abstract element which has a name.
    */
   protected abstract class NamedElement
   {
      protected Task parent;
      protected String name;

      protected NamedElement(Task parent) {
         this.parent = parent;
      }
      
      public void setName(String name) {
         this.name = name;
      }

      public String getName() {
         return this.name;
      }
   }

   /**
    * A module element.
    */
   protected class Module
      extends NamedElement
   {
      protected List depends = new LinkedList();

      public Module(Task parent) {
         super(parent);
      }
      
      public void addDependency(String name) {
         depends.add(name);
      }
      
      public void setDepends(String depends) {
         StringTokenizer stok = new StringTokenizer(depends, ",", false);
         while (stok.hasMoreTokens()) {
            addDependency(stok.nextToken().trim());
         }
      }

      public String toString() {
         return "{ name=" + name + ", depends=" + depends + " }";
      }
   }
   

   /**
    * A group element.
    */
   protected class Group
      extends NamedElement
   {
      /** The list of includes. */
      protected List includes = new LinkedList();
      
      /** The conditional execution helper. */
      protected ConditionalExecution cond;

      public Group(Task parent) {
         super(parent);
         cond = new ConditionalExecution(parent);
      }

      /** Create a condition. */
      public ConditionalExecution.Condition createCondition() {
         return cond.createCondition();
      }
      
      public Include createInclude() {
         Include inc = new Include(this);
         includes.add(inc);
         return inc;
      }

      public void setModules(String modules) {
         Include inc = new Include(this);
         inc.setModules(modules);
         includes.add(inc);
      }

      public void setGroups(String groups) {
         Include inc = new Include(this);
         inc.setGroups(groups);
         includes.add(inc);
      }
      
      public List getModules() {
         List list = new LinkedList();
         if (!cond.canExecute()) return list;
         
         Iterator iter = includes.iterator();
         while (iter.hasNext()) {
            Include inc = (Include)iter.next();
            log.debug("adding modules from include: " + inc);
            list.addAll(inc.getModules());
         }

         return list;
      }

      public String toString() {
         return "{ name=" + name + ", includes=" + includes + " }";
      }

      /**
       * An include elemnt.
       */
      protected class Include
      {
         /** The conditional execution helper. */
         protected ConditionalExecution cond;
         
         protected Group parent;
         protected String modules = "";
         protected String groups = "";

         public Include(Group parent) {
            this.parent = parent;
            cond = new ConditionalExecution(parent.parent);
         }

         /** Create a condition. */
         public ConditionalExecution.Condition createCondition() {
            return cond.createCondition();
         }
         
         public void setModules(String modules) {
            this.modules = modules;
         }

         public void setGroups(String groups) {
            this.groups = groups;
         }

         public List getGroups() {
            List list = new LinkedList();
            if (!cond.canExecute()) return list;
            
            StringTokenizer stok = new StringTokenizer(groups, ",");
            while (stok.hasMoreTokens()) {
               String name = stok.nextToken().trim();
               Group group = getGroup(name);
               if (group == null)
                  throw new BuildException("Invalid group include: " + name);
               
               list.add(group);
               log.debug("adding group: " + group);
            }

            return list;
         }

         public List getModules() {
            List list = new LinkedList();
            if (!cond.canExecute()) return list;
            
            StringTokenizer stok = new StringTokenizer(modules, ",");
            while (stok.hasMoreTokens()) {
               String name = stok.nextToken().trim();
               Module module = getModule(name);
               if (module == null)
                  throw new BuildException("Invalid module include: " + name);
               
               list.add(module);
               log.debug("adding module: " + module);
            }

            List groups = getGroups();
            Iterator iter = groups.iterator();
            while (iter.hasNext()) {
               Group group = (Group)iter.next();
               list.addAll(group.getModules());
            }
            
            return list;
         }

         public String toString() {
            return "{ modules=" + modules + ", groups=" + groups + " }";
         }
      }
   }
}
